/*
 * nzb
 *
 * Copyright (C) 2004-2005 Mattias Nordstrom <matta at ftlight net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * Authors:
 *   Mattias Nordstrom <matta at ftlight net>
 *
 * $Id: mainwindow.cpp,v 1.18 2005/10/16 12:03:32 mnordstr Exp $
 *   This file provides the GUI.
 */


#include <QtGui>

#include "mainwindow.h"
#include "downloader.h"

MainWindow::MainWindow()
{
	shutting_down = false;
	
	split = new QSplitter(Qt::Vertical, this);
	
	l = new QTableWidget(split);
	l->setColumnCount(4);
	l->setHorizontalHeaderLabels((QStringList)"Subject" << "Progress" << "Parts" << "Size");
	l->setSelectionMode(QAbstractItemView::NoSelection);
	l->setSelectionBehavior(QAbstractItemView::SelectRows);
	l->setShowGrid(false);
	
	d = new QTableWidget(split);
	
	d->setColumnCount(3);
	d->setHorizontalHeaderLabels((QStringList)"Thread" << "Status" << "Speed");
	d->setSelectionMode(QAbstractItemView::NoSelection);
	d->setSelectionBehavior(QAbstractItemView::SelectRows);
	d->setShowGrid(false);
	
	setCentralWidget(split);

	createActions();
	createMenus();
	createToolBars();
	createStatusBar();
	
	readSettings();
	
	d->setRowCount(config->value("server/connections").toInt() + 1);
	
	int i;
	for (i=0; i<config->value("server/connections").toInt(); i++) {
		downloaders.append(new Downloader(&nzblist, i, &file_lock, l, this));
		d->setItem(i, 0, new QTableWidgetItem("Downloader #"+QString::number(i+1)));
		d->setItem(i, 2, new QTableWidgetItem("0 kB/s"));
		connect(this, SIGNAL(fullStop()), downloaders[i], SLOT(stop()));
	}

	decoders.append(new Decoder(&nzblist, i, &file_lock, this));
	d->setItem(i, 0, new QTableWidgetItem("Decoder #1"));
	connect(this, SIGNAL(processDecoder()), decoders[0], SLOT(processFiles()));
	connect(this, SIGNAL(fullStop()), decoders[0], SLOT(stop()));
	
	output = new Output(&nzblist, i+1, &file_lock, this);
	connect(this, SIGNAL(processOutput()), output, SLOT(process()));
	connect(this, SIGNAL(fullStop()), output, SLOT(stop()));

	int drows = d->rowCount();
	for (int row=0; row<drows; row++) {
		d->resizeRowToContents(row);
	}
	
	setWindowTitle(tr("nzb"));
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	shutting_down = true;
	
	if (stopAct->isEnabled()) {
		stop();
	}
	
	writeSettings();
	event->accept();
}

void MainWindow::open()
{
	QString fileName = QFileDialog::getOpenFileName(this, "Open NZB File", NULL, "nzb Files (*.nzb)\nAll Files (*)");
	if (!fileName.isEmpty())
		loadFile(fileName);
}

void MainWindow::about()
{
	QMessageBox::about(this, tr("About nzb"), tr("nzb v%1\n\nnzb is a Usenet binary downloader.\n\nSee http://nzb.sf.net/").arg(NZB_VERSION));
}

void MainWindow::createActions()
{
	openAct = new QAction(QIcon(":/images/open.png"), tr("&Open..."), this);
	openAct->setShortcut(tr("Ctrl+O"));
	openAct->setStatusTip(tr("Open a file"));
	connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

	exitAct = new QAction(tr("E&xit"), this);
	exitAct->setShortcut(tr("Ctrl+Q"));
	exitAct->setStatusTip(tr("Exit the application"));
	connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

	sortAct = new QAction(tr("&Sort"), this);
	sortAct->setStatusTip(tr("Sort the nzb list"));
	connect(sortAct, SIGNAL(triggered()), this, SLOT(sortList()));

	clearAct = new QAction(tr("&Clear"), this);
	clearAct->setStatusTip(tr("Clear the nzb list"));
	connect(clearAct, SIGNAL(triggered()), this, SLOT(clearList()));
	
	aboutAct = new QAction(tr("&About"), this);
	aboutAct->setStatusTip(tr("Show the application's About box"));
	connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

	optionsAct = new QAction(tr("&Options"), this);
	optionsAct->setStatusTip(tr("Show the options dialog"));
	connect(optionsAct, SIGNAL(triggered()), this, SLOT(options()));
	
	startAct = new QAction(tr("&Start"), this);
	startAct->setShortcut(tr("Ctrl+S"));
	startAct->setStatusTip(tr("Start downloader"));
	startAct->setEnabled(false);
	connect(startAct, SIGNAL(triggered()), this, SLOT(start()));
	
	stopAct = new QAction(tr("S&top"), this);
	stopAct->setShortcut(tr("Ctrl+T"));
	stopAct->setStatusTip(tr("Stop downloader"));
	stopAct->setEnabled(false);
	connect(stopAct, SIGNAL(triggered()), this, SLOT(stop()));
	
	streamAct = new QAction(tr("St&ream"), this);
	streamAct->setShortcut(tr("Ctrl+R"));
	streamAct->setStatusTip(tr("Start streaming"));
	streamAct->setEnabled(false);
	connect(streamAct, SIGNAL(triggered()), this, SLOT(stream()));
}

void MainWindow::createMenus()
{
	fileMenu = menuBar()->addMenu(tr("&File"));
	fileMenu->addAction(openAct);
	fileMenu->addSeparator();
	fileMenu->addAction(exitAct);

	menuBar()->addSeparator();

	editMenu = menuBar()->addMenu(tr("&Edit"));
	editMenu->addAction(sortAct);
	editMenu->addAction(clearAct);

	menuBar()->addSeparator();
	
	actionsMenu = menuBar()->addMenu(tr("&Actions"));
	actionsMenu->addAction(startAct);
	actionsMenu->addAction(stopAct);
	actionsMenu->addAction(streamAct);

	menuBar()->addSeparator();

	toolsMenu = menuBar()->addMenu(tr("&Tools"));
	toolsMenu->addAction(optionsAct);

	menuBar()->addSeparator();

	helpMenu = menuBar()->addMenu(tr("&Help"));
	helpMenu->addAction(aboutAct);
}

void MainWindow::createToolBars()
{
	fileToolBar = addToolBar(tr("File"));
	fileToolBar->addAction(openAct);
}

void MainWindow::createStatusBar()
{
	statusBar()->showMessage(tr("Ready"));
	
	totalSpeed = new QLabel("0 kB/s");
	totalSize = new QLabel("0 MB");
	totalProgress = new QProgressBar();
	totalProgress->setRange(0, 1);
	totalProgress->setValue(0);
	
	statusBar()->addPermanentWidget(totalSpeed);
	statusBar()->addPermanentWidget(totalProgress);
	statusBar()->addPermanentWidget(totalSize);
}

void MainWindow::readSettings()
{
	config = new QSettings("nzb", "nzb");
	QPoint pos = config->value("mainwindow/pos", QPoint(200, 200)).toPoint();
	QSize size = config->value("mainwindow/size", QSize(400, 400)).toSize();
	resize(size);
	move(pos);
	split->restoreState(config->value("mainwindow/split").toByteArray());
	
	l->horizontalHeader()->resizeSection(0, config->value("mainwindow/listcol0width", 50).toInt());
	l->horizontalHeader()->resizeSection(1, config->value("mainwindow/listcol1width", 50).toInt());
	l->horizontalHeader()->resizeSection(2, config->value("mainwindow/listcol2width", 50).toInt());
	l->horizontalHeader()->resizeSection(3, config->value("mainwindow/listcol3width", 50).toInt());
	d->horizontalHeader()->resizeSection(0, config->value("mainwindow/dlistcol0width", 50).toInt());
	d->horizontalHeader()->resizeSection(1, config->value("mainwindow/dlistcol1width", 50).toInt());
	d->horizontalHeader()->resizeSection(2, config->value("mainwindow/dlistcol2width", 50).toInt());

	// Fix for situation when the default value isn't honored.
	if (l->columnWidth(0) == 0) l->horizontalHeader()->resizeSection(0, 50);
	if (l->columnWidth(1) == 0) l->horizontalHeader()->resizeSection(1, 50);
	if (l->columnWidth(2) == 0) l->horizontalHeader()->resizeSection(2, 50);
	if (l->columnWidth(3) == 0) l->horizontalHeader()->resizeSection(3, 50);
	if (d->columnWidth(0) == 0) d->horizontalHeader()->resizeSection(0, 50);
	if (d->columnWidth(1) == 0) d->horizontalHeader()->resizeSection(1, 50);
	if (d->columnWidth(2) == 0) d->horizontalHeader()->resizeSection(2, 50);
}

void MainWindow::writeSettings()
{
	config->setValue("mainwindow/pos", pos());
	config->setValue("mainwindow/size", size());
	config->setValue("mainwindow/split", split->saveState());
	config->setValue("mainwindow/listcol0width", l->columnWidth(0));
	config->setValue("mainwindow/listcol1width", l->columnWidth(1));
	config->setValue("mainwindow/listcol2width", l->columnWidth(2));
	config->setValue("mainwindow/listcol3width", l->columnWidth(3));
	config->setValue("mainwindow/dlistcol0width", d->columnWidth(0));
	config->setValue("mainwindow/dlistcol1width", d->columnWidth(1));
	config->setValue("mainwindow/dlistcol2width", d->columnWidth(2));
	config->sync();
}

void MainWindow::populateList()
{
	int i;
		
	int row = l->rowCount();
	l->setRowCount(nzblist.getList()->count());
	
	for (i=row; i<l->rowCount(); i++) {
		QTableWidgetItem *newItem = new QTableWidgetItem(nzblist.getFile(i)->getSubject());
		QTableWidgetItem *newItem_prog = new QTableWidgetItem("0");
		QTableWidgetItem *newItem_tot = new QTableWidgetItem(QString::number(nzblist.getFile(i)->getSegments()->size()));
			
		QTableWidgetItem *newItem_size;
		int file_bytes = nzblist.getFile(i)->getBytes();
		
		/*if (file_bytes > 999999999) {
			newItem_size = new QTableWidgetItem(QString::number((file_bytes / 1073741824))+" GB");
		} else*/
		if (file_bytes > 999999) {
			newItem_size = new QTableWidgetItem(QString::number((file_bytes / 1048576))+" MB");
		} else if (file_bytes > 999) {
			newItem_size = new QTableWidgetItem(QString::number((file_bytes / 1024))+" kB");
		} else {
			newItem_size = new QTableWidgetItem(QString::number(file_bytes)+" bytes");
		}
			
		newItem->setCheckState(Qt::Checked);
		
		l->setItem(row, 0, newItem);
		l->setItem(row, 1, newItem_prog);
		l->setItem(row, 2, newItem_tot);
		l->setItem(row, 3, newItem_size);

		l->resizeRowToContents(row);

		row++;
	}
	
	qint64 total_size = nzblist.totalSize();
	
	if (total_size > 999999999) {
		totalSize->setText(QString::number((double)((double)(total_size / 1048576) / 1000), 'f', 2)+" GB");
	} else if (total_size > 999999) {
		totalSize->setText(QString::number((total_size / 1048576))+" MB");
	} else if (total_size > 999) {
		totalSize->setText(QString::number((total_size / 1024))+" kB");
	} else {
		totalSize->setText(QString::number(total_size)+" bytes");
	}
	
	totalProgress->setRange(0, nzblist.totalParts());
}

void MainWindow::loadFile(const QString &fileName)
{
	QFile f(fileName);
	nzblist.importNzb(&f);
	
	if (config->value("list/autosort").toBool()) {
		sortList();
	} else {
		populateList();
	}
		
	if (nzblist.getList()->count()) {
		startAct->setEnabled(true);
		streamAct->setEnabled(true);
	}

	statusBar()->showMessage( tr("Loaded nzb document %1").arg(fileName), 2000 );
}

void MainWindow::sortList()
{
	l->clear();
	l->setRowCount(0);
	l->setHorizontalHeaderLabels((QStringList)"Subject" << "Progress" << "Parts" << "Size");

	nzblist.sortList();
	
	populateList();
}

void MainWindow::start()
{
	startAct->setEnabled(false);
	stopAct->setEnabled(true);
	streamAct->setEnabled(false);
	
	for (int i=0; i<downloaders.size(); i++) {
		downloaders[i]->start(QThread::LowPriority);
	}
	
	decoders[0]->start();
	
	output->setStream(false);
	output->start(QThread::TimeCriticalPriority);
}

void MainWindow::stop()
{
	startAct->setEnabled(true);
	stopAct->setEnabled(false);
	streamAct->setEnabled(true);
	
	qDebug("Stopping");
	
	emit fullStop();
	
	for (int i=0; i<downloaders.size(); i++) {
		downloaders[i]->wait();
	}
	
	decoders[0]->wait();
	
	//output->wait();
	
	clearData();
}

void MainWindow::clearData()
{
	l->clear();
	l->setRowCount(0);
	l->setHorizontalHeaderLabels((QStringList)"Subject" << "Progress" << "Parts" << "Size");
	
	nzblist.clearList();
	
	totalSize->setText("0 MB");
	totalProgress->setRange(0,1);
	totalProgress->setValue(0);
	
	startAct->setEnabled(false);
	streamAct->setEnabled(false);
}

void MainWindow::stream()
{
	startAct->setEnabled(false);
	stopAct->setEnabled(true);
	streamAct->setEnabled(false);
	
	for (int i=0; i<downloaders.size(); i++) {
		downloaders[i]->start(QThread::LowPriority);
	}
	
	decoders[0]->start();

	output->setStream(true);
	output->start(QThread::TimeCriticalPriority);
}

void MainWindow::options()
{
	OptionsDlg *dl = new OptionsDlg(this);
	dl->show();
}

void MainWindow::downloadEvent(QString message, int row, int type)
{
	if (type == 0) {
		d->model()->setData(d->model()->index(row, 1), message);
	} else if (type == 2) {
		l->model()->setData(l->model()->index(row, 1), (l->item(row, 1)->text()).toInt()+1);
		totalProgress->setValue(totalProgress->value() + 1);
		
		if (l->item(row, 1)->text() == "1") {
			if (config->value("list/hidecompleted").toBool()) {
				l->scrollToItem(l->item(row, 1));
			}
		}
		
		emit processDecoder();
		
	} else if (type == 3) {
		d->model()->setData(d->model()->index(row, 1), message);
	} else if (type == 4) {
		d->model()->setData(d->model()->index(row, 2), message+" kB/s");
		
		int total_speed = 0;
		for (int i=0; i<downloaders.size(); i++) {
			QString cur_speed = d->item(i, 2)->text();
			total_speed += cur_speed.mid(0, cur_speed.length() - 5).toInt();
		}
		totalSpeed->setText(QString::number(total_speed)+" kB/s");
	}
}

void MainWindow::decodeEvent(QString message, int row, int type)
{
	d->model()->setData(d->model()->index(row, 1), message);
	
	if (type == 1) {
		emit processOutput();
	} else if (type == 2) {
		QTimer::singleShot(1000, decoders[0], SLOT(processFiles()));
	} else if (type == 3) {
		QTimer::singleShot(0, output, SLOT(process()));
	}

}

void MainWindow::outputEvent(QString message, int type)
{
	if (type == 1) {
		QTimer::singleShot(200, this, SLOT(stop()));
	} else if (type == 2) {
		QTimer::singleShot(200, this, SLOT(stop()));
		QMessageBox *msgbox = new QMessageBox("nzb Error", message, QMessageBox::Critical, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
		msgbox->show();
	}
}

OptionsDlg::OptionsDlg(QWidget *parent) : QDialog(parent)
{
	ui.setupUi(this);
	this->parent = parent;
	        
	connect(ui.okButton, SIGNAL(clicked()), this, SLOT(accept()));
	connect(ui.cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
	
	connect(ui.saveBrowse, SIGNAL(clicked()), this, SLOT(browseSave()));
	connect(ui.mediaBrowse, SIGNAL(clicked()), this, SLOT(browseMedia()));
	
	connect(ui.subfolders, SIGNAL(stateChanged(int)), this, SLOT(enableGuess(int)));
	
	QSettings *config = ((MainWindow*)parent)->getConfig();
	
	ui.host->setText(config->value("server/host").toString());
	ui.port->setText(config->value("server/port").toString());
	ui.username->setText(config->value("server/username").toString());
	ui.password->setText(config->value("server/password").toString());
	ui.connections->setValue(config->value("server/connections").toInt());
	
	if (config->value("server/auth").toBool()) {
		ui.auth->setCheckState(Qt::Checked);
	} else {
		ui.auth->setCheckState(Qt::Unchecked);
	}

	
	if (config->value("list/autosort").toBool()) {
		ui.autosort->setCheckState(Qt::Checked);
	} else {
		ui.autosort->setCheckState(Qt::Unchecked);
	}
	
	if (config->value("list/hidecompleted").toBool()) {
		ui.hidecompleted->setCheckState(Qt::Checked);
	} else {
		ui.hidecompleted->setCheckState(Qt::Unchecked);
	}
	
	ui.savepath->setText(config->value("output/savepath").toString());
	if (config->value("output/subfolders").toBool()) {
		ui.subfolders->setCheckState(Qt::Checked);
		ui.guessalbum->setEnabled(true);
	} else {
		ui.subfolders->setCheckState(Qt::Unchecked);
		ui.guessalbum->setEnabled(false);
	}
	if (config->value("output/guessalbum").toBool()) {
		ui.guessalbum->setCheckState(Qt::Checked);
	} else {
		ui.guessalbum->setCheckState(Qt::Unchecked);
	}
	ui.mediaplayer->setText(config->value("output/mediaplayer").toString());
	if (config->value("output/stream").toBool()) {
		ui.stream->setCheckState(Qt::Checked);
	} else {
		ui.stream->setCheckState(Qt::Unchecked);
	}
}

void OptionsDlg::accept()
{
	QSettings *config = ((MainWindow*)parent)->getConfig();

	config->setValue("server/host", ui.host->text());
	config->setValue("server/port", ui.port->text().toInt());
	config->setValue("server/username", ui.username->text());
	config->setValue("server/password", ui.password->text());
	config->setValue("server/connections", ui.connections->value());
	config->setValue("server/auth", ((ui.auth->checkState() == Qt::Checked) ? true : false));
	
	config->setValue("list/autosort", ((ui.autosort->checkState() == Qt::Checked) ? true : false));
	config->setValue("list/hidecompleted", ((ui.hidecompleted->checkState() == Qt::Checked) ? true : false));
	
	config->setValue("output/savepath", ui.savepath->text());
	config->setValue("output/subfolders", ((ui.subfolders->checkState() == Qt::Checked) ? true : false));
	config->setValue("output/guessalbum", ((ui.guessalbum->checkState() == Qt::Checked) ? true : false));
	config->setValue("output/mediaplayer", ui.mediaplayer->text());
	config->setValue("output/stream", ((ui.stream->checkState() == Qt::Checked) ? true : false));
		
	config->sync();
	
	close();
}

void OptionsDlg::reject()
{
	close();
}

void OptionsDlg::browseSave()
{
	QString dir = QFileDialog::getExistingDirectory(this, "Choose a save directory", "", QFileDialog::DontResolveSymlinks);
	if (dir != "") {
		ui.savepath->setText(dir);
	}
}

void OptionsDlg::browseMedia()
{
	#ifdef WIN32
	QString dir = QFileDialog::getOpenFileName(this, "Choose media player", "", "Executables (*.exe);;All Files (*)");
	#else
	QString dir = QFileDialog::getOpenFileName(this, "Choose media player", "", "All Files (*)");
	#endif
	if (dir != "") {
		ui.mediaplayer->setText("\""+dir+"\"");
	}
}

void OptionsDlg::enableGuess(int state)
{
	if (state == Qt::Checked) {
		ui.guessalbum->setEnabled(true);
	} else {
		ui.guessalbum->setEnabled(false);
	}	
}
