
  (**************************************************************************)
  (*  ocaml-dtools                                                          *)
  (*  Copyright (C) 2003-2006  The Savonet Team                             *)
  (**************************************************************************)
  (*  This program is free software; you can redistribute it and/or modify  *)
  (*  it under the terms of the GNU General Public License as published by  *)
  (*  the Free Software Foundation; either version 2 of the License, or     *)
  (*  any later version.                                                    *)
  (**************************************************************************)
  (*  Contact: stephane.gimenez@ens-lyon.fr                                 *)
  (**************************************************************************)

(* $Id: dtools.mli 2912 2007-03-02 13:51:00Z metamorph68 $ *)

(**
  ocaml-dtools.
  @author Stephane Gimenez
*)

(**
  Initialisation management module.
  Allows to define procedures that must be executed at start up, and
  procedures that are to be executed at exit to have a clean quit.
*)
module Init :
sig

  type t

  val start : t
  val stop : t

  val make :
    ?name:string ->
    ?depends:(t list) -> ?triggers:(t list) ->
    ?after:(t list) -> ?before:(t list) ->
    (unit -> unit) -> t
    (**
      Defines a init atom associated with the given [(unit -> unit)]
      procedure, which eventualy depends on others atoms (these atoms
      will be executed before the one currently defined) an triggers
      other atoms (these atoms will be executed after the one currently
      defined). [after] and [before] allow to register the currently
      defined atom in the depend and triggers lists of other atoms.
    *)

  val at_start :
    ?name:string ->
    ?depends:(t list) -> ?triggers:(t list) ->
    ?after:(t list) -> ?before:(t list) ->
    (unit -> unit) -> t
    (**
      Same as [make] plus a shortcut for "after Init.start".
    *)

  val at_stop :
    ?name:string ->
    ?depends:(t list) -> ?triggers:(t list) ->
    ?after:(t list) -> ?before:(t list) ->
    (unit -> unit) -> t
    (**
      Same as [make] plus a shortcut for "before Init.stop".
    *)

  val exec : t -> unit
    (**
      Launch the execution of a given init atom.
    *)

  val init : ?prohibit_root:bool -> (unit -> unit) -> unit
    (**
      This fuction must be used to launch the main procedure of the
      program. It first execute the registered start atoms, then call
      the main procedure, then execute the registered stop atoms.
      Exceptions raised by the main procedure are catched, in order to
      close properly even in such cases. Exceptions are raised again
      after cleaning.
      When invoqued with [~prohibit_root:true], it checks for root access
      rights (euid, egid) and exit in this case.
    *)

  exception StartError of exn
  exception StopError of exn
end

(**
  Configuration management module.
*)
module Conf :
sig

  type t = string

  val cons : t -> t -> t

  exception Wrong_Conf of t * string
  exception File_Wrong_Conf of t * int * string
  exception Undefined of t * string

  val set_int : ?root:string -> t -> int -> unit
  val set_float : ?root:string -> t -> float -> unit
  val set_bool : ?root:string -> t -> bool -> unit
  val set_string : ?root:string -> t -> string -> unit
  val set_list : ?root:string -> t -> string list -> unit
    (**
      Links a value to the configuration keys.
    *)

  val get_int : ?root:t -> ?default:int -> t -> int
  val get_float : ?root:t -> ?default:float -> t -> float
  val get_bool : ?root:t -> ?default:bool -> t -> bool
  val get_string : ?root:t -> ?default:string -> t -> string
  val get_list : ?root:t -> ?default:string list -> t -> string list
    (**
      Retrieves the value associated with the given configuration key.
      Using the default value if the key was not defined. Raising
      the [Undefined] exception else.
    *)

  val ref_int : ?root:t -> ?default:int -> t -> int ref
  val ref_float : ?root:t -> ?default:float -> t -> float ref
  val ref_bool : ?root:t -> ?default:bool -> t -> bool  ref
  val ref_string : ?root:t -> ?default:string -> t -> string ref
  val ref_list : ?root:t -> ?default:string list -> t -> string list ref
    (**
      Retrieves a reference to the value associated with the given
      configuration key.
    *)

  val conf : string -> unit
    (**
      Adds a value to the configuration keys, according to the given
      correctly formated string: "type key :value"
      Raises [Wrong_Conf] in badly formated cases.
    *)
  val read_file : string -> unit
    (**
      Reads configuration values from the file associated with the given
      filename.
      Raises [File_Wrong_Conf] with filename line and and error message
      in case of a bad configuration file.
    *)

end

module Var :
sig

  type kind = Bool | Int | Float | String | List

  exception Duplicate_definition

  val register   : string -> kind -> unit

  exception Type_error of string*kind

  val set_bool   : string -> bool -> unit
  val set_int    : string -> int -> unit
  val set_float  : string -> float -> unit
  val set_string : string -> string -> unit
  val set_list   : string -> string list -> unit

end

module Log :
sig

  val log : ?label:string -> int -> string -> unit
    (**
      Logs the given string associated with a optional label and a log level.
    *)
  val logl : ?label:string -> int -> string Lazy.t -> unit
    (**
      Same as log with lazy evaluation of the string.
    *)

  val logged : string -> int -> bool
    (**
      [logged label level] returns wether the log level [level] for
      logging instructions associated with the [label] label will be
      considered for logging.
    *)

  val f : ('a, unit, string) format -> 'a
    (**
      Printf.sprintf equivalent that should be used for logging.
    *)

  val start : Init.t
    (**
      An atom that starts the logging.
    *)

  val stop : Init.t
    (**
      An atom that stops the logging.
    *)

end

module Options :
sig

  val list : (string * Arg.spec * string) list

end
