open Ocamldap;;
open Schemaparser;;
open String;;

(* types used throughout the library *)
(* add types *)
type op = string * string list
type op_lst = op list

(* change type for ldap entry *)
type changetype = ADD | DELETE | MODIFY | MODDN | MODRDN

class type ldapentry_t =
object
  method add : op_lst -> unit
  method delete : op_lst -> unit
  method replace : op_lst -> unit
  method attributes : string list
  method exists : string -> bool
  method get_value : string -> string list
  method changes : (Ocamldap.mod_op * string * string list) list
  method changetype : changetype
  method set_changetype : changetype -> unit
  method flush_changes : unit
  method dn : string
  method set_dn : string -> unit
  method print : unit
end;;

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* ldap entry object *)
class ldapentry = 
object (self)
  val mutable dn = ""
  val mutable data = Hashtbl.create 50
  val mutable changes = []
  val mutable changetype = ADD


  method private push_change (t:mod_op) lst = 
    match changetype with
	MODIFY -> (match lst with
		       [] -> ()
		     | (attr, values) :: tail -> 
			 changes <- (t, attr, values) :: changes; self#push_change t tail)
      | _ -> ()

  method changetype = changetype;
  method set_changetype typ = changetype <- typ
  method flush_changes = changes <- []
  method changes = changes

  method exists x = Hashtbl.mem data (lowercase x)
  method add (x:op_lst) = 
    let rec do_add (x:op_lst) =
      match x with
	  [] -> ()
	| (name, value) :: lst -> 
	    let lcname = lowercase name in
	      try
		Ulist.addlst (Hashtbl.find data lcname) value; do_add lst
	      with Not_found ->
		let current = Ulist.create 5 in
		  Hashtbl.add data lcname current; Ulist.addlst current value; do_add lst
    in
      do_add x; self#push_change `ADD x

  method delete (x:op_lst) = 
    let rec do_delete x = 
      match x with
	  [] -> ()
	| (attr, values) :: lst ->
	    (let lcname = lowercase attr in
	       match values with
		   [] -> Hashtbl.remove data lcname
		 | _  -> ((try List.iter (Ulist.remove (Hashtbl.find data lcname)) values
			   with Not_found -> ());
			  (match Ulist.tolst (Hashtbl.find data lcname) with
			       [] -> Hashtbl.remove data lcname
			     | _  -> ());
			  do_delete lst))
    in
      do_delete x; self#push_change `DELETE x

  method replace (x:op_lst) = 
    let rec do_replace x = 
      match x with
	  [] -> ()
	| (attr, values) :: lst -> let n = Ulist.create 5 in
	    Ulist.addlst n values; Hashtbl.replace data (lowercase attr) n; 
	    do_replace lst;
    in
      do_replace x; self#push_change `REPLACE x

  method attributes = 
    let keys hash = 
      let cur = ref [] in
      let key k _ = cur := k :: !cur in
	Hashtbl.iter key hash; !cur
    in
      keys data      

  method get_value attr = Ulist.tolst (Hashtbl.find data (lowercase attr))
  method set_dn x = dn <- x
  method dn = dn
  method print =
    print_endline ("dn: " ^ self#dn);
    (List.iter 
       (fun a -> 
	  (List.iter 
	     (fun b -> print_endline (a ^ ": " ^ b)) 
	     (self#get_value a)))
       self#attributes)

end

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* a connection to an ldap server *)
let finalise_ldapcon con = con#unbind
class ldapcon ?(version = 3) ?(async = false) ?(port = 389) host = 
object (self)
  val hst = host
  val prt = port
  val ver = version
  val async = async
  val mutable bdn = ""
  val mutable pwd = ""
  val mutable mth = `SIMPLE
  val mutable bound = true
  val mutable con = init ~version: version ~port: port host

  initializer Gc.finalise finalise_ldapcon self
  method private reconnect =
    unbind con;
    con <- init ~version: ver ~port: prt hst;
    self#bind bdn ~cred: pwd ~meth: mth

  method unbind = if bound then (unbind con;bound <- false)

  method update_entry (e:ldapentry) = 
    try self#modify e#dn (List.rev e#changes); e#flush_changes
    with LDAP_Failure(`SERVER_DOWN) -> 
      self#reconnect;self#update_entry e
		      
  method bind ?(cred = "") ?(meth = `SIMPLE) dn =
    bind_s ~who: dn ~cred: cred ~auth_method: meth con;
    bdn <- dn; pwd <- cred; mth <- meth

  method add (entry: ldapentry) = 
    let rec mkattrs entry attrs =
      match attrs with
	  [] -> []
	| attr :: tail ->
	    (`ADD, attr, entry#get_value attr) :: mkattrs entry tail
    in
      try add_s con entry#dn (mkattrs entry entry#attributes)
      with LDAP_Failure(`SERVER_DOWN) ->
	self#reconnect;self#add entry
	
  method delete dn =
    try delete_s con dn
    with LDAP_Failure(`SERVER_DOWN) ->
      self#reconnect;self#delete dn

  method modify dn mods = 
    try modify_s con dn mods
    with LDAP_Failure(`SERVER_DOWN) ->
      self#reconnect;self#modify dn mods

  method modrdn dn ?(deleteoldrdn = true) newrdn =
    try modrdn2_s con ~dn:dn ~newdn:newrdn ~deleteoldrdn:deleteoldrdn
    with LDAP_Failure(`SERVER_DOWN) ->
      self#reconnect;self#modrdn dn ~deleteoldrdn:deleteoldrdn newrdn

  method search 
    ?(scope = `SUBTREE)
    ?(attrs = [])
    ?(attrsonly = false)
    ?(base = "")
    filter =
    let to_entry (ent: entry) =
      let rec add_attrs attrs entry =
	match attrs with
	    {attr_name = name; attr_values = values} :: tail ->
	      entry#add [(name, (Array.to_list values))]; add_attrs tail entry
 	  | [] -> entry#set_changetype MODIFY; entry
      in
	match ent with
	    {entry_dn = dn; entry_attrs = attrs} -> 
	      let entry = new ldapentry in
		entry#set_dn dn; add_attrs (Array.to_list attrs) entry
    in
      try 
	List.rev_map to_entry (search_s 
				 ~scope: scope
				 ~base: base
				 ~attrs: attrs
				 ~attrsonly: attrsonly
				 con
				 filter)
      with LDAP_Failure(`SERVER_DOWN) ->
	self#reconnect;self#search ~scope: scope ~attrs: attrs
	  ~attrsonly: attrsonly ~base: base filter

  method schema = 
    try 
      if ver = 3 then
	let schema_base = (match (self#search 
				    ~base: "" 
				    ~scope: `BASE 
				    ~attrs: ["subschemasubentry"] 
				    "(objectclass=*)") with
			       [e] -> List.hd (e#get_value "subschemasubentry")
			     |  _  -> raise Not_found) in
	  (match (self#search
		    ~base: schema_base
		    ~scope: `BASE
		    ~attrs: ["objectClasses";"attributeTypes";
			     "matchingRules";"ldapSyntaxes"]
		    "(objectclass=*)") with
	       [e] -> readSchema (e#get_value "objectclasses") (e#get_value "attributetypes")
	     |  _  -> raise Not_found)
      else
	raise Not_found
    with LDAP_Failure(`SERVER_DOWN) -> self#reconnect;self#schema
 
  method rawschema = 
    try 
      if ver = 3 then
	let schema_base = (match (self#search 
				    ~base: "" 
				    ~scope: `BASE 
				    ~attrs: ["subschemasubentry"] 
				    "(objectclass=*)") with
			       [e] -> List.hd (e#get_value "subschemasubentry")
			     |  _  -> raise Not_found) in
	  (match (self#search
		    ~base: schema_base
		    ~scope: `BASE
		    ~attrs: ["objectClasses";"attributeTypes";
			     "matchingRules";"ldapSyntaxes"]
		    "(objectclass=*)") with
	       [e] -> e
	     |  _  -> raise Not_found)
      else
	raise Not_found
    with LDAP_Failure(`SERVER_DOWN) -> self#reconnect;self#rawschema
end;;

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* A schema checking entry:
   An entry which validates its validity against the server's
   schema *)
module OrdStr =
struct
  type t = Oid.t
  let compare = Oid.compare
end;;


(* type for a set of strings *)
module Setstr = Set.Make (OrdStr);;

(* schema checking flavor *)
type scflavor = Optimistic (* attempt to find objectclasses which make illegal
			      attributes legal, delete them if no objectclass can
			      be found *)
		| Pessimistic (* delete any illegal attributes, do not add 
				 objectclasses to make them legal*)

(* for the schema checker, should never be seen by
   the user *)
exception Invalid_objectclass of string
exception Invalid_attribute of string
exception Single_value of string

let attrToOid schema (attr:Lcstring.t) =
  try (Hashtbl.find schema.attributes attr).at_oid
  with Not_found -> raise (Invalid_attribute (Lcstring.to_string attr));;

let oidToAttr schema (attr:Oid.t) = 
  List.hd (Hashtbl.find schema.attributes_byoid attr).at_name;;

let ocToOid schema (oc:Lcstring.t) =
  try (Hashtbl.find schema.objectclasses oc).oc_oid
  with Not_found -> raise (Invalid_objectclass (Lcstring.to_string oc));;

let oidToOc schema (oc:Oid.t) =
  List.hd (Hashtbl.find schema.objectclasses_byoid oc).oc_name

let getOc schema (oc:Lcstring.t) =
  try Hashtbl.find schema.objectclasses oc
  with Not_found -> raise (Invalid_objectclass (Lcstring.to_string oc));;

let getAttr schema (attr:Lcstring.t) =
  try Hashtbl.find schema.attributes attr
  with Not_found -> raise (Invalid_attribute (Lcstring.to_string attr));;

let rec setOfList (list:Oid.t list) set = 
  match list with
      a :: tail -> setOfList tail (Setstr.add a set)
    | []  -> set;;

class scldapentry schema =
object (self)
  inherit ldapentry as super
  val schemaAttrs = Hashtbl.create 50
  val schema = schema
  val mutable consistent = false
  (* the set of all attibutes actually present *)
  val mutable present       = Setstr.empty
  (* the set of all musts from all objectclasses on the entry *)
  val mutable must          = Setstr.empty
  (* the set of all mays from all objectclasses on the entry *)
  val mutable may           = Setstr.empty
  (* the set of required objectclasses *)
  val mutable requiredOcs   = Setstr.empty
  (* present objectclasses *)
  val mutable presentOcs    = Setstr.empty

  (* must + may *)
  val mutable all_allowed   = Setstr.empty
  (* must - (present * must) *)
  val mutable missingAttrs  = Setstr.empty
  (* requiredOcs - (presentOcs * requiredOcs) *)
  val mutable missingOcs    = Setstr.empty
  (* any objectclass which depends on a missing objectclass *)
  val mutable illegalOcs    = Setstr.empty
  (* present - (present * all_allowed) *)
  val mutable illegalAttrs  = Setstr.empty

  (* schema checking is best expressed as set manipulations.
     I can ascert this having implimented it in other ways *)
  method private update_condition =
    let generate_present attrs schema = 
      setOfList (List.rev_map (attrToOid schema) attrs) Setstr.empty in
    let rec generate_mustmay ocs schema set must =
      match ocs with
	  oc :: tail -> 
	    let musts = setOfList 
			  (List.rev_map 
			     (fun attr -> attrToOid schema attr)
			     (if must then (getOc schema oc).oc_must
			      else (getOc schema oc).oc_may))
			  Setstr.empty in
	      generate_mustmay tail schema (Setstr.union musts set) must
	| [] -> set
    in
    let rec lstRequired schema (oc: Lcstring.t) =
      oc :: (List.flatten (List.rev_map 
			     (fun sup -> lstRequired schema sup) 
			     (getOc schema oc).oc_sup))
    in
    let rec generate_requiredocs schema ocs =
      setOfList 
	(List.rev_map 
	   (ocToOid schema)
	   (List.flatten (List.rev_map (lstRequired schema) ocs))) Setstr.empty
    in
    let generate_illegal_oc missing schema ocs =
      let is_illegal_oc missing schema oc =
	let supchain = lstRequired schema oc in
	  List.exists
	    (fun mis ->
	       List.exists ((=) mis)
	       supchain)
	    missing
      in
	List.filter (is_illegal_oc missing schema) ocs
    in

      present      <- (generate_present 
			 (List.rev_map (Lcstring.of_string) super#attributes) 
			 schema);
      must         <- (generate_mustmay 
			 (List.rev_map 
			    (Lcstring.of_string) 
			    (super#get_value "objectclass"))
			 schema
			 Setstr.empty
			 true);
      may          <- (generate_mustmay 
			 (List.rev_map 
			    (Lcstring.of_string) 
			    (super#get_value "objectclass"))
			 schema
			 Setstr.empty
			 false);
      all_allowed  <- Setstr.union must may;
      missingAttrs <- Setstr.diff must (Setstr.inter must present);
      illegalAttrs <- Setstr.diff present (Setstr.inter all_allowed present);
      requiredOcs  <- (generate_requiredocs 
			 schema 
			 (List.rev_map
			    (Lcstring.of_string) 
			    (super#get_value "objectclass")));
      presentOcs   <- (setOfList 
			 (List.rev_map 
			    (fun attr -> ocToOid schema (Lcstring.of_string attr)) 
			    (super#get_value "objectclass"))
			 Setstr.empty);
      missingOcs   <- Setstr.diff requiredOcs (Setstr.inter requiredOcs presentOcs);
      illegalOcs   <- (setOfList
			 (List.rev_map
			    (ocToOid schema)
			    (generate_illegal_oc 
			       (List.rev_map 
				  (fun x -> Lcstring.of_string (oidToOc schema x))
				  (Setstr.elements missingOcs))
			       schema
			       (List.rev_map
				  (Lcstring.of_string)
				  (super#get_value "objectclass"))))
			 Setstr.empty);
      if Setstr.is_empty (Setstr.union missingAttrs illegalAttrs) then
	consistent <- true
      else
	consistent <- false

  method private drive_updatecon =
    try self#update_condition
    with Invalid_objectclass(s) -> super#delete [("objectclass",[s])]; self#drive_updatecon
      | Invalid_attribute(s) -> super#delete [(s,[])]; self#drive_updatecon

  method private reconsile_illegal flavor =
    let find_in_oc oc attr = (List.exists
				((=) (Lcstring.of_string attr)) 
				oc.oc_must) || 
			     (List.exists
				((=) (Lcstring.of_string attr))
				oc.oc_may) in
    let find_oc schema attr = 
      let oc = ref (Lcstring.of_string "") in
	Hashtbl.iter 
	  (fun key valu -> 
	     if (find_in_oc valu attr) then oc := key)
	  schema.objectclasses;
	if !oc = (Lcstring.of_string "") then raise Not_found;
	!oc
    in
	match flavor with 
	    Optimistic ->
	      if not (Setstr.is_empty illegalAttrs) then
		((List.iter (* add necessary objectclasses *)
		   (fun oc -> super#add [("objectclass",[(Lcstring.to_string oc)])])
		   (List.rev_map
		      (fun attr -> 
			 try find_oc schema attr 
			 with Not_found -> raise (Invalid_attribute attr))
		      (List.rev_map (oidToAttr schema) (Setstr.elements illegalAttrs))));
		 self#drive_updatecon);
	      (* add any objectclasses the ones we just added are dependant on *)
	      if not (Setstr.is_empty missingOcs) then
		((List.iter
		    (fun oc -> super#add [("objectclass", [oc])])
		    (List.rev_map (oidToOc schema) (Setstr.elements missingOcs)));
		 self#drive_updatecon);
	  | Pessimistic ->
	      (List.iter
		 (fun oc -> super#delete [("objectclass",[oc])])
		 (List.rev_map (oidToOc schema) (Setstr.elements illegalOcs)));
	      self#drive_updatecon;
	      (List.iter (* remove disallowed attributes *)
		 (fun attr -> super#delete [(attr, [])])
		 (List.rev_map (oidToAttr schema) (Setstr.elements illegalAttrs)));
	      self#drive_updatecon

  method private drive_reconsile flavor =
    try self#reconsile_illegal flavor
    with Invalid_attribute(a) -> (* remove attributes for which there is no objectclass *)
      (super#delete [(a, [])];
       self#drive_updatecon;
       self#drive_reconsile flavor)

  (* for debugging *)
  method private getCondition = 
    let printLst lst = List.iter print_endline lst in
      print_endline "MAY";
      printLst (List.rev_map (oidToAttr schema) (Setstr.elements may));
      print_endline "PRESENT";
      printLst (List.rev_map (oidToAttr schema) (Setstr.elements present));
(*      printLst (Setstr.elements present);*)
      print_endline "MUST";
      printLst (List.rev_map (oidToAttr schema) (Setstr.elements must));
(*      printLst (Setstr.elements must);*)
      print_endline "MISSING";
      printLst (List.rev_map (oidToAttr schema) (Setstr.elements missingAttrs));
(*      printLst (Setstr.elements missingAttrs);*)
      print_endline "ILLEGAL";
      printLst (List.rev_map (oidToAttr schema) (Setstr.elements illegalAttrs));
      print_endline "REQUIREDOCS";
(*      printLst (List.rev_map (oidToOc schema) (Setstr.elements requiredOcs));*)
      printLst (List.rev_map Oid.to_string (Setstr.elements requiredOcs));
      print_endline "PRESENTOCS";
(*      printLst (List.rev_map (oidToOc schema) (Setstr.elements presentOcs));*)
      printLst (List.rev_map Oid.to_string (Setstr.elements presentOcs));
      print_endline "MISSINGOCS";
(*      printLst (List.rev_map (oidToOc schema) (Setstr.elements missingOcs));*)
      printLst (List.rev_map Oid.to_string (Setstr.elements missingOcs));
      print_endline "ILLEGALOCS";
(*      printLst (List.rev_map (oidToOc schema) (Setstr.elements illegalOcs))*)
      printLst (List.rev_map Oid.to_string (Setstr.elements illegalOcs));

  (* for debugging *)
  method private getData = (must, may, present, missingOcs)

  method of_entry (e:ldapentry) =
    super#set_dn (e#dn);
    super#set_changetype ADD;
    (List.iter
       (fun attr -> 
	  super#add [(attr, (e#get_value attr))])
       e#attributes);
    self#update_condition

  (* raise an exception if the user attempts to have more than
     one value in a single valued attribute. *)
  method private single_val_check (x:op_lst) consider_present =
    let check op =
      let attr = getAttr schema (Lcstring.of_string (fst op)) in
	(if attr.at_single_value then
	   (match op with
		(attr, v1 :: v2 :: tail) -> false
	      | (attr, v1 :: tail) -> 
		  (if consider_present && (super#exists attr) then
		     false
		   else true)
	      | _ -> true)
	 else true)
    in
      match x with
	  op :: tail -> (if not (check op) then
			   raise (Single_value (fst op))
			 else self#single_val_check tail consider_present)
	|  [] -> ()

  method add x = 
    self#single_val_check x true;super#add x;
    self#drive_updatecon;self#drive_reconsile Optimistic
      
  method delete x = 
    super#delete x;self#drive_updatecon;self#drive_reconsile Pessimistic

  method replace x = 
    self#single_val_check x false;super#replace x;
    self#drive_updatecon;self#drive_reconsile Optimistic

  method get_value x =
    try super#get_value x with Not_found ->
      if (Setstr.mem (attrToOid schema (Lcstring.of_string x)) missingAttrs) then
	["required"]
      else
	raise Not_found

  method attributes =
    List.rev_append
      super#attributes
      (List.rev_map
	 (fun a -> oidToAttr schema a) 
	 (Setstr.elements missingAttrs))

  method list_missing = Setstr.elements missingAttrs
  method list_allowed = Setstr.elements all_allowed
  method list_present = Setstr.elements present
  method is_missing x = 
    Setstr.mem (attrToOid schema (Lcstring.of_string x)) missingAttrs
  method is_allowed x = 
    Setstr.mem (attrToOid schema (Lcstring.of_string x)) all_allowed
end;;

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* a high level interface for accounts, and services in the directory *)

type generator = {gen_name:string;
		  required:string list;
		  genfun:(ldapentry_t -> string list)};;

type service = {svc_name: string;
		static_attrs: (string * (string list)) list;
		generate_attrs: string list;
		depends: string list};;

type generation_error = Missing_required of string list
			| Generator_error of string

exception No_generator of string;;
exception Generation_failed of generation_error;;
exception No_service of string;;
exception Service_dep_unsatisfiable of string;;
exception Generator_dep_unsatisfiable of string * string;;
exception Cannot_sort_dependancies of (string list);;

let diff_values convert_to_oid convert_from_oid attr attrvals svcvals =
    (attr, (List.rev_map
	      convert_from_oid
	      (Setstr.elements
		 (Setstr.diff
		    svcvals
		    (Setstr.inter svcvals attrvals)))))

(* compute the intersection of values between an attribute and a service,
   you need to pass this function as an argument to apply_set_op_to_values *)
let intersect_values convert_to_oid convert_from_oid attr attrvals svcvals =
  (attr, (List.rev_map
	    convert_from_oid
	    (Setstr.elements
	       (Setstr.inter svcvals attrvals))))

(* this function allows you to apply a set operation to the values of an attribute, and 
   the static values on a service *)
let apply_set_op_to_values schema (attr:string) e svcval opfun =
  let lc = String.lowercase in
  let convert_to_oid = (match lc ((getAttr schema (Lcstring.of_string attr)).at_equality) with
			    "objectidentifiermatch" -> 
			      (fun oc -> ocToOid schema (Lcstring.of_string oc))
			  | "caseexactia5match" -> Oid.of_string
			  | _ -> (fun av -> Oid.of_string (lc av)))
  in
  let convert_from_oid = (match lc ((getAttr schema (Lcstring.of_string attr)).at_equality) with
			      "objectidentifiermatch" -> (fun av -> oidToOc schema av)
			    | "caseexactia5match" -> Oid.to_string
			    | _ -> Oid.to_string)
  in
  let attrvals = setOfList
		   (List.rev_map
		      convert_to_oid
		      (try e#get_value attr with Not_found -> []))
		   Setstr.empty 
  in
  let svcvals = setOfList
		  (List.rev_map convert_to_oid (snd svcval))
		  Setstr.empty 
  in
    opfun convert_to_oid convert_from_oid attr attrvals svcvals

class ldapaccount 
  schema 
  (generators:(string, generator) Hashtbl.t)
  (services:(string, service) Hashtbl.t) =
object (self)
  inherit scldapentry schema as super
  val mutable toGenerate = Setstr.empty
  val mutable neededByGenerators = Setstr.empty
  val services = services
  val generators = generators

(* evaluates the set of missing attributes to see if any of
   them can be generated, if so, it adds them to be generated *)
  method private resolve_missing =
    let generate_togenerate generators missing togenerate =
      let find_generatable_dep generators generator =
	(List.rev_map
	   (fun e -> attrToOid schema (Lcstring.of_string e))
	   (List.filter
	      (fun g ->
		 if ((Hashtbl.mem generators g) && 
		     (not (Setstr.mem
			     (attrToOid schema (Lcstring.of_string g))
			     (setOfList self#list_present Setstr.empty)))) then
		   true
		 else false)
	      (Hashtbl.find generators generator).required))
      in
      let rec find_generatable_deps generators genlst =
	(List.flatten
	   (List.rev_map
	      (find_generatable_dep generators)
	      genlst))
      in
      let generateing = (List.filter
			   (fun gen -> 
			      if (Hashtbl.mem generators (lowercase (oidToAttr schema gen))) then
				true
			      else false)
			   (List.rev_append
			      missing
			      (Setstr.elements togenerate)))
      in
	setOfList
	  (List.rev_append generateing (find_generatable_deps
					  generators
					  (List.rev_map
					     (fun e -> lowercase (oidToAttr schema e))
					     generateing)))
	  Setstr.empty
    in
    let generate_missing togen generators =
      setOfList
	(Hashtbl.fold 
	   (fun key valu requiredlst -> 
	      if Setstr.mem (attrToOid schema (Lcstring.of_string valu.gen_name)) togen then
		List.rev_append
		  requiredlst
		  (List.rev_map
		     (fun x -> try
			attrToOid schema (Lcstring.of_string x)
		      with Invalid_attribute a -> 
			raise (Generator_dep_unsatisfiable (key, a)))
		     valu.required)
	      else
		requiredlst)
	   generators [])
	Setstr.empty
    in
      toGenerate <- generate_togenerate generators super#list_missing toGenerate;
      neededByGenerators <- generate_missing toGenerate generators;

  method list_missing = 
    let allmissing = 
      Setstr.union neededByGenerators (setOfList super#list_missing Setstr.empty) 
    in
      Setstr.elements
	(Setstr.diff
	   allmissing 
	   (Setstr.inter
	      allmissing
	      (Setstr.union 
		 toGenerate 
		 (setOfList super#list_present Setstr.empty))))
  method attributes =
    (List.rev_map (oidToAttr schema)
       (Setstr.elements
	  (Setstr.union toGenerate
	     (setOfList 
		(List.rev_map
		   (fun a -> attrToOid schema (Lcstring.of_string a))
		   super#attributes)
		Setstr.empty))))
  method is_missing x = (not (Setstr.mem
				(attrToOid schema (Lcstring.of_string x)) 
				toGenerate)) 
			|| (super#is_missing x)

  method generate =
    let sort_genlst generators unsatisfied =
      let satisfied alreadysatisfied present deps =
	List.for_all
	  (fun dep -> 
	     (List.mem dep alreadysatisfied) || 
	     (List.mem (attrToOid schema (Lcstring.of_string dep)) (present)))
	  deps
      in
      let rec sort present ordtogen unsatisfied =
	match unsatisfied with
	    [] -> ordtogen
	  | todo ->
	      let (aresat, notyet) =
		(List.partition
		   (fun attr ->
		      (satisfied ordtogen present
			 (Hashtbl.find generators attr).required))
		   todo)
	      in
		match aresat with
		    [] -> raise (Cannot_sort_dependancies notyet)
		  | _ -> sort present (ordtogen @ aresat) notyet
      in
	sort (self#list_present) [] unsatisfied
    in
      match self#list_missing with
	  [] -> 
	    (List.iter
	       (fun attr ->
		  self#add [(attr, (Hashtbl.find generators attr).genfun (self:>ldapentry_t))])
	       (sort_genlst generators
		  (List.rev_map
		     (fun elt -> String.lowercase (oidToAttr schema elt))
		     (Setstr.elements toGenerate))));
	    toGenerate <- Setstr.empty
	| a  -> raise (Generation_failed
			 (Missing_required (List.rev_map (oidToAttr schema) a)))

  method get_value x =
    if (Setstr.mem (attrToOid schema (Lcstring.of_string x)) toGenerate) then
      ["generate"]
    else
      super#get_value x

(* adapt the passed in service to the current state of the entry
   this may result in a service with applies no changes. The entry
   may already have the service. *)
  method adapt_service svc =    
      {svc_name=svc.svc_name;
       static_attrs=(List.filter
			  (fun cons ->
			     match cons with
				 (attr, []) -> false
			       | _          -> true)
			  (List.rev_map
			     (fun cons -> apply_set_op_to_values schema (fst cons) self cons diff_values)
			     svc.static_attrs));
       generate_attrs=(List.filter
			 (fun attr -> 
			    (try (ignore (super#get_value attr));false
			     with Not_found -> true))			
			 svc.generate_attrs);
       depends=svc.depends}

(* add a service to the account, if they already satisfy the service
   then do nothing *)			     
  method add_service svc =
    let service = try Hashtbl.find services (lowercase svc)
    with Not_found -> raise (No_service svc) in
      (try List.iter (self#add_service) service.depends
       with (No_service x) -> raise (Service_dep_unsatisfiable x));
      let adaptedsvc = self#adapt_service service in
	(let do_adds a =
	   let singlevalu = 
	     (List.filter 
		(fun attr -> (getAttr schema
			     (Lcstring.of_string (fst attr))).at_single_value) a)
	   in
	   let multivalued = 
	     (List.filter 
		(fun attr -> not (getAttr schema
				 (Lcstring.of_string (fst attr))).at_single_value) a)
	   in
	     self#add multivalued;
	     self#replace singlevalu
	 in
	   do_adds adaptedsvc.static_attrs);
	(match adaptedsvc.generate_attrs with
	     [] -> ()
	   | a  -> List.iter (self#add_generate) a)

  method delete_service svc =
    let find_deps services service =
      (Hashtbl.fold
	 (fun serv svcstruct deplst ->
	    if (List.exists ((=) service) svcstruct.depends) then
	      serv :: deplst
	    else
	      deplst)
	 services [])
    in
    let service = try Hashtbl.find services (lowercase svc)
    with Not_found -> raise (No_service svc) in
      (List.iter (self#delete_service) (find_deps services svc));
      (List.iter
	 (fun e -> match e with
	      (attr, []) -> ()
	    | a -> (try (ignore (super#get_value (fst a)));super#delete [a]
		    with Not_found -> ()))
	 (List.rev_map
	    (fun cons ->
	       apply_set_op_to_values schema (fst cons) self cons intersect_values)
	    service.static_attrs));
      (List.iter
	 (fun attr -> 
	    (try (match self#get_value attr with
		      ["generate"] -> self#delete_generate attr
		    | _ -> super#delete [(attr, [])])
	     with Not_found -> ()))
	 service.generate_attrs)	     	     
      
  method of_entry e = super#of_entry e;self#resolve_missing
  method add_generate x = 
    (if (Hashtbl.mem generators (lowercase x)) then
       toGenerate <- Setstr.add (attrToOid schema (Lcstring.of_string x)) toGenerate
     else raise (No_generator x));
    self#resolve_missing
  method delete_generate x =
    let find_dep attr generators =
      (Hashtbl.fold
	 (fun key valu deplst ->
	    if (List.exists ((=) attr) valu.required) then
	      key :: deplst
	    else
	      deplst)
	 generators [])
    in
      (List.iter (self#delete_generate) (find_dep x generators));
      toGenerate <- 
      Setstr.remove
	(attrToOid schema (Lcstring.of_string x)) toGenerate

  method add x = (* add x, remove all attributes in x from the list of generated attributes *)
    super#add x; 
    (List.iter 
      (fun a -> 
	 toGenerate <- (Setstr.remove
			  (attrToOid schema (Lcstring.of_string (fst a)))
			  toGenerate))
       x);
    self#resolve_missing
  method delete x = super#delete x;self#resolve_missing
  method replace x = (* replace x, removeing it from the list of generated attrs *)
    super#replace x;
    (List.iter
       (fun a -> 
	  toGenerate <- (Setstr.remove
			   (attrToOid schema (Lcstring.of_string (fst a)))
			   toGenerate))
       x);
    self#resolve_missing
end;;
