
(* Basic operations over graphs *)

module type S = sig
  type g
  val transitive_closure : ?reflexive:bool -> g -> g
  val add_transitive_closure : ?reflexive:bool -> g -> g
  val mirror : g -> g
  val complement : g -> g
  val intersect : g -> g -> g
  val union : g -> g -> g
end

module Make(G : Sig.G)(B : Builder.S with module G = G) = struct

  (* Roy-Warshall's algorithm *)

  type g = G.t

  let add_transitive_closure ?(reflexive=false) g0 =
    let phi v g =
      let g = if reflexive then B.add_edge g v v else g in
      G.fold_succ
	(fun sv g -> G.fold_pred (fun pv g -> B.add_edge g pv sv) g v g) 
	g v g
    in
    G.fold_vertex phi g0 g0

  let transitive_closure ?(reflexive=false) g0 = 
    add_transitive_closure ~reflexive (B.copy g0)

  module H = Hashtbl.Make(G.V)

  (* copy all vertices of [g] and returns a hash table containing the map *)
  let copy_with_map g =
    let h = H.create 97 in
    let g' =
      G.fold_vertex 
	(fun v g' -> 
	   let v' = G.V.create (G.V.label v) in
	   H.add h v v'; 
	   B.add_vertex g' v') 
	g (B.empty ()) 
    in
    g', h

  let mirror g =
    if G.is_directed then begin
      let g', h = copy_with_map g in
      G.fold_edges_e
	(fun e g' -> 
	   let v1 = H.find h (G.E.src e) in
	   let v2 = H.find h (G.E.dst e) in
	   B.add_edge_e g' (G.E.create v2 (G.E.label e) v1))
	g g'
    end else
      B.copy g

  let complement g =
    let g', h = copy_with_map g in
    G.fold_vertex 
      (fun v g' ->
	 G.fold_vertex
	   (fun w g' -> 
	      if G.mem_edge g v w then g' 
	      else B.add_edge g' (H.find h v) (H.find h w))
	   g g')
      g g'

  let intersect g1 g2 = 
    G.fold_vertex
      (fun v g ->
	 try
	   let succ = G.succ_e g2 v in
	   G.fold_succ_e 
	     (fun e g -> 
		if List.mem e succ 
		then B.add_edge_e g e 
		else B.add_vertex g (G.E.dst e))
	     g1 v (B.add_vertex g v)
	 with Invalid_argument _ -> 
	   (* $v \notin g2$ *)
	   g)
      g1 (B.empty ())

  let union g1 g2 =
    let add g1 g2 = 
      (* add the graph [g1] in [g2] *)
      G.fold_vertex 
	(fun v g -> 
	   G.fold_succ_e (fun e g -> B.add_edge_e g e) g1 v (B.add_vertex g v))
	g1 g2
    in
    add g1 (add g2 (B.empty ()))

end

module P(G : Sig.P) = Make(G)(Builder.P(G))
module I(G : Sig.I) = Make(G)(Builder.I(G))
