(*
 * Graph: generic graph library
 * Copyright (C) 2004
 * Sylvain Conchon, Jean-Christophe Filliatre and Julien Signoles
 * 
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2, as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * See the GNU Library General Public License version 2 for more details
 * (enclosed in the file LGPL).
 *)

(* $Id: sig.mli,v 1.14 2004/02/23 10:06:30 signoles Exp $ *)

(** Signatures for graph implementations *)

(** {1 Signatures for graph implementations} *)

(** Common interface for all graph implementations *)

module type G = sig

  (** {2 Graph structure} *)

  (** Abstract type of graphs *)
  type t

  (** Vertices have type [V.t] and are labeled with type [V.label]
    (note that an implementation may identify the vertex with its
    label) *)
  module V : sig
    (** Vertices are [COMPARABLE] *)

    type t 
    val compare : t -> t -> int 
    val hash : t -> int 
    val equal : t -> t -> bool

    (** Vertices are labeled. *)

    type label
    val create : label -> t
    val label : t -> label
  end

  (** Edges have type [E.t] and are labeled with type [E.label].
      [src] (resp. [dst]) returns the origin (resp. the destination) of a
      given edge. *)
  module E : sig
    (** Edges are [ORDERED]. *)

    type t
    val compare : t -> t -> int

    (** Edges are directed. *)

    val src : t -> V.t
    val dst : t -> V.t

    (** Edges are labeled. *)

    type label
    val create : V.t -> label -> V.t -> t
      (** [create v1 l v2] creates an edge from [v1] to [v2] with label [l] *)
    val label : t -> label
  end

  (** Is this an implementation of directed graphs? *)
  val is_directed : bool

  (** {2 Size functions} *)

  val is_empty : t -> bool
  val nb_vertex : t -> int
  val nb_edges : t -> int

  (** Degree of a vertex *)

  val out_degree : t -> V.t -> int
    (** [out_degree g v] returns the out-degree of [v] in [g].
      Raise [Invalid_argument] if [v] is not in [g]. *)

  val in_degree : t -> V.t -> int
    (** [in_degree g v] returns the in-degree of [v] in [g].
      Raise [Invalid_argument] if [v] is not in [g]. *)

  (** {2 Membership functions} *)

  val mem_vertex : t -> V.t -> bool
  val mem_edge : t -> V.t -> V.t -> bool
  val mem_edge_e : t -> E.t -> bool

  (** {2 Successors and predecessors} *)

  val succ : t -> V.t -> V.t list
    (** [succ g v] returns the successors of [v] in [g].
        Raise [Invalid_argument] if [v] is not in [g]. *)

  val pred : t -> V.t -> V.t list
    (** [pred g v] returns the predecessors of [v] in [g].
        Raise [Invalid_argument] if [v] is not in [g]. *)

  (** Labeled edges going from/to a vertex *)

  val succ_e : t -> V.t -> E.t list
    (** [succ_e g v] returns the edges going from [v] in [g].
        Raise [Invalid_argument] if [v] is not in [g]. *)

  val pred_e : t -> V.t -> E.t list
    (** [pred_e g v] returns the edges going to [v] in [g].
        Raise [Invalid_argument] if [v] is not in [g]. *)

  (** {2 Graph iterators} *)

  (** iter/fold on all vertices/edges of a graph *)

  val iter_vertex : (V.t -> unit) -> t -> unit
  val iter_edges : (V.t -> V.t -> unit) -> t -> unit
  val fold_vertex : (V.t -> 'a -> 'a) -> t  -> 'a -> 'a
  val fold_edges : (V.t -> V.t -> 'a -> 'a) -> t -> 'a -> 'a

  (** map iterator on vertex *)
  val map_vertex : (V.t -> V.t) -> t -> t

  (** iter/fold on all labeled edges of a graph *)

  val iter_edges_e : (E.t -> unit) -> t -> unit
  val fold_edges_e : (E.t -> 'a -> 'a) -> t -> 'a -> 'a

  (** {2 Vertex iterators} 

    Each iterator [iterator f v g] iters [f] to the successors/predecessors
    of [v] in the graph [g] and raises [Invalid_argument] if [v] is not in
    [g]. *)

  (** iter/fold on all successors/predecessors of a vertex. *)

  val iter_succ : (V.t -> unit) -> t -> V.t -> unit
  val iter_pred : (V.t -> unit) -> t -> V.t -> unit
  val fold_succ : (V.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a
  val fold_pred : (V.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a

  (** iter/fold on all edges going from/to a vertex. *)

  val iter_succ_e : (E.t -> unit) -> t -> V.t -> unit
  val fold_succ_e : (E.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a
  val iter_pred_e : (E.t -> unit) -> t -> V.t -> unit
  val fold_pred_e : (E.t -> 'a -> 'a) -> t -> V.t -> 'a -> 'a

end

(** Persistent (i.e. immutable) implementation *)

module type P = sig
  include G

  val empty : t
    (** The empty graph. *)

  val add_vertex : t -> V.t -> t
    (** [add_vertex g v] adds the vertex [v] from the graph [g].
      Just return [g] if [v] is already in [g]. *)

  val remove_vertex : t -> V.t -> t
    (** [remove g v] removes the vertex [v] from the graph [g] 
      (and all the edges going from [v] in [g]).
      Just return [g] if [v] is not in [g]. *)

  val add_edge : t -> V.t -> V.t -> t
    (** [add_edge g v1 v2] adds an edge from the vertex [v1] to the vertex [v2]
      in the graph [g]. 
      Add also [v1] (resp. [v2]) in [g] if [v1] (resp. [v2]) is not in [g]. 
      Just return [g] if this edge is already in [g]. *) 

  val add_edge_e : t -> E.t -> t
    (** [add_edge_e g e] adds the edge [e] in the graph [g].
      Add also [E.src e] (resp. [E.dst e]) in [g] if [E.src e] (resp. [E.dst
      e]) is not in [g]. 
      Just return [g] if [e] is already in [g]. *)

  val remove_edge : t -> V.t -> V.t -> t
    (** [remove_edge g v1 v2] removes the edge going from [v1] to [v2] from the
      graph [g].
      Raise [Invalid_argument] if [v1] or [v2] are not in [g]. 
      Just return [g] if this edge is not in [g]. *)

  val remove_edge_e : t -> E.t -> t
    (** [remove_edge_e g e] removes the edge [e] from the graph [g].
      Raise [Invalid_argument] if [E.src e] or [E.dst e] are not in [g]. 
      Just return [g] if [e] is not in [g]. *)

end

(** Imperative (i.e. mutable) implementation *)

module type I = sig
  include G

  val create : unit -> t
    (** Return an empty graph. *)

  val copy : t -> t
    (** [copy g] returns a copy of [g]. Vertices and edges (and eventually
      marks, see module [Mark]) are duplicated. *)

  val add_vertex : t -> V.t -> unit
    (** [add_vertex g v] adds the vertex [v] from the graph [g].
      Do nothing if [v] is already in [g]. *)

  val remove_vertex : t -> V.t -> unit
    (** [remove g v] removes the vertex [v] from the graph [g] 
      (and all the edges going from [v] in [g]).
      Do nothing if [v] is not in [g]. *)

  val add_edge : t -> V.t -> V.t -> unit
    (** [add_edge g v1 v2] adds an edge from the vertex [v1] to the vertex [v2]
      in the graph [g]. 
      Add also [v1] (resp. [v2]) in [g] if [v1] (resp. [v2]) is not in [g]. 
      Do nothing if this edge is already in [g]. *) 

  val add_edge_e : t -> E.t -> unit
    (** [add_edge_e g e] adds the edge [e] in the graph [g].
      Add also [E.src e] (resp. [E.dst e]) in [g] if [E.src e] (resp. [E.dst
      e]) is not in [g]. 
      Do nothing if [e] is already in [g]. *)

  val remove_edge : t -> V.t -> V.t -> unit
    (** [remove_edge g v1 v2] removes the edge going from [v1] to [v2] from the
      graph [g].
      Raise [Invalid_argument] if [v1] or [v2] are not in [g]. 
      Do nothing if this edge is not in [g]. *)

  val remove_edge_e : t -> E.t -> unit
    (** [remove_edge_e g e] removes the edge [e] from the graph [g].
      Raise [Invalid_argument] if [E.src e] or [E.dst e] are not in [g]. 
      Do nothing if [e] is not in [g]. *)

end

(** Imperative implementation with marks *)

module type IA = sig
  include I
  module Mark : sig
    val clear : t -> unit
      (** [clear g] removes all the marks from all the vertives of [g]. *)
    val get : V.t -> int
    val set : V.t -> int -> unit
  end
end

(** {1 Signature for ordered and hashable types} *)

module type ORDERED_TYPE = sig type t val compare : t -> t -> int end

module type ORDERED_TYPE_DFT = sig include ORDERED_TYPE val default : t end

module type HASHABLE = sig
  type t 
  val hash : t -> int 
  val equal : t -> t -> bool
end

(** Comparable = Ordered + Hashable *)

module type COMPARABLE = sig 
  type t 
  val compare : t -> t -> int 
  val hash : t -> int 
  val equal : t -> t -> bool
end
