/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004, 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "ucs.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"


// Return the filled area of the main blocks only (no recursive)
int Character::area() const throw()
  {
  int a = 0;
  for( int i = 0; i < blocks(); ++i ) a += _block_vector[i].area();
  return a;
  }


Block & Character::block( int i ) throw()
  {
  if( i < 0 || i >= blocks() )
    Ocrad::internal_error( "block, index out of bounds" );
  return _block_vector[i];
  }


void Character::shift_block( const Block & b ) throw()
  {
  if( _blockmap != b.blockmap() )
    Ocrad::internal_error( "can't add block to character of different blockmaps.\n" );
  add_rectangle( b );
  int i = blocks() - 1;
  for( ; i >= 0; --i )
    {
    Block & bi = _block_vector[i];
    if( b.vcenter() > bi.vcenter() ) break;
    if( b.vcenter() == bi.vcenter() && b.hcenter() >= bi.hcenter() ) break;
    }
  _block_vector.insert( _block_vector.begin() + i + 1, b );
  }


void Character::add_guess( int code, int value ) throw()
  {
  _guess.push_back( Guess( code, value ) );
  }


void Character::insert_guess( int i, int code, int value ) throw()
  {
  if( i < 0 || i > guesses() )
    Ocrad::internal_error( "insert_guess, index out of bounds" );
  _guess.insert( _guess.begin() + i, Guess( code, value ) );
  }


void Character::delete_guess( int i ) throw()
  {
  if( i < 0 || i >= guesses() )
    Ocrad::internal_error( "delete_guess, index out of bounds" );
  _guess.erase( _guess.begin() + i );
  }


void Character::only_guess( int code, int value ) throw()
  { _guess.clear(); add_guess( code, value ); }


void Character::swap_guesses( int i, int j ) throw()
  {
  if( i < 0 || i >= guesses() || j < 0 || j >= guesses() )
    Ocrad::internal_error( "swap_guesses, index out of bounds" );
  int code = _guess[i].code;
  _guess[i].code = _guess[j].code; _guess[j].code = code;
  }


const Character::Guess & Character::guess( int i ) const throw()
  {
  if( i < 0 || i >= guesses() )
    Ocrad::internal_error( "guess, index out of bounds" );
  return _guess[i];
  }


bool Character::maybe( int code ) const throw()
  {
  for( int i = 0; i < guesses(); ++i )
    if( code == _guess[i].code ) return true;
  return false;
  }


void Character::join( Character & c ) throw()
  {
  if( _blockmap != c._blockmap )
    Ocrad::internal_error( "can't join characters of different blockmaps.\n" );
  for( int i = 0; i < c.blocks(); ++i ) shift_block( c._block_vector[i] );
  }


void Character::print( const Control & control ) const throw()
  {
  if( guesses() )
    switch( control.format )
      {
      case Control::byte:
        { char ch = UCS::map_to_byte( _guess[0].code );
        if( ch ) std::putc( ch, control.outfile ); }
        break;
      case Control::utf8:
        std::fputs( UCS::ucs_to_utf8( _guess[0].code ), control.outfile );
      }
  else std::putc( '_', control.outfile );
  }


void Character::dprint( const Control & control, const Rectangle & charbox,
                        bool graph, bool recursive ) const throw()
  {
  if( graph || recursive )
    std::fprintf( control.outfile, "%d guesses    ", guesses() );
  for( int i = 0; i < guesses(); ++i )
    {
    switch( control.format )
      {
      case Control::byte:
        { char ch = UCS::map_to_byte( _guess[i].code );
        if( ch ) std::fprintf( control.outfile, "guess '%c', confidence %d    ",
                               ch, _guess[i].value ); }
        break;
      case Control::utf8:
        std::fprintf( control.outfile, "guess '%s', confidence %d    ",
                      UCS::ucs_to_utf8( _guess[i].code ), _guess[i].value );
      }
    if( !graph && !recursive ) break;
    }
  std::fputs( "\n", control.outfile );
  if( graph )
    {
    std::fprintf( control.outfile,
                  "left = %d, top = %d, right = %d, bottom = %d\n",
                  left(), top(), right(), bottom() );
    std::fprintf( control.outfile,
                  "width = %d, height = %d, hcenter = %d, vcenter = %d, black area = %d%%\n\n",
                  width(), height(), hcenter(), vcenter(), ( area() * 100 ) / size() );

    const int minrow = std::min( top(), charbox.top() );
    const int maxrow = std::max( bottom(), charbox.bottom() );
    for( int row = minrow; row <= maxrow; ++row )
      {
      bool istop = ( row == top() );
      bool isvc = ( row == vcenter() );
      bool isbot = ( row == bottom() );
      bool iscbtop = ( row == charbox.top() );
      bool iscbvc = ( row == charbox.vcenter() );
      bool iscbbot = ( row == charbox.bottom() );

      bool ish1top = false, ish1bot = false, ish2top = false, ish2bot = false;
      if( blocks() == 1 && _block_vector[0].blocks() )
        {
        const Block & b = _block_vector[0];
        ish1top = ( row == b.block(0).top() );
        ish1bot = ( row == b.block(0).bottom() );
        if( b.blocks() > 1 )
          {
          ish2top = ( row == b.block(1).top() );
          ish2bot = ( row == b.block(1).bottom() );
          }
        }

      for( int col = left(); col <= right(); ++col )
        {
        char ch = ( isvc && col == hcenter() ) ? '+' : '';
        int id = _blockmap->id( row, col );
        if( id != 0 )
          {
          for( int i = 0; i < blocks(); ++i )
            if( _block_vector[i].compare_id( id, recursive ) )
              {
              if( id > 0 ) ch = (ch == '+') ? 'C' : 'O';
              else ch = (ch == '+') ? '=' : '-';
              break;
              }
          }
        std::fprintf( control.outfile, " %c", ch );
        }
      if( istop ) std::fprintf( control.outfile, "  top(%d)", row );
      if( isvc ) std::fprintf( control.outfile, "  vcenter(%d)", row );
      if( isbot ) std::fprintf( control.outfile, "  bottom(%d)", row );

      if( iscbtop ) std::fprintf( control.outfile, "  box.top(%d)", row );
      if( iscbvc ) std::fprintf( control.outfile, "  box.vcenter(%d)", row );
      if( iscbbot ) std::fprintf( control.outfile, "  box.bottom(%d)", row );

      if( ish1top ) std::fprintf( control.outfile, "  h1.top(%d)", row );
      if( ish1bot ) std::fprintf( control.outfile, "  h1.bottom(%d)", row );
      if( ish2top ) std::fprintf( control.outfile, "  h2.top(%d)", row );
      if( ish2bot ) std::fprintf( control.outfile, "  h2.bottom(%d)", row );

      std::fputs( "\n", control.outfile );
      }
    std::fputs( "\n\n", control.outfile );
    }
  }


void Character::xprint( const Control & control ) const throw()
  {
  std::fprintf( control.exportfile, "%3d %3d %2d %2d; %d",
                left(), top(), width(), height(), guesses() );

  for( int i = 0; i < guesses(); ++i )
    switch( control.format )
      {
      case Control::byte:
        { char ch = UCS::map_to_byte( _guess[i].code );
        if( !ch ) ch = '_';
        std::fprintf( control.exportfile, ", '%c'%d", ch, _guess[i].value ); }
        break;
      case Control::utf8:
        std::fprintf( control.exportfile, ", '%s'%d",
                      UCS::ucs_to_utf8( _guess[i].code ), _guess[i].value );
      }
  std::fputs( "\n", control.exportfile );
  }
