// -*- C++ -*-

// Copyright 2006 Deutsches Forschungszentrum fuer Kuenstliche Intelligenz
// or its licensors, as applicable.
//
// You may not use this file except under the terms of the accompanying license.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you
// may not use this file except in compliance with the License. You may
// obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Project: bpnet -- bpnet network classifier
// File: confusion-matrix.cc
// Purpose: class for a confusion matrix
// Responsible: Hagen Kaprykowsky (kapry@iupr.net)
// Reviewer: Yves Rangoni (rangoni@iupr.dfki.de)
// Primary Repository:
// Web Sites: www.iupr.org, www.dfki.de

#include "confusion-matrix.h"

using namespace ocropus;
using namespace colib;

namespace iupr_bpnet {

    // convert an ascii integer code in a utf8 string with some
    // white spaces before
    void ascii_to_nustring(int code, char** str,int size=1) {
        CHECK_CONDITION(size>0&&code>0);
        nuchar c(code);
        nustring s(1);
        s[0] = c;
        if (*str)
            free(*str);
        *str = s.mallocUtf8Encode();
        if (strlen(*str) > 0) {             // a trick to have a nice alignment
            char* ns = (char*)malloc(32);   // non-ascii characters
            *ns = 0;
            for(int i=0;i<size-1;i++)
                strcat(ns," ");
            strcat(ns,*str);
            free(*str);
            *str = ns;
        }
    }

    // class ConfusionMatrix
    // Keep track of a classifier's errors.

    // constructor
    ConfusionMatrix::ConfusionMatrix(unsigned int nRows, unsigned int nCols) {
        CHECK_CONDITION(nRows>0&&nCols>0);
        confusion.resize(nRows,nCols);
        fill(confusion,0);
    };

    // constructor with intarray
    ConfusionMatrix::ConfusionMatrix(colib::intarray &matrix) {
        CHECK_CONDITION(matrix.rank()==2&&matrix.dim(0)>0&&matrix.dim(0)>0);
        copy(confusion,matrix);
    }

    // destructor
    ConfusionMatrix::~ConfusionMatrix() {
        confusion.dealloc();
    };

    // set matrix entries to zero
    void ConfusionMatrix::clear() {
        fill(confusion,0);
    };

    // increment matrix entry by one
    void ConfusionMatrix::increment(int actual,int predicted) {
        if(actual>confusion.dim(0)) return;
        ASSERT(predicted<confusion.dim(1));
        confusion(actual,predicted)++;
    };

    // erase the current matrix by another intarray
    void ConfusionMatrix::set(intarray &matrix) {
        CHECK_CONDITION(matrix.rank()==2&&matrix.dim(0)>0&&matrix.dim(0)>0);
        copy(confusion,matrix);
    }

    // erase the current matrix by another one
    void ConfusionMatrix::set(ConfusionMatrix &matrix) {
        copy(confusion,matrix.confusion);
    }

    // have a copy of the matrix
    void ConfusionMatrix::get(intarray &matrix) {
        copy(matrix,confusion);
    }

    // create a sorted 'list' of confusions
    void ConfusionMatrix::confusion_to_list(intarray &out) {
        intarray x,y,v;
        int l=confusion.dim(0)*confusion.dim(1);
        x.resize(l);
        y.resize(l);
        v.resize(l);
        int k=0;
        for(int i=0;i<confusion.dim(0);i++) {
            for(int j=0;j<confusion.dim(1);j++) {
                x(k)=i;
                y(k)=j;
                v(k)=confusion(i,j);
                k++;
            }
        }
        intarray perm;
        quicksort(perm,v);
        permute(x,perm);
        permute(y,perm);
        permute(v,perm);
        out.resize(l,3);
        for(int i=0;i<l;i++) {      // reverse order
            out(i,0) = x(l-i-1);
            out(i,1) = y(l-i-1);
            out(i,2) = v(l-i-1);
        }
    }

    // dump confusion matrix in a list-like shape to a stream
    void ConfusionMatrix::print(FILE *stream) {
        intarray conf;
        confusion_to_list(conf);
        fprintf(stream, "cnf  -t- -o-\n");
        for(int i=0;i<conf.dim(0);i++) {
            fprintf(stream, "cnf %4d %4d %6d\n",conf(i,0),conf(i,1),conf(i,2));
        }
    }

    // dump reduced confusion matrix in a list-like shape to a stream
    // only real confusions are displayed (confusion(i,j)>0) && (i!=j)
    void ConfusionMatrix::printReduced(FILE *stream) {
        intarray conf;
        confusion_to_list(conf);
        fprintf(stream, "cnf  -t-  -o-\n");
        for(int i=0;i<conf.dim(0);i++) {
            if ((conf(i,2)>0) && (conf(i,0)!=conf(i,1))) {
                fprintf(stream, "cnf %4d %4d %6d\n",conf(i,0),conf(i,1),conf(i,2));
            }
        }
    }

    // dump confusion matrix to a stream
    // An aligned table is created, size is the field width.
    // If norm is set to true, the values are normalized, and size is
    // the number of digits to be printed after the decimal point
    void ConfusionMatrix::printMatrix(FILE* stream,int size,bool norm) {
        CHECK_CONDITION(size>0);
        char format_v[32];
        char format_b[32];
        float _isum = 1.;
        if(norm==true) {
            _isum = 1./sum(confusion);
            sprintf(format_v,"%%%d.%df",size+3,size);
            size += 3;
        } else {
            sprintf(format_v,"%%%dd",size);
        }
        sprintf(format_b,"%%%dd",size);
        for(int k=0;k<=size;k++) { fprintf(stream," "); }
        for(int j=0;j<confusion.dim(1);j++) {
            fprintf(stream,format_b,j);
        }
        fprintf(stream,"\n");
        for(int k=0;k<=size;k++) { fprintf(stream," "); }
        for(int j=0;j<confusion.dim(1)*size;j++) { fprintf(stream,"-"); }
        fprintf(stream,"\n");
        for(int i=0;i<confusion.dim(0);i++) {
            fprintf(stream,format_b,i);
            fprintf(stream,"|");
            for(int j=0;j<confusion.dim(1);j++) {
                if (norm==true) {
                    fprintf(stream,format_v,confusion(i,j)*_isum);
                } else {
                    fprintf(stream,format_v,confusion(i,j));
                }
            }
            fprintf(stream,"\n");
        }
    }

    // dump confusion matrix with a ClassMap to a stream
    // An aligned table is created, size is the field width.
    // If norm is set to true, the values are normalized, and size is
    // the number of digits to be printed after the decimal point
    void ConfusionMatrix::printMatrix(FILE* stream,ClassMap &map,int size,bool norm) {
        CHECK_CONDITION(size>0);
        char formatn[32];
        char formats[32];
        float _isum = 1.;
        if(norm==true) {
            _isum = 1./sum(confusion);
            sprintf(formatn,"%%%d.%df",size+3,size);
            size += 3;
        } else {
            sprintf(formatn,"%%%dd",size);
        }
        sprintf(formats,"%%%ds",size);
        for(int k=0;k<=size;k++) fprintf(stream," ");
        char* str = 0;
        for(int j=0;j<confusion.dim(1);j++) {
            ascii_to_nustring(map.get_ascii(j),&str,size);
            fprintf(stream,formats,str);
        }
        fprintf(stream,"\n");
        for(int k=0;k<=size;k++) fprintf(stream," ");
        for(int j=0;j<confusion.dim(1)*size;j++) fprintf(stream,"-");

        fprintf(stream,"\n");
        for(int i=0;i<confusion.dim(0);i++) {
            ascii_to_nustring(map.get_ascii(i),&str,size);
            fprintf(stream,formats,str);
            fprintf(stream,"|");
            for(int j=0;j<confusion.dim(1);j++) {
                if (norm==true)
                    fprintf(stream,formatn,confusion(i,j)*_isum);
                else
                    fprintf(stream,formatn,confusion(i,j));
            }
            fprintf(stream,"\n");
        }
        if (str)
            free(str);
    }

    // dump confusion matrix with a Classmap in a list-like shape to stream.
    void ConfusionMatrix::printReduced(FILE* stream,ClassMap &map) {
        intarray conf;
        confusion_to_list(conf);
        fprintf(stream,"         internal           ascii             utf8\n");
        fprintf(stream,"       -t-      -o-      -t-        -o-   -t-     -o-\n");
        for(int i=0;i<conf.dim(0);i++) {
            if (conf(i,2)>0 && (conf(i,0)==conf(i,1))) {
                print_one_confusion(stream,map,conf(i,0),conf(i,1),conf(i,2),"good");
            }
        }
        fprintf(stream, "--------------\n");
        for(int i=0;i<conf.dim(0);i++) {
            if ((conf(i,2)>0) && (conf(i,0)!=conf(i,1))) {
                print_one_confusion(stream,map,conf(i,0),conf(i,1),conf(i,2),"conf");
            }
        }
    }

    // dump one line of confusion to a stream
    // should be note use from the outside, but through printReduced
    void ConfusionMatrix::print_one_confusion(FILE* stream,ClassMap &map,int i,int j,int v,
                                const char* prefix) {
        char* str1 = 0;
        char* str2 = 0;
        ascii_to_nustring(map.get_ascii(i),&str1,3);
        ascii_to_nustring(map.get_ascii(j),&str2,3);
        fprintf(stream, "%s (%4d <-> %4d) (%6d <-> %6d)"
                        " (%3s <-> %3s):\t%6d\n",
                        prefix,i,j,
                        map.get_ascii(i),map.get_ascii(j),
                        (map.get_ascii(i)==172)?"*G*":str1,
                        (map.get_ascii(j)==172)?"*G*":str2,
                        v);
        if (str1)
            free(str1);
        if (str2)
            free(str2);
    }


    ConfusionMatrix *make_ConfusionMatrix(int ncls) {
        return new ConfusionMatrix(ncls,ncls);
    }

    ConfusionMatrix *make_ConfusionMatrix(colib::intarray &c) {
        return new ConfusionMatrix(c);
    }
}

