/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2; coding: utf-8 -*- 
 *
 * Copyright (C) 2007, 2008, 2009 John P. Swensen
 *
 * This file is as a part of OctaveDE.
 *
 * Octave is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Octave; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * */
#include "HistoryTreeView.h"

#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#include "octave/config.h"
#include "octave/octave.h"
#include "octave/str-vec.h"
#include "octave/cmd-hist.h"


#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/liststore.h>
#include <gtkmm/menu.h>

#include <iostream>
#include <sstream>
using std::ostringstream;
using std::cout;
using std::endl;

HistoryTreeView::HistoryTreeView() 
{
  mHistory = new HistoryColumns();

  mHistoryTreeModel = Gtk::ListStore::create( *mHistory );
  this->set_model( mHistoryTreeModel );
  this->append_column("Command", mHistory->m_col_command);
  mHistoryTreeModel->clear();
  mHistoryTreeSelection = this->get_selection();
  mHistoryTreeSelection->set_mode(Gtk::SELECTION_MULTIPLE);

  mPopupMenu = new Gtk::Menu();

  Gtk::Menu::MenuList& menulist = mPopupMenu->items();

  menulist.push_back( Gtk::Menu_Helpers::MenuElem("_Copy",
                                                  sigc::mem_fun(*this, &HistoryTreeView::on_popup_menu_copy) ) );
  menulist.push_back( Gtk::Menu_Helpers::MenuElem("_Execute commands",
                                                  sigc::mem_fun(*this, &HistoryTreeView::on_popup_menu_execute) ) );
  menulist.push_back( Gtk::Menu_Helpers::MenuElem("C_lear",
                                                  sigc::mem_fun(*this, &HistoryTreeView::on_popup_menu_clear) ) );
}

HistoryTreeView::~HistoryTreeView() 
{
  delete mHistory;
  delete mPopupMenu;
}

void HistoryTreeView::on_popup_menu_copy()
{
  //std::cout << "A popup menu item was selected." << std::endl;

  // copy the selected items to the system clipboard
  Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();
  refClipboard->set_text( getSelectionText() );
  onCommandCopy.emit();
}

void HistoryTreeView::on_popup_menu_execute()
{
  //std::cout << "The execute commands popup menu item was selected." << std::endl;

  // Execute all selected commands
  executeSelection();
}

void HistoryTreeView::on_popup_menu_clear()
{
  //std::cout << "A popup menu item was selected." << std::endl;

  // clear the treeview
  mHistoryTreeModel->clear();
}

void HistoryTreeView::executeSelection( void )
{
  Glib::ListHandle<TreeModel::Path, TreePath_Traits> paths =  mHistoryTreeSelection->get_selected_rows();
  Glib::ListHandle<TreeModel::Path, TreePath_Traits>::iterator it = paths.begin();

  //cout << "PathSize: " << paths.size() << endl;

  // Build up the vector of commands to execute and emit the signal to the main UI	
  vector<string> commands;
  for ( it = paths.begin() ; it != paths.end() ; it++ )
  {
    Gtk::TreeModel::Row row = *(mHistoryTreeModel->get_iter(*it));
    Glib::ustring command = row[mHistory->m_col_command] + '\n';
    commands.push_back( command );
  } 
  onCommandExecute.emit( commands );
}

string HistoryTreeView::getSelectionText( void )
{
  ostringstream selectionText;  

  Glib::ListHandle<TreeModel::Path, TreePath_Traits> paths =  mHistoryTreeSelection->get_selected_rows();
  Glib::ListHandle<TreeModel::Path, TreePath_Traits>::iterator it = paths.begin();

  // Build up the vector of commands to execute and emit the signal to the main UI
  vector<string> commands;
  for ( it = paths.begin() ; it != paths.end() ; it++ )
  {
    Gtk::TreeModel::Row row = *(mHistoryTreeModel->get_iter(*it));
    Glib::ustring command = row[mHistory->m_col_command] + '\n';
    selectionText << command;
  }

  return selectionText.str();
}

bool HistoryTreeView::on_button_press_event( GdkEventButton* evt )
{
  //cout << "press" << endl;
  static long unsigned lastclick = 0;

  // If a right click, pop-up a context menu
  if ( evt->type == GDK_BUTTON_PRESS && evt->button ==3 )
  {
    mPopupMenu->popup(evt->button, evt->time);
    return  false;
  }
  // If it was a double click
  else if ( evt->type == GDK_2BUTTON_PRESS && evt->button ==1 )
  {
    //std::cout << "Dble click" << std::endl;

    executeSelection();

    return TreeView::on_button_press_event(evt);
  }
  // Handle all other clicks normally
  else
  {
    return TreeView::on_button_press_event(evt);
  }
}

bool HistoryTreeView::on_key_press_event( GdkEventKey* evt )
{
  if ( (evt->type == GDK_KEY_PRESS) && 
       (evt->state & GDK_CONTROL_MASK) && 
       (evt->keyval == GDK_c)  )
  {
    // copy the selected items to the system clipboard
    Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();
    refClipboard->set_text( getSelectionText() );
    onCommandCopy.emit();

    cout << getSelectionText() << endl;
  }

  return TreeView::on_key_press_event(evt);
}

void HistoryTreeView::clearRows() 
{ 
  gdk_threads_enter();
  mHistoryTreeModel->clear();
  gdk_threads_leave();
  command_history::set_size(0);
}

void HistoryTreeView::addRow( string command )
{
  Gtk::TreeModel::Row row = *(mHistoryTreeModel->append());
  row[mHistory->m_col_command] = Glib::locale_to_utf8(command);
} 

void HistoryTreeView::addRows( vector<string> commands )
{
  gdk_threads_enter();
  for ( size_t i = 0 ; i < commands.size() ; i++ )
  {
    Gtk::TreeModel::Row row = *(mHistoryTreeModel->append());
    row[mHistory->m_col_command] = Glib::locale_to_utf8(commands[i]);
  }
  gdk_threads_leave();
}

void HistoryTreeView::addRows( string_vector commands )
{
  gdk_threads_enter();
  for ( size_t i = 0 ; i < commands.length() ; i++ )
  {
    Gtk::TreeModel::Row row = *(mHistoryTreeModel->append());
    row[mHistory->m_col_command] = Glib::locale_to_utf8(commands[i]);
  }
  gdk_threads_leave();
}
