/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2; coding: utf-8 -*- 
 *
 * Copyright (C) 2007, 2008, 2009 John P. Swensen
 *
 * This file is as a part of OctaveDE.
 *
 * Octave is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Octave; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * */
#include "OctaveUI.h"

#include "octave/pt-pr-code.h"


#include "errno.h"

#if defined(HAVE_PTY_H)
  #include <pty.h>
#elif defined(HAVE_UTIL_H) /* OpenBSD */
  #include <util.h>
#elif defined(HAVE_LIBUTIL_H) /* FreeBSD */
  #include <libutil.h>
#elif defined(HAVE_LIBUTIL) /* BSDI has libutil, but no libutil.h */
  #error "We can't handle this.....sorry!"
#endif

#include <readline/readline.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <string>
using std::ostringstream;
using std::cout;
using std::endl;
using std::vector;
using std::string;

using namespace Gtk;

extern OctaveUI* octaveUI;

#ifdef __WIN32__
#define usleep(x) Sleep((x)/1000)
#include <io.h>
#include <vte/ptywin.h>
#endif

#ifdef __APPLE__
#include <syslog.h>
#include <mach-o/dyld.h>
#endif

/////////////////
#ifdef __WIN32__
DWORD WINAPI octave_main_wrapper(LPVOID dummy)
#else
void* octave_main_wrapper(void *dummy)
#endif
{
  // TODO: pass in the real argc, argv
#ifdef __WIN32__
  int argc = 3;
  char* argv[] = {"octave", "-i", "--line-editing"};
#else
  int argc = 1;
  char* argv[] = {"/sw/bin/octave"};
#endif

#ifdef __APPLE__ // this should only be done if referencing the Octave.app in Applications
  //setenv("OCTAVE_HOME", "/Applications/Octave.app/Contents/Resources",1);
#endif

  octave_main(argc,argv,0);
  
  //octave_stdout << "Out of octave main loop" << std::endl;
  
  //do_octave_atexit();
  //clean_up_and_exit(0);

  return 0;
}


#ifdef __WIN32__
DWORD WINAPI do_octave_monitor(LPVOID param)
{
  return (DWORD)OctaveUI::octave_monitor(param);
}
#endif

bool ui_can_interrupt = true;
/*static*/ void* OctaveUI::octave_monitor(void *octaveUI)
{
  OctaveUI* oui = (OctaveUI*)octaveUI;

  while ( oui->isRunning )
  {
    // collect all the variable info
    vector<variable_info_t> variables = oct_octave_server.get_variable_info_list();
    if ( variables.size() > 0 )
    {
      oui->mVariablesTreeView.clearRows();
      oui->mVariablesTreeView.addVariables(variables);
    }

    // Check whether any requested variables have been returned
    vector<requested_variable_t> reqVars = oct_octave_server.get_requested_variables();
    for( vector<requested_variable_t>::iterator it = reqVars.begin() ;
	 it != reqVars.end() ;
	 it++ )
    {
      std::cout << "Requested variable returned: " << (*it).name << std::endl;
    }

    // collect history info
    string_vector historyList = oct_octave_server.get_history_list();
    if( historyList.length() > 0 )
    {
      oui->mHistoryTreeView.addRows(historyList);
    }


    // TODO: Don't remove all marks as this is inefficient. Instead make it so MEditor accepts a complete list of breakpoints
    // and is smart about adding and removing only those necessary.
    // Also, we may need to do the GTK locking for this to not crash, since I saw a crash when updating breakpoints.

    // Remove all markers first
    //MEditor::GetInstance()->remove_all_breakpoints ();

    // Put a marker in each buffer at the proper location
    std::vector<bp_info_t> bps = oct_octave_server.get_breakpoint_list();
    MEditor::GetInstance()->process_breakpoint_list (bps);

    // Find out if a breakpoint is hit
    static bool line_number = -1;
    if (oct_octave_server.is_breakpoint_reached())
    {
      
      std::vector<bp_info_t> hit_breakpoint = oct_octave_server.get_breakpoint_reached();
      if (hit_breakpoint.size() > 0 && (hit_breakpoint[0].line_number != line_number))
      {
	MEditor::GetInstance()->remove_hit_breakpoint_marker ();
	MEditor::GetInstance()->add_breakpoint_marker(hit_breakpoint[0], BP_MARKER_TYPE_HIT);
	line_number = hit_breakpoint[0].line_number;
      }
      //octave_stdout << "A breakpoint has been reached: " << hit_breakpoint[0].filename << " : " << hit_breakpoint[0].line_number << std::endl;
    }
    else if (line_number>0)
    {
      MEditor::GetInstance()->remove_hit_breakpoint_marker ();
      line_number = -1;
    }

    if( global_command == 0 )
      ui_can_interrupt = true;

    usleep(100000);
    //    cout << "The ID of the sockets window is: " << oui->mEmacsSocket.get_id() << endl;
  }

  oui->threadEnded = 1;
  //octave_stdout << "Thread ended" << endl;
  return 0;
}

OctaveUI::OctaveUI() : Gtk::Window(), initializeComplete(false), isRunning(true), threadEnded(0),
		       variablesDock(dockPane, "Variables","Variables","VariablesIcon", DockItem::DOCKED_STATE),
		       historyDock(dockPane, "History","History","HistoryIcon", DockItem::DOCKED_STATE),
		       fileChooserDock(dockPane, "Files","Files","FilesIcon", DockItem::DOCKED_STATE),
		       octaveDock(dockPane, "Octave","Octave","OctaveIcon", DockItem::DOCKED_STATE),
		       meditor(NULL)
{
  this->set_size_request(800,600);

  this->signal_delete_event().connect( sigc::mem_fun( *this, &OctaveUI::onCloseWindow ) );

  if (meditor==NULL)
    meditor = MEditor::GetInstance(dockPane,false);

  // Create the main octave terminal window
  createOctaveVteDockItem();

  // Create the history window
  createHistoryDockItem();

  // Create the file chooser window
  createFileChooserDockItem();

  // Create the variables window
  createVariablesDockItem();
  
  GdlDockPlacement placement = GDL_DOCK_NONE;
  GdlDockObject* parent = gdl_dock_object_get_parent_object (GDL_DOCK_OBJECT(variablesDock.getWidget().gobj()));
  GdlDockObject* child = GDL_DOCK_OBJECT(variablesDock.getWidget().gobj());
  gdl_dock_object_child_placement (parent, child, &placement);
  prevChildImmediateParentMap[child] = parent;
  prevChildPlacementMap[child] = placement;
  
  // Output for debugging placement
  //cout << "Variables placement: " << (int)placement << endl;

  placement = GDL_DOCK_NONE;
  parent = gdl_dock_object_get_parent_object (GDL_DOCK_OBJECT(historyDock.getWidget().gobj()));
  child = GDL_DOCK_OBJECT(historyDock.getWidget().gobj());
  gdl_dock_object_child_placement (parent, child, &placement);
  prevChildImmediateParentMap[child] = parent;
  prevChildPlacementMap[child] = placement;

  // Output for debugging placement
  //cout << "History placement: " << (int)placement << endl;
 
  placement = GDL_DOCK_NONE;
  parent = gdl_dock_object_get_parent_object (GDL_DOCK_OBJECT(fileChooserDock.getWidget().gobj()));
  child = GDL_DOCK_OBJECT(fileChooserDock.getWidget().gobj());
  gdl_dock_object_child_placement (parent, child, &placement);
  prevChildImmediateParentMap[child] = parent;
  prevChildPlacementMap[child] = placement;
  
  // Output for debugging placement
  //cout << "FileChooser placement: " << (int)placement << endl;

  placement = GDL_DOCK_NONE;
  parent = gdl_dock_object_get_parent_object (GDL_DOCK_OBJECT(octaveDock.getWidget().gobj()));
  child = GDL_DOCK_OBJECT(octaveDock.getWidget().gobj());
  gdl_dock_object_child_placement (parent, child, &placement);
  prevChildImmediateParentMap[child] = parent;
  prevChildPlacementMap[child] = placement;
  
  // Output for debugging placement
  //cout << "Octave placement: " << (int)placement << endl;


  // Create the menubar and toolbar
  createMenubar();
  createToolbar();
  
  // Fit all the UI pieces together
  vbox.pack_start(menuBar, Gtk::PACK_SHRINK);
  vbox.pack_start(m_toolbar, Gtk::PACK_SHRINK);
  vbox.pack_start( hbox, true, true, 0);
  hbox.pack_start(dockPane.getWidget(), true, true, 0);  

  this->add( vbox );
  this->show_all_children();


  std::string currdir = mFileChooserWidget.get_current_folder();

  usleep(1000000);
  command_editor::add_event_hook (server_rl_event_hook_function);
  
#ifndef __WIN32__
  int rc2 = pthread_create (&octave_monitor_thread, 0, OctaveUI::octave_monitor, this);
#else
  octave_monitor_thread = CreateThread(0, 0, do_octave_monitor, this, 0, NULL);
#endif


  dockPane.signal_layout_changed ().connect( sigc::mem_fun( *this, &OctaveUI::onLayoutChanged) );
  initializeComplete = true;

#ifdef HAVE_GLRENDERER
  ___gtkhandles_init();
#endif // HAVE_GLRENDERER

}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// This function is very messy, but due to the strangeness (and incompleteness) of the libgdl dock widget, I know of
// no other way to accomplish this behavior of only allowing docking between the editor dock and the main dock.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void OctaveUI::onLayoutChanged(void)
{
  static std::map<GdlDockObject*,GdlDock*> prevChildParentMap;

  MEditor* editor = MEditor::GetInstance(dockPane,false); 
  
  // Used for debugging the layout changes
  // cout <<"OctaveUI LayoutChanged" << endl;

  // Get a handle to the two windows to which we will allow docking: the main window and the editor window
  GdlDock* mainDock = GDL_DOCK(dockPane.getGdlWidget ());
  GdlDock* editorDock = GDL_DOCK(editor->getDock ());

  // Get a list of all DockItems from the main dock
  GList* dockItemList = gdl_dock_get_named_items (mainDock);
  
  int mainDockChildrenCount = 0;
  int editorDockChildrenCount = 0;
  GList *l;
  for (l = dockItemList; l; l = l->next) 
  {
    GdlDockObject *dockItem = GDL_DOCK_OBJECT (l->data);
    GdlDock* parentDock = gdl_dock_object_get_toplevel (dockItem);
    if ( GDL_DOCK(parentDock) == mainDock )
    {
      mainDockChildrenCount++;
    }
    else if ( GDL_DOCK (parentDock) == editorDock )
    {
      editorDockChildrenCount++;
    }
    else
    {
      // Don't allow floating windows, so dock this to something
      // We need to make sure we don't allow the main dockitems (e.g. history, variables,
      // filebrowser, octaveDock) to be moved to the editorDock.  Only allow editor dockitems
      // to transfer between docks.
      // I may need to store the parent and location, so we can re-attach at the correct location.
      if( GDL_DOCK_ITEM(dockItem) == GDL_DOCK_ITEM(variablesDock.gobj()) ||
	  GDL_DOCK_ITEM(dockItem) == GDL_DOCK_ITEM(historyDock.gobj()) ||
	  GDL_DOCK_ITEM(dockItem) == GDL_DOCK_ITEM(fileChooserDock.gobj()) ||
	  GDL_DOCK_ITEM(dockItem) == GDL_DOCK_ITEM(octaveDock.gobj()) )
      {
	cout << "TODO: We shouldn't allow the core dockitems to be moved out of the mainDock" << endl;
	GdlDockObject* parent = prevChildImmediateParentMap[dockItem];
	GdlDockPlacement placement = prevChildPlacementMap[dockItem];

	gdl_dock_item_dock_to ( GDL_DOCK_ITEM(dockItem), GDL_DOCK_ITEM(octaveDock.gobj()), placement, 0);
	  
	//gdl_dock_item_dock_to ( GDL_DOCK_ITEM(dockItem), GDL_DOCK_ITEM(parent), placement, 0);

      }
      else
      {
	GdlDock* lastParent = prevChildParentMap[dockItem];
	if (lastParent == mainDock)
	{
	  // Dock to the editor
	  gdl_dock_item_dock_to ( GDL_DOCK_ITEM(dockItem), GDL_DOCK_ITEM(editorDock), GDL_DOCK_CENTER, 0);
	  parentDock = editorDock;
	  editor->show_all();
	  editorDockChildrenCount++;
	}
	else
	{
	  // Dock to the main window
	  // If there is already an editor docked to the main dock, we should place this one in the center
	  
	  // Otherwise, place it just above the editor window
	  gdl_dock_item_dock_to ( GDL_DOCK_ITEM(dockItem), GDL_DOCK_ITEM(octaveDock.gobj()), GDL_DOCK_TOP, 0);
	  gdl_dock_placeholder_attach (editorPlaceholder, GDL_DOCK_OBJECT(dockItem));
	  parentDock = mainDock;
	  mainDockChildrenCount++;
	}
      }
      
      GdlDockPlacement placement = GDL_DOCK_NONE;
      GdlDockObject* parent = gdl_dock_object_get_parent_object (dockItem);
      gdl_dock_object_child_placement (parent, dockItem, &placement);
      
      // Debug output for placement debugging
      //cout << "Placement: " << (int)placement << endl;

      prevChildImmediateParentMap[dockItem] = parent;
      prevChildPlacementMap[dockItem] = placement;
      prevChildParentMap[dockItem] = parentDock;
    }
  }

  /*if (editorDockChildrenCount == 0)
  {
    cout << "Hiding the MEditor widget - it has no editor panes" << endl;
    editor->hide();
  }
  */

}

OctaveUI::~OctaveUI()
{
  variablesDock.hide();
  historyDock.hide();
  fileChooserDock.hide();
  dockPane.hide();
  hbox.hide();
  vbox.hide();
  this->hide();
}

void OctaveUI::onFileActivated( void )
{
  string filename = mFileChooserWidget.get_filename();
  //std::cout << "File selected: " << filename << std::endl;
  meditor->openFile( filename );
  meditor->show_all();
}

void OctaveUI::onCommandExecute( vector<string> commands )
{
  for ( int i = 0 ; i < commands.size(); i++ )
  {
    vte_terminal_feed_child( (VteTerminal*)vteTerminal, commands[i].c_str(), commands[i].size() );  
  }
}

void OctaveUI::onCommandCopy( void )
{
  // Send the current clipboard buffer to the VTE widget's clipboard
  //vte_terminal_paste_clipboard( (VteTerminal*)vteTerminal );

  cout << "Paste to clipboard" << endl;
}



void OctaveUI::onNew(void)
{
  if (meditor == NULL)
  {
    meditor = MEditor::GetInstance(dockPane,false);
  }

  // Check if the editor is already open, and open an empty file accordingly
  // If there is an editor window attached to the main window, and the editor window is not open, then
  // then we should attach the new editor window to the center of the old one
  meditor->onNew();

}

void OctaveUI::onOpen(void)
{
  if (meditor == NULL)
    {
      meditor = MEditor::GetInstance(dockPane,false);
    }

  // Show the open file dialog
  if( meditor->openFileDialog() == 0 )
  {
    // Check if the editor is already open, and open the selected file(s) accordingly
    meditor->show_all();
  }

}

void OctaveUI::onCut(void)
{
  std::cout << "Cut: hard to do with readline" << endl;
}

void OctaveUI::onCopy(void)
{
  // Feed the VTE data to the clipboard (with X windows, the selection process should have already done this)
  vte_terminal_copy_clipboard( (VteTerminal*)vteTerminal );
}

void OctaveUI::onPaste(void)
{
  // Feed the clipboard data to the VTE
  vte_terminal_paste_clipboard( (VteTerminal*)vteTerminal );
    
  // Set the focus to the command window
  vtePanel->grab_focus();
}
  

void OctaveUI::onPreferences(void)
{
  Gtk::FontSelection* fontSelection = NULL;  

  // Create new dialog box
  // TODO: make this new dialog box its own class

  // Currently, only the font is selectable
  Glib::ustring fontName;
  FontSelectionDialog fsd("VTE Font");
  int result = fsd.run();

  switch (result)
  {
    case GTK_RESPONSE_OK:
      fontSelection = fsd.get_font_selection();
      fontName = fontSelection->get_font_name();
      vte_terminal_set_font_from_string( (VteTerminal*)vteTerminal, fontName.c_str() );
      
      break;

    default:
      // do nothing
      break;
  }
}

void OctaveUI::onQuit(void)
{
  //cout << "OnQuit" << endl;
  command_editor::remove_event_hook (server_rl_event_hook_function);
    

  this->isRunning = false;
  while (!this->threadEnded)
    ::usleep(200000);

  /*
  kill( getpid(), SIGINT );
  char* cmd = "exit\n";
  vte_terminal_feed_child( (VteTerminal*)vteTerminal, cmd, strlen(cmd) );  
  */

  //octave_stdout << "After waiting for the thread" << std::endl;

  /*
  octave_run_main_loop = false;

  rl_point = 0;
  //rl_line_buffer[0] = '\0';
  //rl_end = 0;
  rl_done = 1;
  */

  
  int rc = 0;
#ifndef __WIN32__
  rc = kill( getpid(), SIGKILL );
#else
  Gtk::Main::quit();
#endif

  /*
  static ostringstream oss;
  if (global_command)
  {
    tree_print_code tpc(oss);
    tpc.visit_statement_list(*global_command);

    //printf(oss.str().c_str());
  }
  else
  {
    //octave_stdout << "No global command" << std::endl;
    //octave_stdout << oss.str() << std::endl;

    //oss.str("");
  }
  */

  return;
}

bool OctaveUI::onCloseWindow( GdkEventAny* evt )
{
  //cout << "OnCloseWindow" << endl;
  command_editor::remove_event_hook (server_rl_event_hook_function);

  this->isRunning = false;
  while (!this->threadEnded)
    ::usleep(200000);

  /*
  kill( getpid(), SIGINT );
  char* cmd = "exit\n";
  vte_terminal_feed_child( (VteTerminal*)vteTerminal, cmd, strlen(cmd) );  
  */

  //octave_stdout << "After waiting for the thread" << std::endl;

  /*
  octave_run_main_loop = false;
  ::usleep(20000);

  rl_catch_signals = 0;
  rl_point = 0;
  //rl_line_buffer[0] = '\0';
  //rl_end = 0;
  rl_done = 1;
  //rl_reset_after_signal();
  rl_clear_signals();
  */


#ifndef __WIN32__
  kill( getpid(), SIGKILL );
#else
  Gtk::Main::quit();
#endif // __WIN32__

  return true;
}

// NOT USED ANYMORE...menu key accelerator events are now used to handle this
// THIS IS WORSE ANYWAY, SINCE IT REQUIRED INTERCEPTION OF ALL KEY EVENTS
bool OctaveUI::onKeyPress( GdkEventKey* evt )
{
  //std::cout << "key pressed: " << (int)evt->state << ":" << (int)evt->keyval << endl;
  if ( evt->state & Gdk::CONTROL_MASK && evt->keyval == 'w' )       // CUT
  {
    
  }
  else if ( evt->state & Gdk::CONTROL_MASK && evt->keyval == 'y' )  // PASTE
  {
    // Feed the clipboard data to the VTE
    vte_terminal_paste_clipboard( (VteTerminal*)vteTerminal );
    
    // Set the focus to the command window
    vtePanel->grab_focus();

  }
  else if ( evt->state & Gdk::MOD1_MASK && evt->keyval == 'w' )     // COPY
  {

  }


  if( evt->state != 0 )
    return false; 
  else
    return true;
}

/*static*/ void
OctaveUI::terminal_widget_commit (GtkWidget *widget,
                                  gchar          *data,
                                  guint           length,
                                  void* parm)
{
  /*cout << "Len: " << length << "     Data: ";
  for( int i = 0 ; i < length ; i++ )
    cout << (int)data[i] << " ";
  cout << endl;*/

  // Detect a CTRL-C
  if ( length == 1 && data[0] == 0x03 )
  {
    // Send a CTRL-C signal to octave
    // TODO - I think 'global_command' is a bad, non-threadsafe way to determine
    //  if we are currently executing a command
    //if ( global_command != 0 )
    //if( global_command != 0 && can_interrupt == true && octave_interrupt_state < 1)
    //  ::raise(SIGINT);
    if( global_command != 0 && ui_can_interrupt )
    {
      ::raise(SIGINT);
      ui_can_interrupt = false;
    }
  }

  // Place code here to detect some other character combination(s)
  // Cut:   CTRL+W
  // Copy:  ALT+W
  // Paste: CTRL+Y


}

void OctaveUI::onVteSizeChanged( Gtk::Allocation& alloc )
{
/*	// Get the VTE size in pixels
	int width = alloc.get_width();
	int height = alloc.get_height();
	
	// Determine the character width
	long charWidth = vte_terminal_get_char_width( (VteTerminal*)vteTerminal );
	long charHeight = vte_terminal_get_char_height( (VteTerminal*)vteTerminal );
	
	// Set the readline size based on pixels/charwidth
	//cout << "Resize of VTE occurred (C,R): (" << width/charWidth+1 << ", " << height/charHeight+1 << ")" << endl;
	rl_set_screen_size( height/charHeight+1, width/charWidth+1 );*/
#ifndef __WIN32__
	rl_resize_terminal();
#endif
}

void OctaveUI::createMenubar(void)
{

  // File Menu
  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem("_File", fileMenu) );
  fileMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_New", 
                                                           Gtk::AccelKey("<control>N"),
                                                           sigc::mem_fun(*this, &OctaveUI::onNew) ) );
  fileMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Open", 
                                                           Gtk::AccelKey("<control>O"),
                                                           sigc::mem_fun(*this, &OctaveUI::onOpen) ) );
  fileMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Save") );
  fileMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "Save _As...") );
  fileMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Quit",  
                                                           Gtk::AccelKey("<control>Q"),
                                                           sigc::mem_fun(*this, &OctaveUI::onQuit) ) );

  // Edit Menu
  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem("_Edit", editMenu) );
  editMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "Cu_t", 
                                                           Gtk::AccelKey("<control>W"),
                                                           sigc::mem_fun(*this, &OctaveUI::onCut) ) );
  editMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Copy", 
                                                           Gtk::AccelKey("<alt>W"),
                                                           sigc::mem_fun(*this, &OctaveUI::onCopy) ) );
  editMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Paste", 
                                                           Gtk::AccelKey("<control>Y"),
                                                           sigc::mem_fun(*this, &OctaveUI::onPaste) ) );
  editMenu.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Preferences", 
                                                           sigc::mem_fun(*this, &OctaveUI::onPreferences) ) );

  // Debug Menu
  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem("De_bug", debugMenu) );

  // Window Menu
  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem("Win_dow", windowMenu) );

  Gtk::Menu_Helpers::MenuList& windowList = windowMenu.items();
  windowList.push_back( Gtk::Menu_Helpers::CheckMenuElem( "Octave Window", sigc::mem_fun(*this, &OctaveUI::on_toggle_octave) ) );
  windowMenuMap["octave"] = dynamic_cast<Gtk::CheckMenuItem*>(&windowList.back());
  (windowMenuMap["octave"])->property_active().set_value(true);
  windowList.push_back( Gtk::Menu_Helpers::CheckMenuElem( "Command History", sigc::mem_fun(*this, &OctaveUI::on_toggle_history) ) );
  windowMenuMap["history"] = dynamic_cast<Gtk::CheckMenuItem*>(&windowList.back());
  (windowMenuMap["history"])->property_active().set_value(true);
  windowList.push_back( Gtk::Menu_Helpers::CheckMenuElem( "File Browser", sigc::mem_fun(*this, &OctaveUI::on_toggle_filechooser) ) );
  windowMenuMap["filechooser"] = dynamic_cast<Gtk::CheckMenuItem*>(&windowList.back());
  (windowMenuMap["filechooser"])->property_active().set_value(true);
  windowList.push_back( Gtk::Menu_Helpers::CheckMenuElem( "Variables View", sigc::mem_fun(*this, &OctaveUI::on_toggle_variables) ) );
  windowMenuMap["variables"] = dynamic_cast<Gtk::CheckMenuItem*>(&windowList.back());
  (windowMenuMap["variables"])->property_active().set_value(true);


  // Help Menu
  //  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem("_Help", helpMenu) );
  menuBar.items().push_back( Gtk::Menu_Helpers::MenuElem( "_Help",  
                                                           Gtk::AccelKey("<control>H"),
                                                           sigc::mem_fun(*this, &OctaveUI::onHelp) ) );


}

void OctaveUI::createToolbar(void)
{
  m_toolbar.set_tooltips (true);
  m_toolbar.m_path_chooser.m_dir_button.signal_selection_changed ().connect (sigc::mem_fun (*this, &OctaveUI::onPathChooserChanged));
  m_toolbar.m_tool_path_up.signal_clicked().connect( sigc::mem_fun( *this, &OctaveUI::onDirUpButton ) );

  // Add callbacks for the toolbar buttons
  m_toolbar.m_tool_new.signal_clicked().connect( sigc::mem_fun( *this, &OctaveUI::onNew) );
  m_toolbar.m_tool_open.signal_clicked().connect( sigc::mem_fun( *this, &OctaveUI::onOpen) );

  m_toolbar.m_tool_help.signal_clicked().connect( sigc::mem_fun( *this, &OctaveUI::onHelp) );

}

void OctaveUI::createFileChooserDockItem(void)
{
  mFileChooserWidget.signal_file_activated().connect( sigc::mem_fun( *this, &OctaveUI::onFileActivated ) );
  mFileChooserWidget.signal_current_folder_changed().connect( sigc::mem_fun( *this, &OctaveUI::onDirChanged ) );
  fileChooserDock.get_vbox()->add(mFileChooserWidget);
  dockPane.addItem( fileChooserDock, DockItem::TOP );
}

void OctaveUI::createVariablesDockItem(void)
{
  mVariablesScrollWindow.add(mVariablesTreeView);
  variablesDock.get_vbox()->add( mVariablesScrollWindow );
  //mVariablesTreeView.set_size_request( 256,256 );
  dockPane.addItem( variablesDock, DockItem::CENTER );

}

void OctaveUI::createHistoryDockItem(void)
{
  mHistoryTreeView.onCommandExecute.connect( sigc::mem_fun( *this, &OctaveUI::onCommandExecute ) );
//	mHistoryTreeView.onCommandCopy.connect( sigc::mem_fun( *this, &OctaveUI::onCommandCopy ) );
  mHistoryScrollWindow.add(mHistoryTreeView);
  historyDock.get_vbox()->add(mHistoryScrollWindow);
  //mHistoryTreeView.set_size_request( 256,256 );
  dockPane.addItem( historyDock, DockItem::LEFT );
}


void OctaveUI::createOctaveVteDockItem(void)
{
  // Create the VTE terminal window
  gboolean committed;
  vteTerminal = vte_terminal_new();

  // Set the terminla emulation for mac
#if 0
  //#ifdef __APPLE__
  char        execPath[1024+1];
  uint32_t    execPathSize = 1024;
  execPathSize = sizeof(execPath);
  (void) _NSGetExecutablePath(execPath, &execPathSize);

  std::string execPathCpp(execPath);
  std::string path2 = execPathCpp.substr(0,execPathCpp.length()-10) +
    std::string("../Resources/share/vte/termcap/xterm");
  vte_terminal_set_termcap( VTE_TERMINAL(vteTerminal),path2.c_str(),true);
  printf("Setting the termcap file: \n%s\n", path2.c_str());
#endif

  //vte_terminal_set_size( (VteTerminal*)vteTerminal, 80,25 );
#ifdef __WIN32__
  vte_terminal_set_font_from_string( (VteTerminal*)vteTerminal, "Lucida Console 10" );
#else
  vte_terminal_set_font_from_string( (VteTerminal*)vteTerminal, "Monospace 12" );
#endif

  // NEW PTY SINGLE PROCESS METHOD
  // Starting the new pty process
  
#if 1
#ifndef __WIN32__
  int fdm;
  int fds;
  if ( openpty (&fdm, &fds, 0, 0, 0) < 0 )
  {
    fprintf (stderr, "oops!\n");
  }
  
  dup2 (fds, 0);
  dup2 (fds, 1);
  dup2 (fds, 2);
  
  int rc = pthread_create (&octave_thread, 0, octave_main_wrapper, 0);

  usleep(2000000);

#else


  int fdm;
  int fds;

  if (_vte_ptywin_openpty (&fdm, &fds) < 0)
  {
    fprintf (stderr, "oops!\n");
  }

  dup2 (_vte_ptywin_get_read (fds), 0);
  dup2 (_vte_ptywin_get_write (fds), 1);
  dup2 (_vte_ptywin_get_write (fds), 2);

  SetStdHandle (STD_INPUT_HANDLE, (HANDLE)_get_osfhandle (0));
  SetStdHandle (STD_OUTPUT_HANDLE, (HANDLE)_get_osfhandle (1));
  SetStdHandle (STD_ERROR_HANDLE, (HANDLE)_get_osfhandle (2));

  _vte_ptywin_close (fds);

  octave_thread = CreateThread(0, 0, octave_main_wrapper, 0, 0, NULL);
#endif // if..else.. on __WIN32__
  
  vte_terminal_set_pty (VTE_TERMINAL (vteTerminal), fdm);
  // END NEW

#else
  // OLD METHOD
  vte_terminal_fork_command( VTE_TERMINAL(vteTerminal), "octave", NULL, NULL, NULL, 0,0,0);
#endif // if..else.. on (1) for old or new method of VTE

  GtkAdjustment* adjust = vte_terminal_get_adjustment(  (VteTerminal*)vteTerminal );
  Gtk::Adjustment* adjust_mm = Glib::wrap(adjust);

  g_signal_connect (G_OBJECT (vteTerminal), "commit", G_CALLBACK (terminal_widget_commit), this); 
  vtePanel = Glib::wrap(vteTerminal);
  vtePanel->signal_size_allocate().connect( sigc::mem_fun( *this, &OctaveUI::onVteSizeChanged ) );
  vscroll.set_adjustment( *adjust_mm );

  // Create the dock for the vte panel
  //octaveDock = new Gnome::Gdl::DockItem("Octave", "Octave", Gnome::Gdl::DOCK_ITEM_BEH_CANT_ICONIFY|Gnome::Gdl::DOCK_ITEM_BEH_CANT_CLOSE );
  mOctaveScrollWindow.set_size_request(1,1);
  mOctaveScrollWindow.pack_start (*vtePanel);
  mOctaveScrollWindow.pack_end (vscroll, Gtk::PACK_SHRINK);
  octaveDock.get_vbox()->add (mOctaveScrollWindow);
  g_object_set (octaveDock.getWidget().gobj(), "resize", TRUE, NULL);
  dockPane.addItem( octaveDock, DockItem::CENTER );

  //octaveDock.get_vbox()->set_spacing(0);

  editorPlaceholder = GDL_DOCK_PLACEHOLDER(gdl_dock_placeholder_new ("ph1", GDL_DOCK_OBJECT (octaveDock.getWidget().gobj()), GDL_DOCK_TOP, TRUE));

  //octaveDock->show_all();
}

///////////////////////////////////////////////////////////////////////////////////////
void OctaveUI::onPathChooserChanged(void)
{
  std::string olddir = mFileChooserWidget.get_current_folder();
  std::string newdir = m_toolbar.m_path_chooser.m_dir_button.get_current_folder();
  
  octave_value up (newdir);
  octave_value_list out = feval ("cd", up, 1);

  /*
  int cd_ok = octave_env::chdir ("..");
  */
  newdir = octave_env::getcwd();

  //int cd_ok = octave_env::chdir (newdir);
  //if( cd_ok && (newdir.compare(olddir)!=0) )
  if( (newdir.compare(olddir)!=0) )
  {
    //load_path::update ();
    mFileChooserWidget.set_current_folder(newdir);
  }
}

///////////////////////////////////////////////////////////////////////////////////////
void OctaveUI::onDirChanged( void )
{
  std::string olddir = m_toolbar.m_path_chooser.m_dir_button.get_current_folder();
  std::string newdir = mFileChooserWidget.get_current_folder();
  int cd_ok = octave_env::chdir (newdir);
  if( cd_ok && (newdir.compare(olddir) != 0) )
  {
    m_toolbar.m_path_chooser.m_dir_button.set_current_folder(newdir);
  }
  //rl_point = 0;
  //rl_line_buffer[0] = '\0';
  //rl_end = 0;
  rl_done = 1;
}


///////////////////////////////////////////////////////////////////////////////////////
void OctaveUI::onDirUpButton( void )
{
  octave_value up ("..");
  octave_value_list out = feval ("cd", up, 1);

  /*
  int cd_ok = octave_env::chdir ("..");
  */
  std::string newdir = octave_env::getcwd();
  
  //if( cd_ok )
  {
    mFileChooserWidget.set_current_folder(newdir);
  }

  //if( cd_ok )
  {
    m_toolbar.m_path_chooser.m_dir_button.set_current_folder(newdir);
  }

  //load_path::update();
  //rl_point = 0;
  //rl_line_buffer[0] = '\0';
  //rl_end = 0;
  rl_done = 1;
}


void OctaveUI::onHelp(void)
{
#if HAVE_WEBKIT
  HelpWindow::GetInstance(true);
#else
  std::cout << "WebKit Help Window was not compiled in" << std::endl;
#endif
}
