/*

Copyright (C) 2008 Michael Goffioul,
              2008,2009 John P. Swensen.

This file is part of GtkHandles.

GtkHandles is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

GtkHandles is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GtkHandles; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>.

*/

#include <octave/config.h>
#include <octave/graphics.h>
#include <octave/gl-render.h>

#include "gh-figure.h"



#include <gdk/gdk.h>
#include <gtkgl/gdkgl.h>
#include <gtkgl/gtkglarea.h>



#include<iostream>
using namespace std;

typedef struct __GtkHandles_Figure GtkHandlesFigure;
typedef struct __GtkHandles_Sync GtkHandlesSync;


/*static*/ GtkmmGLFigure* GtkmmGLFigure::figureNew (double handle)
{
  GtkmmGLSync_t sync;
  sync.func = sigc::ptr_fun(&GtkmmGLFigure::__figureMake);
  sync.data = &handle;
  GtkmmGLSync::ExecuteInMainLoop (sync);

  usleep(2000);

  ((GtkmmGLFigure*)sync.returnData)->show_all();

  return (GtkmmGLFigure*)sync.returnData;

}

/*static*/ gpointer GtkmmGLFigure::__figureMake (gpointer data)
{
  double handle = *(double *)data;
  GtkmmGLFigure* tmp = new GtkmmGLFigure(handle);
  return tmp;
}

/*static*/ void GtkmmGLFigure::figureClose (GtkmmGLFigure* fig)
{
  GtkmmGLSync_t sync;
  sync.func = sigc::ptr_fun(&GtkmmGLFigure::__figureClose);
  sync.data = fig;

  GtkmmGLSync::ExecuteInMainLoop (sync);
}

/*static*/ gboolean GtkmmGLFigure::figureDraw (GtkWidget *area,
					       GdkEventExpose *event,
					       gpointer user_data)
{
  static int prevWidth=-1, prevHeight=-1, prevX=-1, prevY=-1;
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  opengl_renderer r;

  int width, height, rootx, rooty;
  fig->get_size(width,height);
  fig->get_position(rootx, rooty);

  if (event->count > 0)
    return TRUE;

  if (gtk_gl_area_make_current (GTK_GL_AREA (fig->m_glArea)))
  {
    
    graphics_object go;
    go = gh_manager::get_object (graphics_handle (fig->m_handle));
    figure::properties& fp =
      dynamic_cast<figure::properties&> (go.get_properties ());

    Matrix pos = fp.get_position().matrix_value();
    if (pos(2)!=width || pos(3)!=height)
    {
      fig->resize(pos(2),pos(3));
    }
    if (pos(0)!=rootx || pos(1)!=rooty)
    {
      fig->move(pos(0),pos(1));
      prevX = pos(0);
      prevY = pos(1);
    }

    r.set_viewport (fig->m_glArea->allocation.width,
    		    fig->m_glArea->allocation.height);
    r.draw (fig->m_handle);

    gtk_gl_area_swap_buffers (GTK_GL_AREA (fig->m_glArea));
  }

  return TRUE;
}

/*static*/ gboolean GtkmmGLFigure::figureInit (GtkWidget *area,
						 gpointer user_data)
{
  GtkmmGLFigure *fig = (GtkmmGLFigure*)user_data;

  if (gtk_gl_area_make_current (GTK_GL_AREA (fig->m_glArea)))
  {
    // do nothing
  }

  return TRUE;
}

/*static*/ gboolean GtkmmGLFigure::figureReshape (GtkWidget *area,
						  GdkEventConfigure *event,
						  gpointer user_data)
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  opengl_renderer r;

  if (gtk_gl_area_make_current (GTK_GL_AREA (fig->m_glArea)))
  {
    int width, height, rootx, rooty;
    fig->get_size(width,height);
    fig->get_position(rootx, rooty);

    graphics_object go;
    go = gh_manager::get_object (graphics_handle (fig->m_handle));
    figure::properties& fp =
      dynamic_cast<figure::properties&> (go.get_properties ());

    Matrix pos = fp.get_position().matrix_value();
    pos(0) = rootx;
    pos(1) = rooty;
    pos(2) = width;
    pos(3) = height;
    
    fp.set_position(pos);
    r.set_viewport (fig->m_glArea->allocation.width,
    		    fig->m_glArea->allocation.height);
    
    //      glViewport (0, 0,
    //		  fig->m_glArea->allocation.width,
    //		  fig->m_glArea->allocation.height);
  }

  return TRUE;
}

gpointer figureDeleteHelper (gpointer data)
{
  usleep(200000);

  double h = *(double*)data;
  graphics_object go = gh_manager::get_object (h);
  graphics_handle parent_h = go.get_parent ();
  graphics_object parent_obj = gh_manager::get_object (parent_h);
  gh_manager::free (h);
  parent_obj.remove_child (h);
}

/*static*/ gboolean GtkmmGLFigure::figureDelete (GtkWidget *window,
						 GdkEvent* evt,
						 gpointer user_data)
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  double h = fig->m_handle;

  // I had to spawn a new thread, because the sync method wasn't
  // working from inside a callback (??never getting back to idle
  // loop??)
  //g_thread_create (figureDeleteHelper, (gpointer)&h, FALSE, NULL);

  // Correct method
  graphics_object go = gh_manager::get_object (h);
  graphics_handle hndl = go.get_handle ();
  gh_manager::post_callback(hndl, "closerequestfcn");
  
  
  return TRUE;
}

GtkmmGLFigure::GtkmmGLFigure (double h)
  : m_handle(h), m_mode(GLFIGURE_EDIT_MODE)
{
  Matrix m;
  graphics_object go;

    /////////
 int attr_list[] = {
      GDK_GL_RGBA,
      GDK_GL_RED_SIZE, 1,
      GDK_GL_GREEN_SIZE, 1,
      GDK_GL_BLUE_SIZE, 1,
      GDK_GL_DOUBLEBUFFER,
      GDK_GL_NONE
  };

  go = gh_manager::get_object (graphics_handle (h));

  m = go.get_properties ().get_boundingbox ();
  set_default_size (m(2), m(3));


  m_glArea = gtk_gl_area_new (attr_list);
  gtk_widget_set_events (GTK_WIDGET (m_glArea),
			 GDK_EXPOSURE_MASK|
			 GDK_BUTTON_PRESS_MASK|
			 GDK_BUTTON_RELEASE_MASK|
			 GDK_POINTER_MOTION_MASK|
			 GDK_SCROLL_MASK);
 
  g_signal_connect (G_OBJECT (m_glArea), "expose-event",
		    G_CALLBACK (figureDraw), (gpointer)this);
 
  g_signal_connect (G_OBJECT (m_glArea), "realize",
		    G_CALLBACK (figureInit), (gpointer)this);

  g_signal_connect (G_OBJECT (m_glArea), "configure-event",
		    G_CALLBACK (figureReshape), (gpointer)this);
  
  g_signal_connect (G_OBJECT (this->gobj()), "delete-event",
		    G_CALLBACK (figureDelete), (gpointer)this);

  g_signal_connect (G_OBJECT (m_glArea), "button_press_event",
		    GTK_SIGNAL_FUNC (onMouseDown), (gpointer)this);

  g_signal_connect (G_OBJECT (m_glArea), "button_release_event",
		    GTK_SIGNAL_FUNC (onMouseUp), (gpointer)this);

  g_signal_connect (G_OBJECT (m_glArea), "motion_notify_event",
		    GTK_SIGNAL_FUNC (onMouseMotion), (gpointer)this);


  g_signal_connect (G_OBJECT (m_glArea), "scroll_event",
		    GTK_SIGNAL_FUNC (onMouseScroll), (gpointer)this);

  m_glAreaMM = Glib::wrap(m_glArea);
  
  gtk_widget_set_usize(GTK_WIDGET(m_glArea), m(2), m(3));

  m_toolbar.m_tool_edit.set_active(true);
  m_toolbar.m_tool_edit.signal_toggled().connect(sigc::mem_fun( *this, &GtkmmGLFigure::onEditModeSelected) );
  m_toolbar.m_tool_zoompan.signal_toggled().connect(sigc::mem_fun( *this, &GtkmmGLFigure::onZoompanModeSelected) );
  m_toolbar.m_tool_rotate.signal_toggled().connect(sigc::mem_fun( *this, &GtkmmGLFigure::onRotateModeSelected) );
  m_toolbar.m_tool_datainspect.signal_toggled().connect(sigc::mem_fun( *this, &GtkmmGLFigure::onDatainspectModeSelected) );

  this->signal_key_press_event().connect(sigc::mem_fun(*this, &GtkmmGLFigure::onKeyPress));
  

  m_vbox.pack_start(m_toolbar,Gtk::PACK_SHRINK);
  m_vbox.pack_start(*m_glAreaMM,Gtk::PACK_EXPAND_WIDGET);
  this->add(m_vbox);

  m_toolbar.show();
  gtk_widget_show (m_glArea);
  m_vbox.show();

  char buf[120];
  sprintf(buf,"Figure: %d", (int)m_handle);
  set_title(std::string(buf));

  if (go.get_properties ().is_visible ())
    show();
}

void pixel2pos (const figure::properties& fp, int px, int py, double& xx, double& yy)
{
  graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());

  if (ax && ax.isa ("axes"))
  {
    axes::properties& ap =
      dynamic_cast<axes::properties&> (ax.get_properties ());
    ColumnVector pp = ap.pixel2coord (px, py);
    xx = pp(0);
    yy = pp(1);
  }
}

bool GtkmmGLFigure::onKeyPress(GdkEventKey* event)
{
  double h = this->m_handle;

  if (this->m_mode == GLFIGURE_ZOOMPAN_MODE)
  {
    if (event->keyval == GDK_z || event->keyval == GDK_Z)
    {
      // Get the current mouse position and zoom in
      // If the mouse is outside of the GL area, then zoom to the center
      int x,y;
      this->get_pointer(x,y);

      double x0,y0,x1,y1;
      graphics_object go = gh_manager::get_object (graphics_handle (this->m_handle));
      figure::properties& fp = dynamic_cast<figure::properties&> (go.get_properties ());
      graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());
      if (ax && ax.isa ("axes"))
      {
	axes::properties& ap = dynamic_cast<axes::properties&> (ax.get_properties ());
	
	int px0 = x-10;
	int py0 = y-10;
	int px1 = x+10;
	int py1 = y+10;
	
	pixel2pos (fp, px0, py0, x0, y0);
	pixel2pos (fp, px1, py1, x1, y1);
	Matrix xl (1,2,0);
	Matrix yl (1,2,0);
	if (x0 < x1)
	  {
	    xl(0) = x0;
	    xl(1) = x1;
	  }
	else
	  {
	    xl(0) = x1;
	    xl(1) = x0;
	  }
	
	if (y0 < y1)
	  {
	    yl(0) = y0;
	    yl(1) = y1;
	  }
	else
	  {
	    yl(0) = y1;
	    yl(1) = y0;
	  }
	ap.zoom (xl, yl);
      } 

      GdkRectangle area;
      area.x = 0;
      area.y = 0;
      area.width  = this->m_glArea->allocation.width;
      area.height = this->m_glArea->allocation.height;
      gtk_widget_draw(this->m_glArea, &area);
      
      return true;

    }
    else if (event->keyval == GDK_a || event->keyval == GDK_A)
    {
      // Zoom out one level
      graphics_object go = gh_manager::get_object (graphics_handle (this->m_handle));
      figure::properties& fp = dynamic_cast<figure::properties&> (go.get_properties ());
      graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());
      if (ax && ax.isa ("axes"))
      {
	axes::properties& ap = dynamic_cast<axes::properties&> (ax.get_properties ());
	ap.unzoom();
      }

      GdkRectangle area;
      area.x = 0;
      area.y = 0;
      area.width  = this->m_glArea->allocation.width;
      area.height = this->m_glArea->allocation.height;
      gtk_widget_draw(this->m_glArea, &area);
      
      return true;
    }
    //std::cout << "Key pressed in zoom mode" << std::endl;

  }

  return false;
}

void GtkmmGLFigure::onEditModeSelected()
{
  // Deselect all other modes
  m_toolbar.m_tool_zoompan.set_active(false);
  m_toolbar.m_tool_rotate.set_active(false);
  m_toolbar.m_tool_datainspect.set_active(false);

  m_mode = GLFIGURE_EDIT_MODE;
}

void GtkmmGLFigure::onZoompanModeSelected()
{
  // Deselect all other modes
  m_toolbar.m_tool_edit.set_active(false);
  m_toolbar.m_tool_rotate.set_active(false);
  m_toolbar.m_tool_datainspect.set_active(false);

  m_mode = GLFIGURE_ZOOMPAN_MODE;
}

void GtkmmGLFigure::onRotateModeSelected()
{
  // Deselect all other modes
  m_toolbar.m_tool_edit.set_active(false);
  m_toolbar.m_tool_zoompan.set_active(false);
  m_toolbar.m_tool_datainspect.set_active(false);

  m_mode = GLFIGURE_ROTATE_MODE;
}

void GtkmmGLFigure::onDatainspectModeSelected()
{
  // Deselect all other modes
  m_toolbar.m_tool_edit.set_active(false);
  m_toolbar.m_tool_zoompan.set_active(false);
  m_toolbar.m_tool_rotate.set_active(false);

  m_mode = GLFIGURE_DATAINSPECT_MODE;
}

GtkmmGLFigure::~GtkmmGLFigure (void)
{
  gtk_widget_hide (m_glArea);
  m_vbox.hide();
  hide();
  gtk_gl_area_endgl ((GtkGLArea*)m_glArea);
  m_vbox.remove(*m_glAreaMM);
  gtk_widget_destroy (m_glArea);
}

bool inClickMove = false;
int xStart, yStart;

/*static*/ gint GtkmmGLFigure::onMouseDown (GtkWidget* widget, 
					    GdkEventButton* event,
					    gpointer user_data) 
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  double h = fig->m_handle;

  int x = event->x;
  int y = event->y;

  if (fig->m_mode == GLFIGURE_ZOOMPAN_MODE)
  {

    double x0,y0,x1,y1;
    graphics_object go = gh_manager::get_object (graphics_handle (fig->m_handle));
    figure::properties& fp = dynamic_cast<figure::properties&> (go.get_properties ());
    graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());
    if (ax && ax.isa ("axes"))
    {
      axes::properties& ap = dynamic_cast<axes::properties&> (ax.get_properties ());

      if (event->button == 1 && event->type != GDK_2BUTTON_PRESS)
      {

	int px0 = x-10;
	int py0 = y-10;
	int px1 = x+10;
	int py1 = y+10;
	
	pixel2pos (fp, px0, py0, x0, y0);
	pixel2pos (fp, px1, py1, x1, y1);
	Matrix xl (1,2,0);
	Matrix yl (1,2,0);
	if (x0 < x1)
	  {
	    xl(0) = x0;
	    xl(1) = x1;
	  }
	else
	  {
	    xl(0) = x1;
	    xl(1) = x0;
	  }
	
	if (y0 < y1)
	  {
	    yl(0) = y0;
	    yl(1) = y1;
	  }
	else
	  {
	    yl(0) = y1;
	    yl(1) = y0;
	  }
	ap.zoom (xl, yl);
      }
      else if (event->button == 2)
      {
	
      }
      else if (event->button == 3 || (event->button==1&&event->type==GDK_2BUTTON_PRESS))
      {
	ap.clear_zoom_stack();
      }
      else
      {
	std::cout << "Unknown event: " << event->button << std::endl;
      }

      GdkRectangle area;
      area.x = 0;
      area.y = 0;
      area.width  = fig->m_glArea->allocation.width;
      area.height = fig->m_glArea->allocation.height;
      gtk_widget_draw(fig->m_glArea, &area);

      return TRUE;
    }
  }
  else if (fig->m_mode == GLFIGURE_ROTATE_MODE)
  {
    cout << "Begin rotate" << endl;
    // Set the current (X,Y) as the basis of the rotate operation
    xStart = x;
    yStart = y;

    // Put the figure in drag mode
    inClickMove = true;
  }
    
  return FALSE;
}

/*static*/ gint GtkmmGLFigure::onMouseUp (GtkWidget* widget, 
					    GdkEventButton* event,
					    gpointer user_data) 
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  double h = fig->m_handle;

  int x = event->x;
  int y = event->y;

  if (fig->m_mode == GLFIGURE_ZOOMPAN_MODE)
  {
  }
  else if (fig->m_mode == GLFIGURE_ROTATE_MODE)
  {
    cout << "End rotate" << endl;

    // Take out of rotate mode
    inClickMove = false;
  }
  return FALSE;
}


/*static*/ gint GtkmmGLFigure::onMouseMotion (GtkWidget* widget, 
					      GdkEventMotion* event,
					      gpointer user_data) 
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  double h = fig->m_handle;

  int x = event->x;
  int y = event->y;

  if (fig->m_mode == GLFIGURE_ZOOMPAN_MODE)
  {
  }
  else if (fig->m_mode == GLFIGURE_ROTATE_MODE)
  {
    if (inClickMove)
    {
      cout << "Position from nominal: (" << x-xStart << "," << y-yStart << ")" << endl;
      // Use the (X,Y)-(Xstart,Ystart) to set the azimuth and elevation      
      Matrix v(1,2);
      //v(0) = -(x-xStart);
      //v(1) = y-yStart;
      //octave_value_list args;
      //args(0) = static_cast<plot_window*> (data)->number ();
      //feval ("close", args);

      graphics_object go = gh_manager::get_object (graphics_handle (fig->m_handle));
      figure::properties& fp = dynamic_cast<figure::properties&> (go.get_properties ());
      graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());
      if (ax && ax.isa ("axes"))
      {
	axes::properties& ap = dynamic_cast<axes::properties&> (ax.get_properties ());
	property tmp = ap.get_property("View");
	Matrix m = tmp.get().matrix_value();
	v(0) = m(0) - (x-xStart);
	v(1) = m(1) + (y-yStart);
	ap.set("View",octave_value(v));

	xStart = x;
	yStart = y;
      }

      GdkRectangle area;
      area.x = 0;
      area.y = 0;
      area.width  = fig->m_glArea->allocation.width;
      area.height = fig->m_glArea->allocation.height;
      gtk_widget_draw(fig->m_glArea, &area);

      return TRUE;
    }
  }
  
  return FALSE;
}


/*static*/ gint GtkmmGLFigure::onMouseScroll (GtkWidget* widget, 
					      GdkEventScroll* event,
					      gpointer user_data) 
{
  GtkmmGLFigure *fig = (GtkmmGLFigure *)user_data;
  double h = fig->m_handle;

  int x = event->x;
  int y = event->y;

  if (fig->m_mode == GLFIGURE_ZOOMPAN_MODE)
  {

    double x0,y0,x1,y1;
    graphics_object go = gh_manager::get_object (graphics_handle (fig->m_handle));
    figure::properties& fp = dynamic_cast<figure::properties&> (go.get_properties ());
    graphics_object ax = gh_manager::get_object (fp.get_currentaxes ());
    if (ax && ax.isa ("axes"))
    {
      axes::properties& ap = dynamic_cast<axes::properties&> (ax.get_properties ());

      if (event->direction == GDK_SCROLL_UP)
      {

	int px0 = x-10;
	int py0 = y-10;
	int px1 = x+10;
	int py1 = y+10;
	
	pixel2pos (fp, px0, py0, x0, y0);
	pixel2pos (fp, px1, py1, x1, y1);
	Matrix xl (1,2,0);
	Matrix yl (1,2,0);
	if (x0 < x1)
	  {
	    xl(0) = x0;
	    xl(1) = x1;
	  }
	else
	  {
	    xl(0) = x1;
	    xl(1) = x0;
	  }
	
	if (y0 < y1)
	  {
	    yl(0) = y0;
	    yl(1) = y1;
	  }
	else
	  {
	    yl(0) = y1;
	    yl(1) = y0;
	  }
	ap.zoom (xl, yl);
      }
      else if (event->direction == GDK_SCROLL_DOWN)
      {
	ap.unzoom();
      }
      else
      {
	std::cout << "Unknown event: " << event->direction << std::endl;
      }

      GdkRectangle area;
      area.x = 0;
      area.y = 0;
      area.width  = fig->m_glArea->allocation.width;
      area.height = fig->m_glArea->allocation.height;
      gtk_widget_draw(fig->m_glArea, &area);

      return TRUE;
    }
  }
  else if (fig->m_mode == GLFIGURE_ROTATE_MODE)
  {
    // We want to detect when the mouse first is pressed and then how it moves relative to that point
  }

  return FALSE;
}

/*static*/ gpointer GtkmmGLFigure::__figureRedraw(gpointer data)
{
  GtkmmGLFigure *fig = (GtkmmGLFigure*)data;
  gtk_widget_queue_draw_area (fig->m_glArea, 0, 0,
			      fig->m_glArea->allocation.width,
			      fig->m_glArea->allocation.height);
  return NULL;
}

void GtkmmGLFigure::figureRedraw (void)
{
  GtkmmGLSync_t sync;
  sync.func = sigc::ptr_fun(&GtkmmGLFigure::__figureRedraw);
  sync.data = (gpointer)this;

  GtkmmGLSync::ExecuteInMainLoop (sync);
}

