/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2; coding: utf-8 -*- 
 *
 * Copyright (C) 2007, 2008, 2009 John P. Swensen
 *
 * This file is as a part of OctaveDE.
 *
 * Octave is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Octave; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * */

#include "mEditor.h"
#include "GtkSourceViewUtilities.h"
//#include "OctaveUI.h"

#if defined(__APPLE__)
#include <sys/param.h>
#include <mach-o/dyld.h>
#endif

#include <sys/stat.h>

#define MARK_TYPE_1      "one"
#define MARK_TYPE_2      "two"


GtkSourceLanguage *get_language_for_mime_type (const gchar *mime);
//static gboolean gtk_source_buffer_save_file (GtkSourceBuffer *source_buffer, const gchar *filename, GError **error);

/*static*/ MEditor* MEditor::__instance = NULL;

//////////////////////////////////////////////////////////////////////////////////
/*static*/ MEditor* MEditor::GetInstance (bool show_it)
{
	/*
  if ( __instance == NULL )
  {
    __instance = new MEditor ();
  }

  if (show_it)
	  __instance->show_all ();
	*/

  return __instance;
}

//////////////////////////////////////////////////////////////////////////////////
/*static*/ MEditor* MEditor::GetInstance (Dock& mainDock, bool show_it)
{
  if ( __instance == NULL )
  {
    __instance = new MEditor (mainDock);
  }

  if (show_it)
	  __instance->show_all ();

  return __instance;
}

////////////////////////////////////////////////////////////////////////////////////
MEditor::MEditor (Dock& mainDock) : Window (), dockPane(mainDock)
{
	set_title ("OctaveDE Editor");
  idx = 0;
  set_size_request (800,600);

  add(m_Box); //We can put a MenuBar at the top of the box and other stuff below it.

  //Fill menus:
	//// FILE MENU
  menuBar.items ().push_back( Gtk::Menu_Helpers::MenuElem("_File", fileMenu) );
  fileMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_New", 
                                                           Gtk::AccelKey("<control>N"),
                                                           sigc::mem_fun(*this, &MEditor::onNew) ) );
  fileMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_Open", 
                                                           Gtk::AccelKey("<control>O"),
                                                           sigc::mem_fun(*this, &MEditor::onOpen) ) );
  fileMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_Save",
																													 Gtk::AccelKey("<control>S"),
																													 sigc::mem_fun(*this, &MEditor::onSave) ) );
  fileMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "Save _As...",
																													 Gtk::AccelKey("<control>A"),
																													 sigc::mem_fun(*this, &MEditor::onSaveAs) ) );
  fileMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_Quit",  
                                                           Gtk::AccelKey("<control>Q"),
                                                           sigc::mem_fun(*this, &MEditor::onQuit) ) );

	// EDIT MENU
  menuBar.items ().push_back( Gtk::Menu_Helpers::MenuElem("_Edit", editMenu) );
	editMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "Cu_t", 
                                                           Gtk::AccelKey("<control>X"),
                                                           sigc::mem_fun(*this, &MEditor::onCut) ) );
	editMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_Cut", 
                                                           Gtk::AccelKey("<control>C"),
                                                           sigc::mem_fun(*this, &MEditor::onCopy) ) );
	editMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "_Paste", 
                                                           Gtk::AccelKey("<control>V"),
                                                           sigc::mem_fun(*this, &MEditor::onPaste) ) );
	editMenu.items ().push_back( Gtk::Menu_Helpers::MenuElem( "Preferences", 
                                                           sigc::mem_fun(*this, &MEditor::onPreferences) ) );


  menuBar.items ().push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::HELP, helpMenu) );

	lm = gtk_source_language_manager_get_default ();

  //Add the MenuBar and the dock to the window
  m_Box.pack_start (menuBar, Gtk::PACK_SHRINK);
  m_Box.pack_start (dockPane.getWidget ());
}

////////////////////////////////////////////////////////////////////////////////////
MEditor::~MEditor ()
{
  // Free up all the buffers
	while (!m_editorPanels.empty())
	{
		MEditorPanel* tmp = m_editorPanels.begin()->second;
		m_editorPanels.erase (m_editorPanels.begin());
		delete tmp;
	}

}


////////////////////////////////////////////////////////////////////////////////////
gboolean button_press_cb (GtkWidget *widget, GdkEventButton *ev, gpointer user_data)
{
	cout << "button press caught" << endl;
	return false;
}

////////////////////////////////////////////////////////////////////////////////////
void MEditor::openFile (std::string filename)
{
	GError *error;
	GdkPixbuf *pixbuf;

	// Check to see if the same file is already open, so we can bring it to the
  // top and have it grab the focus
	bool fileAlreadyOpen = false;
	
	std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
	for (it=m_editorPanels.begin () ; it!=m_editorPanels.end () ; it++)
	{
		if ((it->second)->get_title ().compare (filename)==0)
		{
			fileAlreadyOpen = true;
			break;
		}
	}

	if (fileAlreadyOpen)
	{
		// Simply bring it to the front and give it focus
		(it->second)->present ();
		(it->second)->grab_focus ();
	}
	else
	{
		// Get a short filename to diplay on the tab
		std::string shortName = Glib::path_get_basename (filename);
	
		// Create the new DockItem and gsv buffer
		MEditorPanel* panel = new MEditorPanel(dockPane,filename);
		panel->set_tablabel (shortName);
		panel->setShortName (shortName);
		dockPane.addItem (*panel, DockItem::CENTER);
		
		Gtk::TextView* view = panel->getView();
		m_editorPanels[view] = panel;

		this->show_all();

	}
}

////////////////////////////////////////////////////////////////////////////////////
int MEditor::openFileDialog (void)
{
  int retval = 0;
  // Open a file dialog
  Gtk::FileChooserDialog dialog ("Please choose a file", Gtk::FILE_CHOOSER_ACTION_OPEN);
  //dialog.set_transient_for(*this);

  //Add response buttons the the dialog:
  dialog.add_button (Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  dialog.add_button (Gtk::Stock::OPEN, Gtk::RESPONSE_OK);

  //Add filters, so that only certain file types can be selected:
  Gtk::FileFilter filter_m;
  filter_m.set_name ("m files");
  filter_m.add_pattern ("*.m");
  dialog.add_filter (filter_m);

  Gtk::FileFilter filter_any;
  filter_any.set_name ("Any files");
  filter_any.add_pattern ("*");
  dialog.add_filter (filter_any);

  //Show the dialog and wait for a user response:
  int result = dialog.run ();

  //Handle the response:
  std::string filename;
  switch (result)
  {
    case (Gtk::RESPONSE_OK):
      filename = dialog.get_filename (); //Notice that it is a std::string, not a Glib::ustring.
      openFile (filename);
      retval = 0;
      break;

    case (Gtk::RESPONSE_CANCEL):
      retval = 1;
      break;

    default:
      retval = 2;
      break;

  }

  return retval;
}

////////////////////////////////////////////////////////////////////////////////////
int MEditor::saveAsFileDialog( Glib::ustring& filename )
{
  int retval = 0;
  // Open a file dialog
  Gtk::FileChooserDialog dialog("Please choose a file to save as...", Gtk::FILE_CHOOSER_ACTION_SAVE);
  //dialog.set_transient_for(*this);

  //Add response buttons the the dialog:
  dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);

  //Add filters, so that only certain file types can be selected:
  Gtk::FileFilter filter_m;
  filter_m.set_name("m files");
  filter_m.add_pattern("*.m");
  dialog.add_filter(filter_m);

  Gtk::FileFilter filter_any;
  filter_any.set_name("Any files");
  filter_any.add_pattern("*");
  dialog.add_filter(filter_any);

  //Show the dialog and wait for a user response:
  int result = dialog.run();

  //Handle the response:
	//  std::string filename;
  switch (result)
  {
    case(Gtk::RESPONSE_OK):
      filename = dialog.get_filename(); //Notice that it is a std::string, not a Glib::ustring.
      //openFile( filename );
      retval = 0;
      break;

    case(Gtk::RESPONSE_CANCEL):
      retval = 1;
      break;

    default:
      retval = 2;
      break;

  }

  return retval;
}


////////////////////////////////////////////////////////////////////////////////////
bool doesFileExist (string filename)
{
	struct stat statVar;
	int rv = stat(filename.c_str(), &statVar);
	if (rv<0)
		return false;
	else
		return true;
}

////////////////////////////////////////////////////////////////////////////////////
void MEditor::onNew(void)
{
  char windowName[80];
  sprintf(windowName, "Untitled%d.m", idx);

	MEditorPanel *panel = new MEditorPanel(dockPane,windowName);
	dockPane.addItem(*panel,DockItem::CENTER);
	idx++;

	// TODO: add back in all the stuff for managing the buffers and how they are attached to each other
	Gtk::TextView* view = panel->getView();
	m_editorPanels[view] = panel;

	this->show_all();

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onOpen(void)
{
  if (openFileDialog() == 0)
	  show_all();
}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onSaveAs(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onSaveAs();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onSave(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onSave();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onQuit(void)
{
  // Just hide in a real app
  this->hide();
}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onUndo(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onUndo();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onRedo(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onRedo();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onCut(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onCut();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onCopy(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onCopy();
	}

}
  

////////////////////////////////////////////////////////////////////////////////////
void MEditor::onPaste(void)
{
	Gtk::TextView* view = (Gtk::TextView*)this->get_focus();
	if (view != NULL && m_editorPanels.find(view)!=m_editorPanels.end() )
	{
		MEditorPanel* panel = m_editorPanels[view];
		panel->onPaste();
	}

}


////////////////////////////////////////////////////////////////////////////////////
void MEditor::onPreferences(void)
{
	// Open up a dialog to mess around with fonts, tab styles, emacs keybindings, and such.
}


////////////////////////////////////////////////////////////////////////////////////
/*static*/ void MEditor::loadDebugPixmaps (Gtk::Widget* newBufView)
{
	GError *error;
	GdkPixbuf *pixbuf;
	int i = 0;
	const gchar * const*sys_data_dir = g_get_system_data_dirs();

	error = NULL;

	// load from the datadir location as passed by the compiler
	gchar* path = g_build_filename(DATA_DIRECTORY,"octavede","BP.png",NULL);
	if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
	{
		gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, pixbuf);
		gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, 1);
		g_object_unref (pixbuf);
		
		//			cout << "MARKER 1 loaded" << endl;
	}
	else
	{
		// try to find the images in the default system data directories
		sys_data_dir = g_get_system_data_dirs();
		i = 0;
		while (sys_data_dir[i]!=NULL)
		{
			gchar* path = g_build_filename(sys_data_dir[i],"octavede","BP.png",NULL);

			if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
			{
				gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, pixbuf);
				gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, 1);
				g_object_unref (pixbuf);

				//			cout << "MARKER 1 loaded" << endl;
				break;
			}
			
			i++;
		}

		// if all other methods fail, try to find the images in the local directory
		if (sys_data_dir[i]==NULL)
		{
			gchar* path = g_build_filename("./","BP.png",NULL);
			if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
			{
				gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, pixbuf);
				gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_1, 1);
				g_object_unref (pixbuf);
			}
		}
	}

	error = NULL;
	i = 0;

	// load from the datadir location as passed by the compiler
	path = g_build_filename(DATA_DIRECTORY,"octavede","STOPPED.png",NULL);
	if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
	{
		gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, pixbuf);
		gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, 2);
		g_object_unref (pixbuf);
		
		//			cout << "MARKER 1 loaded" << endl;
	}
	else
	{
		while (sys_data_dir[i]!=NULL)
		{
			path = g_build_filename(sys_data_dir[i],"octavede","STOPPED.png",NULL);
			
			if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
			{
				gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, pixbuf);
				gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, 2);
				g_object_unref (pixbuf);
				
				//			cout << "MARKER 2 loaded" << endl;
				break;
			}

			i++;
		}

		if (sys_data_dir[i]==NULL)
		{
			gchar* path = g_build_filename("./","STOPPED.png",NULL);
			if (doesFileExist(path) && (pixbuf = gdk_pixbuf_new_from_file (path, &error)))
			{
				gtk_source_view_set_mark_category_pixbuf (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, pixbuf);
				gtk_source_view_set_mark_category_priority (GTK_SOURCE_VIEW (newBufView->gobj()), MARK_TYPE_2, 2);
				g_object_unref (pixbuf);
			}
		}
		
	}
}

void MEditor::remove_all_breakpoints ()
{
	std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
	for (it = m_editorPanels.begin() ; it != m_editorPanels.end() ; it++)
  {
		it->second->remove_all_markers();
	}
}

void MEditor::remove_hit_breakpoint_marker ()
{
	std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
	for (it = m_editorPanels.begin() ; it != m_editorPanels.end() ; it++)
  {
		it->second->remove_all_markers(BP_MARKER_TYPE_HIT);
	}
}

int MEditor::process_breakpoint_list (std::vector<bp_info_t>& bps)
{
	// Add breakpoints that aren't currently in the buffer
	for (int i = 0 ; i < bps.size() ; i++)
  {
		int isCurrentBp = false;
		std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
		for (it = m_editorPanels.begin() ; it != m_editorPanels.end() ; it++)
    {
			isCurrentBp = (isCurrentBp || it->second->has_breakpoint_marker(bps[i]));
		}
		if (!isCurrentBp)
    {
			//std::cout << "Adding a breakpoint" << std::endl;
			add_breakpoint_marker (bps[i]);
		}
	}
	
	// Remove breakpoints that are no longer part of bps
	std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
	for (it = m_editorPanels.begin() ; it != m_editorPanels.end() ; it++)
  {
		std::vector<bp_info_t> markers = it->second->get_breakpoint_markers();
		//std:cout << "Buffer: " << it->second->getFuncName() << "   Count: " << markers.size() << std::endl;
		
		for (int i = 0 ; i < markers.size() ; i++)
    {
			bool isMarkerCurrent = false;
			for (int j = 0 ; j < bps.size() ; j++)
			{
				isMarkerCurrent = (isMarkerCurrent || 
													 (bps[j].filename.compare (markers[i].filename)==0 && bps[j].line_number==markers[i].line_number));
			}
			if (!isMarkerCurrent)
			{
				//std::cout << "Removing a breakpoint" << std::endl;
				it->second->remove_breakpoint_marker (markers[i]);
			}
		}
		
	}
	
	return 0;
}

int MEditor::add_breakpoint_marker (bp_info_t bp, bp_marker_type_t type)
{
	// Search through the panels to find the right name
	std::map <Gtk::TextView*, MEditorPanel*>::iterator it;
	for (it = m_editorPanels.begin() ; it != m_editorPanels.end() ; it++)
  {
		string shortName = it->second->getShortName();
		std::string funcName;
		int dot = shortName.find_first_of(".");
		if (dot!=string::npos)
    {
			funcName = shortName.substr (0,dot);
		}
		
		if (bp.filename.compare(funcName)==0)
    {
			it->second->add_breakpoint_marker (bp.line_number,type);
		}
	}
	
	return 0;
}
