/*
-----------------------------------------------------------------------------
This source file is part of OGRE
    (Object-oriented Graphics Rendering Engine)
For the latest info, see http://www.ogre3d.org/

Copyright (c) 2000-2005 The OGRE Team
Also see acknowledgements in Readme.html

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General  License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General  License for more details.

You should have received a copy of the GNU Lesser General  License along with
this program; if not, write to the Free Software Foundation, Inc., 59 Temple
Place - Suite 330, Boston, MA 02111-1307, USA, or go to
http://www.gnu.org/copyleft/lesser.txt.
-----------------------------------------------------------------------------
*/
/***************************************************************************
OgreMouseEvent.h  -  
 * An event which indicates that a mouse action occurred in a MouseTarget (e.g. MouseTarget).
 * This event is used both for mouse events (click, enter, exit) and mouse 
 * motion events (moves and drags). 
 * <P>
 * This low-level event is generated by a MouseTarget object for:
 * <ul>
 * <li>Mouse Events
 *     <ul>
 *     <li>a mouse button is pressed
 *     <li>a mouse button is released
 *     <li>a mouse button is clicked (pressed and released)
 *     <li>the mouse cursor enters a MouseTarget
 *     <li>the mouse cursor exits a MouseTarget
 *     </ul>
 * <li> Mouse Motion Events
 *     <ul>
 *     <li>the mouse is moved
 *     <li>the mouse is dragged
 *     </ul>
 * </ul>
 * <P>
 * A MouseEvent object is passed to every MouseListener
 * object which registered to receive 
 * the "interesting" mouse events using MouseTarget's 
 * <code>addMouseListener</code> method.
 *
 * A MouseEvent object is also passed to every MouseMotionListener
 * object which registered to receive 
 * mouse motion events using the MouseTarget's addMouseMotionListener
 * method
 *
 * When a mouse button is clicked, events are generated and sent to the
 * registered MouseListeners, with the button mask set in the modifier field.
 * For example, if the first mouse button is pressed, events are sent in the
 * following order:
 * <PRE>
 *    MOUSE_PRESSED:  BUTTON1_MASK
 *    MOUSE_RELEASED: BUTTON1_MASK
 *    MOUSE_CLICKED:  BUTTON1_MASK
 * </PRE>
 * When multiple mouse buttons are pressed, each press, release, and click
 * results in a separate event. The button mask in the modifier field reflects
 * only the button that changed state, not the current state of all buttons.
 * <P> 
 * For example, if the user presses button 1 followed by button 2 and
 * releases them in the same order, the following sequence of events is
 * generated:
 * <PRE>
 *    MOUSE_PRESSED:  BUTTON1_MASK
 *    MOUSE_PRESSED:  BUTTON2_MASK
 *    MOUSE_RELEASED: BUTTON1_MASK
 *    MOUSE_CLICKED:  BUTTON1_MASK
 *    MOUSE_RELEASED: BUTTON2_MASK
 *    MOUSE_CLICKED:  BUTTON2_MASK
 * </PRE>
 * If button2 is released first, the MOUSE_RELEASED/MOUSE_CLICKED pair
 * for BUTTON2_MASK arrives first, followed by the pair for BUTTON1_MASK.
 *
-------------------
begin                : Dec 03 2002
copyright            : (C) 2002 by Kenny Sabir
email                : kenny@sparksuit.com
***************************************************************************/
#ifndef __MouseEvent_H__
#define __MouseEvent_H__

#include "OgrePrerequisites.h"
#include "OgreInputEvent.h"

namespace Ogre {

/** An event which indicates that a mouse action occurred in a MouseTarget (e.g. MouseTarget).
@remarks
 * This event is used both for mouse events (click, enter, exit) and mouse 
 * motion events (moves and drags). 
 * <P>
 * This low-level event is generated by a MouseTarget object for:
 * <ul>
 * <li>Mouse Events
 *     <ul>
 *     <li>a mouse button is pressed
 *     <li>a mouse button is released
 *     <li>a mouse button is clicked (pressed and released)
 *     <li>the mouse cursor enters a MouseTarget
 *     <li>the mouse cursor exits a MouseTarget
 *     </ul>
 * <li> Mouse Motion Events
 *     <ul>
 *     <li>the mouse is moved
 *     <li>the mouse is dragged
 *     </ul>
 * </ul>
 * <P>
 * A MouseEvent object is passed to every MouseListener
 * object which registered to receive 
 * the "interesting" mouse events using MouseTarget's 
 * <code>addMouseListener</code> method.
 *
 * A MouseEvent object is also passed to every MouseMotionListener
 * object which registered to receive 
 * mouse motion events using the MouseTarget's addMouseMotionListener
 * method
 *
 * When a mouse button is clicked, events are generated and sent to the
 * registered MouseListeners, with the button mask set in the modifier field.
 * For example, if the first mouse button is pressed, events are sent in the
 * following order:
 * <PRE>
 *    MOUSE_PRESSED:  BUTTON1_MASK
 *    MOUSE_RELEASED: BUTTON1_MASK
 *    MOUSE_CLICKED:  BUTTON1_MASK
 * </PRE>
 * When multiple mouse buttons are pressed, each press, release, and click
 * results in a separate event. The button mask in the modifier field reflects
 * only the button that changed state, not the current state of all buttons.
 * <P> 
 * For example, if the user presses button 1 followed by button 2 and
 * releases them in the same order, the following sequence of events is
 * generated:
 * <PRE>
 *    MOUSE_PRESSED:  BUTTON1_MASK
 *    MOUSE_PRESSED:  BUTTON2_MASK
 *    MOUSE_RELEASED: BUTTON1_MASK
 *    MOUSE_CLICKED:  BUTTON1_MASK
 *    MOUSE_RELEASED: BUTTON2_MASK
 *    MOUSE_CLICKED:  BUTTON2_MASK
 * </PRE>
 * If button2 is released first, the MOUSE_RELEASED/MOUSE_CLICKED pair
 * for BUTTON2_MASK arrives first, followed by the pair for BUTTON1_MASK.
 *
 */
    class _OgreExport MouseEvent : public InputEvent
    {
    protected:
		/**
		 * The mouse events x coordinate.
		 * The x value is relative to the MouseTarget
		 * that fired the event.
		 */
		Real mX;
		/**
		 * The mouse events y coordinate.
		 * The y value is relative to the MouseTarget
		 * that fired the event.
		 */
		Real mY;

		/**
		 * The mouse events z coordinate.
		 * The z value is relative to the MouseTarget
		 * that fired the event.
		 */
		Real mZ;


        Real mRelX;
        Real mRelY;
        Real mRelZ;

		/**
		 * Which button was pressed
		 */

		int mButtonID;
		/**
		 * not implemented yet
		 */
		int mClickCount;




	public:

		enum
		{
			ME_FIRST_EVENT = 500,
			ME_LAST_EVENT = 510
		};

		enum 
		{
			ME_MOUSE_CLICKED 	= ME_FIRST_EVENT,
			ME_MOUSE_PRESSED,
			ME_MOUSE_RELEASED,
			ME_MOUSE_MOVED,
			ME_MOUSE_ENTERED,
			ME_MOUSE_EXITED,
			ME_MOUSE_DRAGGED,

      ME_MOUSE_DRAGENTERED,
      ME_MOUSE_DRAGEXITED,
      ME_MOUSE_DRAGMOVED,
      ME_MOUSE_DRAGDROPPED
		};

		/**
		 * Constructs a MouseEvent object with the specified source MouseTarget,
		 * type, modifiers, coordinates, and click count.
		 *
		 * @param source       the MouseTarget that originated the event
		 * @param id           the integer that identifies the event
		 * @param when         a long int that gives the time the event occurred
		 * @param modifiers    the modifier keys down during event
		 *                     (shift, ctrl, alt, meta)
		 * @param x            the horizontal x coordinate for the mouse location
		 * @param y            the vertical y coordinate for the mouse location
		 * @param clickCount   the number of mouse clicks associated with event
		 */
		MouseEvent(PositionTarget* source, int id, int whichButton, Real when, int modifiers,
			Real x, Real y, Real z, int clickCount);

        MouseEvent(PositionTarget* source, int id, int whichButton, Real when, int modifiers,
			Real x, Real y, Real z, Real relx, Real rely, Real relz, int clickCount);
		
         /**
		 * Return the number of mouse clicks associated with this event.
		 *
		 * @return integer value for the number of clicks - NOT IMPLEMENTED
		 */
		 int getClickCount();

		/**
		 * Returns the horizontal x position of the event relative to the 
		 * source MouseTarget.
		 *
		 * @return x  an integer indicating horizontal position relative to
		 *            the MouseTarget
		 */
		 Real getX() const;

		/**
		 * Returns the vertical y position of the event relative to the
		 * source MouseTarget.
		 *
		 * @return y  an integer indicating vertical position relative to
		 *            the MouseTarget
		 */
		 Real getY() const;

		/**
		 * Returns the scrollwheel z position of the event relative to the
		 * source MouseTarget.
		 *
		 * @return y  an integer indicating scrollwheel position relative to
		 *            the MouseTarget
		 */
		 Real getZ() const;

         /** get relative X cursor movement */
         Real getRelX() const {return mRelX;}
         
         /** get relative Y cursor movement */
         Real getRelY() const {return mRelY;}
         
         /** get relative Z cursor movement */
         Real getRelZ() const {return mRelZ;}

		/**
		 * Returns a parameter string identifying this event.
		 * This method is useful for event-logging and for debugging.
		 *
		 * @return a string identifying the event and its attributes
		 */
		 String paramString() const;

		/**
		 * Translates the event's coordinates to a new position
		 * by adding specified x (horizontal) and y (veritcal) offsets.
		 *
		 * @param x the horizontal x value to add to the current x coordinate position
		 * @param y the vertical y value to add to the current y coordinate position
		 */
		 void translatePoint(Real x, Real y);

		 /** return the ID of the button */
		 int getButtonID() const;
    };


}


#endif 

