/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PORT_H
#define PORT_H

#include <string>
#include <list>
#include <libgnomecanvasmm.h>

using std::string; using std::list;

namespace PatchBay {
	
class Connection;
class Module;
class PatchBayArea;


const int PORT_LABEL_SIZE = 8000; // in thousandths of a point


/** A port on a module on the canvas.
 *
 * This is a group that contains both the label and rectangle for a port.
 *
 * \ingroup Canvas
 */
class Port : public Gnome::Canvas::Group
{
public:
	Port(Module* module, const string& name, bool is_input, int color);
	
	virtual ~Port() {};
	
	const string&        name() const        { return m_name; }
	Module*              module() const      { return m_module; }
	Gnome::Canvas::Rect* rect()              { return &m_rect; }
	Gnome::Canvas::Text* label()             { return &m_label; }
	
	virtual void name(const string& n);

	list<Connection*>& connections() { return m_connections; }
	
	void add_connection(Connection* c) { m_connections.push_back(c); }
	void remove_connection(Connection* c);
	
	void move_connections();
	void hilite(bool b);
	void raise_connections();

	void popup_menu(guint button, guint32 activate_time) {
		m_menu.popup(button, activate_time);
	}

	Gnome::Art::Point connection_coords();

	bool   is_input() const  { return m_is_input; }
	bool   is_output() const { return !m_is_input; }
	double width() const     { return m_width; }
	void   width(double w);
	double height() const    { return m_height; }
	int    color() const     { return m_color; }
	
	void zoom(float z);

protected:
	PatchBayArea* m_patch_bay;
	Module*       m_module;
	string        m_name;
	bool          m_is_input;
	double        m_width;
	double        m_height;
	int           m_color;
	
	list<Connection*> m_connections; // needed for dragging
	
	Gnome::Canvas::Text m_label;
	Gnome::Canvas::Rect m_rect;
	Gtk::Menu           m_menu;
};


typedef list<Port*> PortList;

} // namespace PatchBay

#endif // PORT_H
