/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "MidiLearnEvent.h"
#include <string>
#include <cmath>
#include "Om.h"
#include "OSCSender.h"
#include "MidiControlNode.h"
#include "Port.h"
#include "Patch.h"
#include "ObjectStore.h"

using std::string;

namespace Om {


MidiLearnEvent::MidiLearnEvent(Request* request, const string& node_path)
: SlowEvent(request),
  m_node_path(node_path),
  m_patch(NULL),
  m_node(NULL),
  m_value(0.0f),
  m_succeeded(false)
{
}


void
MidiLearnEvent::prepare()
{
	Node* node = om->object_store()->find_node(m_node_path);

	if (node != NULL && node->plugin()->type() == Plugin::Internal
			&& node->plugin()->plug_label() == "midi_control_in") {
		m_node = (MidiControlNode*)node;
		m_patch = m_node->parent_patch();
		m_succeeded = true;
	}

	SlowEvent::prepare();
}


void
MidiLearnEvent::execute(uint sample_offset)
{
	SlowEvent::execute(sample_offset);
}


void MidiLearnEvent::set_control(const snd_seq_ev_ctrl_t& ctrl)
{
	if (m_node != NULL) {
		m_node->m_param_port->set_value(0, ctrl.param);
		m_value = ctrl.param;
	}
}


void
MidiLearnEvent::post_process()
{
	string msg;
	if (m_node == NULL) {
		m_request->respond_error("Unable to find MIDI control node for MIDI learn.");
	} else if (!m_succeeded) {
		m_request->respond_error("Invalid node type specified for MIDI learn.");
	} else {
		m_request->respond_ok();
		om->osc_sender()->send_control_change(m_node->m_param_port->path(), m_value);
	}
}


} // namespace Om

