/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "MidiTriggerNode.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Plugin.h"
#include "util.h"

namespace Om {


MidiTriggerNode::MidiTriggerNode(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: MidiInNode(path, 1, parent, srate, buffer_size)
{
	m_num_ports = 4;
	m_ports.alloc(m_num_ports);

	// Add note-selection control in port
	InputPort* note_port = new InputPort(this, "Note Number", 0, 1,
		new PortInfo("Note Number", CONTROL, INPUT, INTEGER, 60, 0, 127), 1);
	m_ports.at(0) = note_port;
	
	// Add gate out port
	OutputPort* gate_port = new OutputPort(this, "Gate", 1, 1,
		new PortInfo("Gate", AUDIO, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(1) = gate_port;

	// Add trigger out port
	OutputPort* trigger_port = new OutputPort(this, "Trigger", 2, 1,
		new PortInfo("Trigger", AUDIO, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(2) = trigger_port;
	
	OutputPort* velocity_port = new OutputPort(this, "Velocity", 1, poly,
		new PortInfo("Velocity", AUDIO, OUTPUT, 60, 0, 127), m_buffer_size);
	m_ports.at(3) = velocity_port;
	
	m_plugin.type(Plugin::Internal);
	m_plugin.lib_path("");
	m_plugin.lib_name("");
	m_plugin.plug_label("trigger_in");
	m_plugin.name("Om patch trigger input node (MIDI/OSC)");
}


void
MidiTriggerNode::note_on(int note_num, int velocity, int start_sample)
{
	//std::cerr << "Note on starting at sample " << start_sample << std::endl;
	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	if (note_num == (int)(*(m_ports.at(0)->buffer(0)))) {
		// See comments in MidiNoteNode::note_on
		if (start_sample == (int)(m_buffer_size-1))
			--start_sample;
		
		m_ports.at(1)->set_value(0, start_sample, 1.0f);
		m_ports.at(2)->buffer(0)[start_sample] = 1.0f;
		m_ports.at(2)->set_value(0, start_sample+1, 0.0f); // this takes care of it
		m_ports.at(3)->set_value(0, start_sample, velocity/127.0);
	}
}


void
MidiTriggerNode::note_off(int note_num, int start_sample)
{
	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	if (note_num == (int)(*(m_ports.at(0)->buffer(0)))) {
		m_ports.at(1)->set_value(0, start_sample, 0.0f);
	}
}


void
MidiTriggerNode::run(size_t nframes)
{
	NodeBase::run(nframes);
}


} // namespace Om

