(*
 * Cache information about past runs.
 *
 * ----------------------------------------------------------------
 *
 * @begin[license]
 * Copyright (C) 2003 Jason Hickey, Caltech
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Jason Hickey
 * @email{jyh@cs.caltech.edu}
 * @end[license]
 *)
open Lm_printf
open Lm_string_set

open Omake_node
open Omake_cache_type
open Omake_command_type

(* Debugging *)
val debug_cache : bool ref

(* The abstract type for the cache database *)
type t

(* Keys for various functions *)
type key

(* Directory listing *)
type dir_listing
type exe_listing

(* Manifest keys *)
val scanner_fun         : key
val rule_fun            : key
val env_fun             : key
val include_fun         : key

val env_target  : Node.t

(* Fetch the cache *)
val create       : unit -> t
val from_channel : in_channel -> t
val to_channel   : Pervasives.out_channel -> t -> unit

(*
 * Stats.
 *)
val stats : t -> int * int

(*
 * Clear an entry.
 *)
val clear : t -> key -> unit

(*
 * File digest information.
 * The reset functions indicate that stat info may need to be recomputed.
 * The stat functions get the digest.
 *)
val stat             : t -> Node.t -> digest
val stat_set         : t -> NodeSet.t -> digest NodeTable.t
val stat_table       : t -> 'b NodeTable.t -> digest NodeTable.t
val stat_unix        : t -> Node.t -> Unix.LargeFile.stats
val is_dir           : t -> Node.t -> bool

val reset            : t -> Node.t -> unit
val reset_set        : t -> NodeSet.t -> unit
val reset_table      : t -> 'b NodeTable.t -> unit

val force_stat       : t -> Node.t -> digest
val force_stat_set   : t -> NodeSet.t -> digest NodeTable.t
val force_stat_table : t -> 'b NodeTable.t -> digest NodeTable.t

val stat_changed     : t -> Node.t -> bool

(*
 * Check if a file exists.
 *)
val exists           : t -> Node.t -> bool
val exists_dir       : t -> Dir.t -> bool

(*
 * Directory listings.
 *)
val rehash           : t -> unit

val ls_dir           : t -> Dir.t -> dir_listing
val ls_path          : t -> Dir.t list -> dir_listing
val listing_find     : t -> dir_listing -> string -> dir_entry

val ls_exe_path      : t -> Dir.t list -> exe_listing
val exe_find         : t -> exe_listing -> string -> Node.t

(*
 * Memoizing commands.
 *
 * add                : key targets sources commands result
 * up_to_date         : key sources commands
 * find_result        : key sources commands
 * find_result_sloppy : key target
 *
 * The sloppy function just returns the results for the target
 * without even considering the command and dependencies.
 *)
val add                : t -> key -> Node.t -> NodeSet.t -> NodeSet.t -> command_digest -> memo_result -> unit
val up_to_date         : t -> key -> NodeSet.t -> command_digest -> bool
val up_to_date_status  : t -> key -> NodeSet.t -> command_digest -> memo_status
val find_result        : t -> key -> NodeSet.t -> command_digest -> NodeSet.t NodeTable.t
val find_result_sloppy : t -> key -> Node.t -> NodeSet.t NodeTable.t

(*
 * Printing.
 *)
val pp_print_digest            : formatter -> digest -> unit
val pp_print_node_digest_table : formatter -> digest NodeTable.t -> unit
val pp_print_memo_result       : (formatter -> NodeSet.t NodeTable.t -> unit) -> formatter -> memo_result -> unit

(*!
 * @docoff
 *
 * -*-
 * Local Variables:
 * Caml-master: "compile"
 * End:
 * -*-
 *)
