# -*- coding: utf-8 -*-

# Copyright (C) 2004-2005 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from gettext import gettext as _

import gtk
import gconf
import gobject

import Configuration
from GUI import GUI, FileChooserDialog
from ontv import NAME, GLADE_DIR, LOCALE_DIR

class PreferencesDialog(GUI):
    def __init__(self, config, xmltvfile, pw):
        GUI.__init__(self)
        self.config = config
        self.xmltvfile = xmltvfile
        self.pw = pw

        self.xmltvfile.connect("loaded", self.__xmltvfile_loaded)

        self.__get_widgets()
        self.__connect_widgets()
        self.dialog.set_icon(self.icon)
        self.__make_channels_treeview()
        self.__add_config_notifications()

        self.__auto_reload_id = None

    def __get_widgets(self):
        self.dialog = self.xml.get_widget("preferences_dialog")
        self.closebutton = self.xml.get_widget("closebutton")
        self.notebook = self.xml.get_widget("notebook")
        self.xmltv_file_entry = self.xml.get_widget("xmltv_file_entry")
        self.xmltv_browse_button = self.xml.get_widget("xmltv_browse_button")
        self.reload_xmltv_file_checkbutton = self.xml.get_widget("reload_xmltv_file_checkbutton")
        self.reload_xmltv_file_spinbutton = self.xml.get_widget("reload_xmltv_file_spinbutton")
        self.current_programs_checkbutton = self.xml.get_widget("current_programs_checkbutton")
        self.upcoming_programs_checkbutton = self.xml.get_widget("upcoming_programs_checkbutton")
        self.channels_treeview = self.xml.get_widget("channels_treeview")

    def __connect_widgets(self):
        self.closebutton.connect("clicked", self.__closebutton_clicked)
        self.xmltv_file_entry.connect("activate", self.__xmltv_file_entry_activate)
        self.xmltv_file_entry.connect("changed", self.__xmltv_file_entry_changed)
        self.xmltv_browse_button.connect("clicked", self.__xmltv_browse_button_clicked)
        self.reload_xmltv_file_checkbutton.connect("toggled", self.__reload_xmltv_file_checkbutton_toggled)
        self.reload_xmltv_file_spinbutton.connect("value_changed", self.__reload_xmltv_file_spinbutton_value_changed)
        self.current_programs_checkbutton.connect("toggled", self.__current_programs_checkbutton_toggled)
        self.upcoming_programs_checkbutton.connect("toggled", self.__upcoming_programs_checkbutton_toggled)

    def __make_channels_treeview(self):
        ls = gtk.ListStore(object)
        ls.set_sort_func(0, self.__ls_sort_func)
        ls.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.channels_treeview.set_model(ls)

        crtoggle = gtk.CellRendererToggle()
        crtoggle.props.xpad = 6
        crtoggle.props.ypad = 3
        crtoggle.props.activatable = True
        crtoggle.connect("toggled", self.__channels_treeview_toggled, ls)
        selected_column = gtk.TreeViewColumn("Selected", crtoggle)
        selected_column.set_cell_data_func(crtoggle,
                                           self.__crtoggle_cell_data_func)
        self.channels_treeview.append_column(selected_column)

        crpixbuf = gtk.CellRendererPixbuf()
        crpixbuf.props.xpad = 3
        crpixbuf.props.ypad = 3
        logo_column = gtk.TreeViewColumn("Logo", crpixbuf)
        logo_column.set_cell_data_func(crpixbuf, self.__crpixbuf_cell_data_func)
        self.channels_treeview.append_column(logo_column)

        crtext = gtk.CellRendererText()
        crtext.props.xpad = 3
        crtext.props.ypad = 3
        channel_column = gtk.TreeViewColumn("Channel", crtext)
        channel_column.set_cell_data_func(crtext, self.__crtext_cell_data_func)
        self.channels_treeview.append_column(channel_column)
        self.channels_treeview.set_search_column(2)
        self.channels_treeview.set_search_equal_func(self.__channels_treeview_search_equal)

        channels_treeview_menu = gtk.Menu()

        self.add_logo_imi = gtk.ImageMenuItem(gtk.STOCK_ADD)
        self.add_logo_imi.connect("activate", self.__add_logo_imi_activate)
        self.add_logo_imi.child.set_markup_with_mnemonic("_Add logo")
        self.add_logo_imi.show()
        channels_treeview_menu.append(self.add_logo_imi)

        self.remove_logo_imi = gtk.ImageMenuItem(gtk.STOCK_REMOVE)
        self.remove_logo_imi.connect("activate", self.__remove_logo_imi_activate)
        self.remove_logo_imi.child.set_markup_with_mnemonic("_Remove logo")
        self.remove_logo_imi.show()
        channels_treeview_menu.append(self.remove_logo_imi)

        self.channels_treeview.connect("row-activated",
                                       self.__channels_treeview_row_activated)
        self.channels_treeview.connect("button-press-event",
                                       self.__channels_treeview_button_press,
                                       channels_treeview_menu)

    def __add_config_notifications(self):
        self.config.add_notify(Configuration.KEY_XMLTV_FILE,
                               self.__xmltv_file_key_changed)
        self.config.add_notify(Configuration.KEY_AUTO_RELOAD_XMLTV_FILE,
                               self.__auto_reload_xmltv_file_key_changed)
        self.config.add_notify(Configuration.KEY_AUTO_RELOAD_XMLTV_FILE_RATE,
                               self.__auto_reload_xmltv_file_rate_key_changed)
        self.config.add_notify(Configuration.KEY_DISPLAY_CURRENT_PROGRAMS,
                               self.__display_current_programs_key_changed)
        self.config.add_notify(Configuration.KEY_DISPLAY_UPCOMING_PROGRAMS,
                               self.__display_upcoming_programs_key_changed)

    def __xmltvfile_loaded(self, xmltvfile, listings):
        if listings:
            self.listings = listings
            model = self.channels_treeview.get_model()
            model.clear()
            for channel in self.listings.channels.values():
                model.append([channel])

    def show(self, uicomponent=None, verb=None):
        self.xmltv_file_entry.set_text(self.config.xmltv_file)
        self.reload_xmltv_file_checkbutton.set_active(self.config.auto_reload_xmltv_file)
        self.reload_xmltv_file_spinbutton.set_value(self.config.auto_reload_xmltv_file_rate)
        self.current_programs_checkbutton.set_active(self.config.display_current_programs)
        self.upcoming_programs_checkbutton.set_active(self.config.display_upcoming_programs)
        self.dialog.show()

    def __closebutton_clicked(self, button):
        self.dialog.hide()

    def __xmltv_file_entry_changed(self, entry):
        if len(entry.get_text()) > 0:
            self.reload_xmltv_file_checkbutton.set_sensitive(True)
            self.current_programs_checkbutton.set_sensitive(True)
            self.upcoming_programs_checkbutton.set_sensitive(True)
        else:
            self.reload_xmltv_file_checkbutton.set_sensitive(False)
            self.current_programs_checkbutton.set_sensitive(False)
            self.upcoming_programs_checkbutton.set_sensitive(False)

    def __xmltv_file_entry_activate(self, entry):
        self.config.xmltv_file = entry.get_text()

    def __xmltv_browse_button_clicked(self, button):
        xml_filter = gtk.FileFilter()
        xml_filter.set_name(_("XML files"))
        xml_filter.add_mime_type("text/xml")
        fcd = FileChooserDialog(_("Select XMLTV file..."), (xml_filter,))
        response = fcd.run()
        if response == gtk.RESPONSE_OK:
            self.config.xmltv_file = fcd.get_filename()
            fcd.hide()
        elif response == gtk.RESPONSE_CANCEL:
            fcd.hide()
        fcd.hide()

    def __reload_xmltv_file_checkbutton_toggled(self, checkbutton):
        self.config.auto_reload_xmltv_file = checkbutton.get_active()
        if checkbutton.get_active():
            self.reload_xmltv_file_spinbutton.set_sensitive(True)
        else:
            self.reload_xmltv_file_spinbutton.set_sensitive(False)

    def __reload_xmltv_file_spinbutton_value_changed(self, spinbutton):
        self.config.auto_reload_xmltv_file_rate = spinbutton.get_value_as_int()

    def __current_programs_checkbutton_toggled(self, checkbutton):
        self.config.display_current_programs = checkbutton.get_active()

    def __upcoming_programs_checkbutton_toggled(self, checkbutton):
        self.config.display_upcoming_programs = checkbutton.get_active()

    def __ls_sort_func(self, model, iter1, iter2):
        channel = model.get_value(iter1, 0)
        other_channel = model.get_value(iter2, 0)
        return cmp(channel.name.lower(), other_channel.name.lower())

    def __crtoggle_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        cell.props.active = channel.selected

    def __crpixbuf_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        cell.props.pixbuf = channel.logo

    def __crtext_cell_data_func(self, column, cell, model, iter):
        channel = model.get_value(iter, 0)
        markup = "<b>%s</b>"
        cell.props.markup = markup % channel.name

    def __add_logo_imi_activate(self, menuitem):
        selection = self.channels_treeview.get_selection()
        (model, iter) = selection.get_selected()
        path = model.get_path(iter)
        channel = model.get_value(iter, 0)

        png_filter = gtk.FileFilter()
        png_filter.set_name(_("PNG images"))
        png_filter.add_mime_type("image/png")

        jpeg_filter = gtk.FileFilter()
        jpeg_filter.set_name(_("JPEG images"))
        jpeg_filter.add_mime_type("image/jpeg")

        gif_filter = gtk.FileFilter()
        gif_filter.set_name(_("GIF images"))
        gif_filter.add_mime_type("image/gif")

        fcd = FileChooserDialog(_("Select logo image..."),
                                (png_filter, gif_filter, jpeg_filter))

        response = fcd.run()
        if response == gtk.RESPONSE_OK:
            channel.logo_file = fcd.get_filename()
            channel.set_logo(channel.logo_file)
            model.row_changed(path, iter)
            self.pw.update()
            fcd.hide()
            self.listings.save(self.config.debug)
        elif response == gtk.RESPONSE_CANCEL:
            fcd.hide()
        fcd.hide()

    def __remove_logo_imi_activate(self, menuitem):
        selection = self.channels_treeview.get_selection()
        (model, iter) = selection.get_selected()
        path = model.get_path(iter)
        channel = model.get_value(iter, 0)
        channel.logo = channel.logo_small = None
        model.row_changed(path, iter)
        self.pw.update()
        self.listings.save(self.config.debug)

    def __channels_treeview_search_equal(self, model, column, key, iter):
        channel = model.get_value(iter, 0)
        return key.lower() not in channel.name.lower()

    def __channels_treeview_row_activated(self, treeview, path, column):
        model = treeview.get_model()
        self.__channels_treeview_toggled(None, path, model)

    def __channels_treeview_toggled(self, cell, path, model):
        channel = model[path][0]
        channel.selected = not channel.selected
        if channel.selected:
            self.listings.selected_channels.append(channel.name)
        else:
            self.listings.selected_channels.remove(channel.name)

        self.pw.update()
        self.listings.save(self.config.debug)

    def __channels_treeview_button_press(self, treeview, event, menu):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
            model = treeview.get_model()
            path = treeview.get_path_at_pos(int(event.x), int(event.y))[0]
            channel = model[path][0]
            if channel.logo:
                self.add_logo_imi.set_sensitive(False)
                self.remove_logo_imi.set_sensitive(True)
            else:
                self.add_logo_imi.set_sensitive(True)
                self.remove_logo_imi.set_sensitive(False)

            menu.popup(None, None, None, event.button, event.time)

    def start_auto_reload(self, rate):
        if rate >= 1:
            timeout = rate * 3600 * 1000
            self.__auto_reload_id = gobject.timeout_add(timeout, self.xmltvfile.load, (None, True))
            if self.config.debug:
                print("[Added timeout %s]: %sh" % (str(self.__auto_reload_id),
                                                   str(rate)))

    def stop_auto_reload(self):
        if self.__auto_reload_id:
            gobject.source_remove(self.__auto_reload_id)
            if self.config.debug:
                print("[Removed timeout %s]" % (str(self.__auto_reload_id)))

    def __xmltv_file_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.xmltv_file_entry.set_text("")
        elif entry.value.type == gconf.VALUE_STRING:
            self.xmltv_file_entry.set_text(entry.value.to_string())
            self.xmltvfile.props.path = entry.value.to_string()
        else:
            self.xmltv_file_entry.set_text("!type error!")

    def __auto_reload_xmltv_file_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.reload_xmltv_file_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            self.reload_xmltv_file_checkbutton.set_active(entry.value.get_bool())
            if entry.value.get_bool():
                self.start_auto_reload(self.config.auto_reload_xmltv_file_rate)
            else:
                self.stop_auto_reload()
        else:
            self.reload_xmltv_file_checkbutton.set_active(True)

    def __auto_reload_xmltv_file_rate_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.reload_xmltv_file_spinbutton.set_value(12)
        elif entry.value.type == gconf.VALUE_INT:
            value = entry.value.get_int()
            self.reload_xmltv_file_spinbutton.set_value(value)
            if self.config.auto_reload_xmltv_file:
                self.stop_auto_reload()
                self.start_auto_reload(value)
        else:
            self.reload_xmltv_file_spinbutton.set_value(12)

    def __display_current_programs_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.current_programs_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            value = entry.value.get_bool()
            self.current_programs_checkbutton.set_active(value)
            if value:
                self.pw.cpt.show()
            else:
                self.pw.cpt.hide()
        else:
            self.current_programs_checkbutton.set_active(True)

    def __display_upcoming_programs_key_changed(self, client, cnxn_id, entry, data):
        if not entry.value:
            self.upcoming_programs_checkbutton.set_active(True)
        elif entry.value.type == gconf.VALUE_BOOL:
            value = entry.value.get_bool()
            self.upcoming_programs_checkbutton.set_active(value)
            if value:
                self.pw.upt.show()
            else:
                self.pw.upt.hide()
        else:
            self.upcoming_programs_checkbutton.set_active(True)

# vim: set sw=4 et sts=4 tw=79 fo+=l:
