/** Tutorial Arrow
* Arrow to show important stuff to the player, can be created by using one of the global functions.
* @author Sven2, Maikel
*/

// Remove arrow if it was showing a target object which got removed.
protected func AttachTargetLost() { RemoveObject(); }
// Remove arrow if target object entered a container.
protected func Entrance() { RemoveObject(); }

/*-- Arrow Creation --*/

/** Removes all tutorial arrows.
*/
global func TutArrowClear()
{
	for (arrow in FindObjects(Find_ID(TutorialArrow)))
		arrow->RemoveObject();
	return;
}

/** Creates an arrow to indicate a position.
* @param x X-coordinate of the position.
* @param y Y-coordinate of the position.
* @param angle angle at which the arrow should be drawn, standard \c 135 degrees.
* @param dist distance of the arrow to the position, standard 16 pixels.
* @return the arrow created.
*/
global func TutArrowShowPos(int x, int y, int angle, int dist)
{
	if (angle == nil) 
		angle = 135;
	if (dist == nil) 
		dist = 16;
	var arrow = CreateObjectAbove(TutorialArrow, x, y, NO_OWNER);
	if (!arrow) 
		return;
	// Display bouncing arrow, corrected for arrow size.
	dist += 8;
	x -= Sin(angle, dist);
	y += Cos(angle, dist);
	arrow->SetAction("Show");
	arrow->SetPosition(x, y);
	arrow->SetR(angle);
	return arrow;
}

/** Creates an arrow to indicate the target.
* @param target target object which should be indicated by the arrow.
* @param angle angle at which the arrow should be drawn, standard \c 135 degrees.
* @param dist distance of the arrow to the target object, standard 16 pixels.
* @return the arrow created.
*/
global func TutArrowShowTarget(object target, int angle, int dist)
{
	var container = target->Contained(), index;
	if (container &&
	    container.Prototype == Clonk &&
		(index = GetIndexOf(container.inventory, target)) != -1) // Is the object inventory in a clonk?
	{
		var itemslot = container.HUDcontroller.inventory[index];
		return TutArrowShowGUIPos(itemslot->GetX(), itemslot->GetY(), -90, itemslot->GetDefHeight() / 2);
	}
	if (angle == nil) 
		angle = 135;
	if (dist == nil) 
		dist = 16;
	var arrow = CreateObjectAbove(TutorialArrow, target->GetX(), target->GetY(), NO_OWNER);
	if (!arrow) 
		return;
	// Display spinning arrow, corrected for arrow size.
	dist += 8;
	arrow->SetAction("Attach", target);
	arrow->SetR(angle);
	arrow->SetVertex(0, VTX_Y, -dist, VTX_SetPermanentUpd);
	return arrow;
}

/** Creates an arrow to indicate a GUI position.
* @param x X-coordinate of the GUI position.
* @param y Y-coordinate of the GUI position.
* @param angle angle at which the arrow should be drawn, standard \c 135 degrees.
* @param dist distance of the arrow to the position, standard 16 pixels.
* @return the arrow created.
*/
global func TutArrowShowGUIPos(int x, int y, int angle, int dist)
{
	if (angle == nil) 
		angle = 135;
	if (dist == nil) 
		dist = 16;
	var arrow = CreateObjectAbove(TutorialArrow, x, y, NO_OWNER);
	if (!arrow) 
		return;
	// Change arrow category to C4D_Gui.
	arrow->SetCategory(C4D_IgnoreFoW | C4D_Foreground | C4D_Parallax);
	// Display bouncing arrow, corrected for arrow size.
	dist += 8;
	x -= Sin(angle, dist);
	y += Cos(angle, dist);
	arrow->SetAction("Show");
	arrow->SetPosition(x, y);
	arrow->SetR(angle);
	return arrow;
}

/** Creates an arrow to indicate the target.
* @param target GUI object which should be indicated by the arrow.
* @param angle angle at which the arrow should be drawn, standard \c 135 degrees.
* @param dist distance of the arrow to the target object, automatically corrects for GUI object's size.
* @return the arrow created.
*/
global func TutArrowShowGUITarget(object target, int angle, int dist)
{
	if (angle == nil) 
		angle = 135;
	if (dist == nil) 
		dist = 16;
	var arrow = CreateObjectAbove(TutorialArrow, target->GetX(), target->GetY(), NO_OWNER);
	if (!arrow) 
		return;
	// Change arrow category to C4D_Gui.
	arrow->SetCategory(C4D_IgnoreFoW | C4D_Foreground | C4D_Parallax);
	// Display spinning arrow, corrected for GUI and arrow size.
	dist += 8 + target->GetID()->GetDefHeight() / 2;
	arrow->SetAction("Attach", target);
	arrow->SetR(angle);
	arrow->SetVertex(0, VTX_Y, -dist, VTX_SetPermanentUpd);
	return arrow;
}

/*-- Proplist --*/

local Name = "$Name$";
local MeshTransformation = [1400, 0, 0, 0, 0, 1400, 0, 0, 0, 0, 1400, 0];
local Parallaxity = [0, 0];
local ActMap = {
	Attach = {
		Prototype = Action,
		Name = "Attach",
		Procedure = DFA_ATTACH,
		Length = 20,
		Delay = 2,
		X = 0,
		Y = 0,
		Wdt = 32,
		Hgt = 32,
		NextAction = "Attach",
		Animation = "Spin",
	},
	Show = {
		Prototype = Action,
		Name = "Show",
		Procedure = DFA_FLOAT,
		Length = 20,
		Delay = 2,
		X = 0,
		Y = 0,
		Wdt = 32,
		Hgt = 32,
		NextAction = "Show",
		Animation = "Bounce",
	},
};
