/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Compiler.h"

// stdc++
#include <list>
#include <map>
#include <sstream>
#include <float.h>
#include <climits>

// LLVM
#include <llvm/InstrTypes.h>
// #include <llvm/Linker.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Module.h>
// #include <llvm/Transforms/Utils/Cloning.h>
// Library Loader
#include <llvm/System/DynamicLibrary.h>
// Passes
#include <llvm/PassManager.h>

#include "LexerNG.h"
#include "ParserNG.h"
#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"

#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ErrorMessage.h"

#include "OpenCTL/Debug.h"
#include "OpenCTL/Module.h"
#include "GTLCore/Function.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/Value.h"
#include "OpenCTL/ModulesManager.h"
#include "GTLCore/ScopedName.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/Optimiser_p.h"
#include "GTLCore/Macros_p.h"

using namespace OpenCTL;

struct Compiler::Private {
  Private() : module(0), moduleData(0), lexer(0), parser(0), codeGenerator(0) {}
  llvm::Module* module;
  GTLCore::ModuleData* moduleData;
  LexerNG* lexer;
  ParserNG* parser;
  int lineNumber;
  GTLCore::String errorMessage;
  std::list< GTLCore::String > importedModules;
  GTLCore::CodeGenerator* codeGenerator;
  static bool disableStandardLibrary;
};

bool Compiler::Private::disableStandardLibrary = false;

Compiler::Compiler() : d(new Private)
{
}

Compiler::~Compiler()
{
  delete d;
}

float floatPosInf ()
{
  union {float f; unsigned int i;} u;
  u.i = 0x7f800000;
  return u.f;
}


float floatNegInf ()
{
  union {float f; unsigned int i;} u;
  u.i = 0xff800000;
  return u.f;
}


float floatNaN ()
{
  union {float f; unsigned int i;} u;
  u.i = 0x7fffffff;
  return u.f;
}

bool Compiler::compile(bool _isStandardLibrary, const GTLCore::String& sourceCode, const GTLCore::String& moduleName, GTLCore::ModuleData* moduleData)
{
  OCTL_DEBUG("Compile: " << moduleName << " : " << sourceCode);
  // Clear some variables
  d->importedModules.clear();
  d->errorMessage = "";
  setModuleName( moduleName );
  
  // Create LLVM module
  d->module = moduleData->llvmModule();
  d->moduleData = moduleData;
  d->codeGenerator = new GTLCore::CodeGenerator( moduleData );
  setModuleData(d->moduleData);
  
  llvm::LLVMContext& context = moduleData->llvmModule()->getContext();
  
  // Create Standard Library functions
  // CtlStdLib functions (except print which get a special treatement)
  createStdLibFunction( context, "assert", "assert", GTLCore::Type::Void, 1, GTLCore::Type::Boolean, false);
  createStdLibFunction( context, "isnan_f", "isnan_f", GTLCore::Type::Boolean, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "isnan_h", "isnan_h", GTLCore::Type::Boolean, 1, GTLCore::Type::Float, false);
  // C Math functions
  createStdLibFunction( context, "acos", "acosf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "asin", "asinf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "atan", "atanf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "atan2", "atan2f", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  createStdLibFunction( context, "cos", "cosf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sin", "sinf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "tan", "tanf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "cosh", "coshf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sinh", "sinhf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "tanh", "tanhf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "exp", "expf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "exp_h", "expf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log", "logf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log_h", "logf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log10", "log10f", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "log10_h", "log10f", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "pow", "powf", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  createStdLibFunction( context, "pow_h", "powf", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  createStdLibFunction( context, "sqrt", "sqrtf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "fabs", "fabsf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "floor", "floorf", GTLCore::Type::Float, 1, GTLCore::Type::Float, false);
  createStdLibFunction( context, "fmod", "fmodf", GTLCore::Type::Float, 2, GTLCore::Type::Float, false, GTLCore::Type::Float, false);
  
  if( not _isStandardLibrary and not Compiler::Private::disableStandardLibrary)
  { // of course you don't want to import the standard library when building the standard library
    importModule("ctlstdlib");
  }
  // Reset line number
  d->lineNumber = 1;
  
  // Init the lexer
  std::istringstream iss(sourceCode);
  d->lexer = new LexerNG( &iss );
  d->parser = new ParserNG( this, d->lexer );
  GTLCore::AST::Tree* tree = d->parser->parse();
  if( tree and errorMessages().size() == 0)
  {
    tree->generate( d->moduleData, d->codeGenerator, &context );
    for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = tree->functionsDeclarations().begin();
         it != tree->functionsDeclarations().end(); ++it)
    {
      moduleData->appendFunction( (*it)->function()->name(), (*it)->function());
    }
    OCTL_DEBUG( *d->module );
    // Success
    
    OCTL_DEBUG( "After optimization");
    OCTL_DEBUG( *d->module );
  } else {
    // Failure
    OCTL_DEBUG("failure " << (*errorMessages().begin()).line() << ": " << (*errorMessages().begin()).errorMessage());
    d->module = 0;
  }
  

  // Clean up
  OCTL_DEBUG("Clean up");
  
  delete tree;

  foreach(GTLCore::Function* function, functionsToDelete())
  {
    delete function;
  }
  delete d->lexer;
  d->lexer = 0;
  delete d->parser;
  d->parser = 0;
  llvm::Module* mod = d->module;
  delete d->codeGenerator;
  d->codeGenerator = 0;
  d->module = 0;
  d->moduleData = 0;
  OCTL_DEBUG("Compilation finished");
  return mod;
}

#include <llvm/ValueSymbolTable.h>

bool Compiler::importModule(const GTLCore::String& name)
{
  Module* m = ModulesManager::instance()->loadModule( name );
  GTL_DEBUG("Importing module : " << name << " " << m );
  if( m )
  {
    // Attempt to compile the module if it isn't yet compiled
    if(not m->isCompiled())
    {
      m->compile();
      if(not m->isCompiled())
      {
        OCTL_DEBUG("Compilation error: " << std::endl << m->compilationErrorsMessage() ); // TODO: report the error
        appendErrors( m->compilationErrors() );
        return false;
      }
    }
    d->moduleData->linkWith( m->data() );
    // Append the function coming from the imported module
    std::list<GTLCore::Function*> functions = m->functions();
    foreach( GTLCore::Function* function, functions )
    {
      GTLCore::Function* newFunction = GTLCore::Function::Private::createExternalFunction( d->moduleData, d->moduleData->llvmModule()->getContext(), function );
      OCTL_DEBUG("Import function: " << newFunction->name() );
      declareFunction( newFunction->name(), newFunction );
      functionsToDelete().push_back( newFunction );
    }
#if 0
    // Link a copy of the module
    llvm::Module* clone = llvm::CloneModule( m->data()->llvmModule() );
    llvm::Linker linker("", d->module);
    GTLCore::String errorMessage;
    linker.LinkInModule( clone, &errorMessage );
    OCTL_DEBUG("Linking error: " << errorMessage );
    linker.releaseModule();
    for (llvm::ValueSymbolTable::const_iterator I = clone->getValueSymbolTable().begin(), E = clone->getValueSymbolTable().end(); I != E; ++I) {
      OCTL_DEBUG( I->getKeyData() );
    }
    for (llvm::ValueSymbolTable::const_iterator I =  d->module->getValueSymbolTable().begin(), E =  d->module->getValueSymbolTable().end(); I != E; ++I) {
      OCTL_DEBUG( I->getKeyData() );
    }
    delete clone;
//     OCTL_DEBUG( *d->module );
    // Append the function coming from the imported module
    std::list<GTLCore::Function*> functions = m->functions();
    OCTL_DEBUG(functions.size());
    for(std::list<GTLCore::Function*>::iterator it = functions.begin();
        it != functions.end(); ++it)
    {
      OCTL_DEBUG("Importing " << (*it)->name() );
      int min = (*it)->d->data->minimumParameters();
      int max = (*it)->d->data->maximumParameters();
      std::vector<llvm::Function*> functions(max + 1);
      OCTL_DEBUG( "min = " << min << " max = " << max );
      for(int i = min; i <= max; ++i)
      {
        GTLCore::String name = GTLCore::Function::Data::symbolName( (*it)->name(), (*it)->parameters() );
        OCTL_DEBUG("Looking for " << name );
        OCTL_ASSERT( d->module->getFunction(name) );
        functions[i] = d->module->getFunction(name); //llvm::GlobalValue::ExternalLinkage
      }
      GTLCore::Function::Data* data = new GTLCore::Function::Data( (*it)->parameters(), min );
      data->setFunctions( functions );
      data->setModule( d->module );
      GTLCore::Function* function = new GTLCore::Function( (*it)->name(), (*it)->returnType(), data);
      declareFunction( (*it)->name(), function );
      functionsToDelete().push_back( function );
    }
#endif
  }
  return m;
}

/*********** Handlers ***********/

llvm::Module* Compiler::module()
{
  return d->module;
}

GTLCore::TypesManager* Compiler::typesManager()
{
  GTL_ASSERT(d->moduleData);
  return d->moduleData->typesManager();
}

GTLCore::CodeGenerator* Compiler::codeGenerator()
{
  return d->codeGenerator;
}

GTLCore::AST::Expression* Compiler::standardConstant( const GTLCore::String& _name )
{
  OCTL_DEBUG("requesting constant " << _name);
  if(_name == "M_E") return new GTLCore::AST::NumberExpression<float>( M_E );
  if(_name == "M_PI") return new GTLCore::AST::NumberExpression<float>( M_PI );
  if(_name == "FLT_MAX") return new GTLCore::AST::NumberExpression<float>(  FLT_MAX);
  if(_name == "FLT_MIN") return new GTLCore::AST::NumberExpression<float>( FLT_MIN );
  if(_name == "FLT_POS_INF") return new GTLCore::AST::NumberExpression<float>( floatPosInf() );
  if(_name == "FLT_NEG_INF") return new GTLCore::AST::NumberExpression<float>( floatNegInf() );
  if(_name == "FLT_NAN") return new GTLCore::AST::NumberExpression<float>( floatNaN() );
  if(_name == "HALF_MAX") return new GTLCore::AST::NumberExpression<float>(  FLT_MAX);
  if(_name == "HALF_MIN") return new GTLCore::AST::NumberExpression<float>( FLT_MIN );
  if(_name == "HALF_POS_INF") return new GTLCore::AST::NumberExpression<float>( floatPosInf() );
  if(_name == "HALF_NEG_INF") return new GTLCore::AST::NumberExpression<float>( floatNegInf() );
  if(_name == "HALF_NAN") return new GTLCore::AST::NumberExpression<float>( floatNaN() );
  if(_name == "INT_MAX") return new GTLCore::AST::NumberExpression<int>( INT_MAX );
  if(_name == "INT_MIN")return new GTLCore::AST::NumberExpression<int>( INT_MIN );
  if(_name == "UINT_MAX")return new GTLCore::AST::NumberExpression<int>( UINT_MAX );
  return 0;
}

void Compiler::disableStandardLibrary()
{
  Private::disableStandardLibrary = true;
}

