/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "BinaryExpression.h"

#include <llvm/Instructions.h>

#include <GTLCore/Debug.h>
#include <GTLCore/CodeGenerator_p.h>
#include <GTLCore/ExpressionResult_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/Visitor_p.h>

#include "AccessorExpression.h"
#include "Expression.h"
#include "../ExpressionGenerationContext_p.h"

using namespace GTLCore::AST;

BinaryExpression::~BinaryExpression()
{
  delete m_lhs;
  delete m_rhs;
}


const GTLCore::Type* BinaryExpression::type() const
{
  const GTLCore::Type* type1 = m_lhs->type();
  const GTLCore::Type* type2 = m_rhs->type();
  GTL_ASSERT( type1 == type2 );
  UNUSED(type2);
  return type1;
}

void BinaryExpression::markAsReturnExpression()
{
  m_lhs->markAsReturnExpression();
  m_rhs->markAsReturnExpression();
}

const GTLCore::Type* OrBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult OrBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createOrExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

const GTLCore::Type* AndBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult AndBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createAndExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc));
}

GTLCore::ExpressionResult BitOrBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitOrExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

GTLCore::ExpressionResult BitXorBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitXorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

GTLCore::ExpressionResult BitAndBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitAndExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* EqualEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult EqualEqualBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* DifferentBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult DifferentBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createDifferentExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* InferiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult InferiorEqualBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createInferiorOrEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* InferiorBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult InferiorBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createStrictInferiorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* SupperiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult SupperiorEqualBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createSupperiorOrEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

const GTLCore::Type* SupperiorBinaryExpression::type() const
{
  return Type::Boolean;
}

GTLCore::ExpressionResult SupperiorBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createStrictSupperiorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

GTLCore::ExpressionResult RightShiftBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createRightShiftExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

GTLCore::ExpressionResult LeftShiftBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createLeftShiftExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

GTLCore::ExpressionResult AdditionBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createAdditionExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

GTLCore::ExpressionResult SubstractionBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createSubstractionExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc));
}

GTLCore::ExpressionResult MultiplicationBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createMultiplicationExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

GTLCore::ExpressionResult DivisionBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createDivisionExpression(_egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

GTLCore::ExpressionResult ModuloBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createModuloExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

AssignementBinaryExpression::AssignementBinaryExpression( AccessorExpression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs ), m_lhs(lhs)
{
  GTL_ASSERT(not lhs->isConstant() );
}

llvm::BasicBlock* AssignementBinaryExpression::generateStatement( GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_ASSERT( m_lhs);
  GTL_DEBUG( *m_lhs->type() );
  GTL_DEBUG( *rightHandSide()->type() );
  ExpressionGenerationContext egc(_bb);
  ExpressionResult rhsValue = rightHandSide()->generateValue( _gc, egc);
  return m_lhs->affect( _gc, egc, rhsValue );
}

GTLCore::ExpressionResult AssignementBinaryExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  GTL_ASSERT( m_lhs);
  llvm::BasicBlock* bbr = generateStatement( _gc, _egc.currentBasicBlock() );
  _egc.setCurrentBasicBlock( bbr );
  return GTLCore::ExpressionResult( m_lhs->generateValue( _gc, _egc) );
}

