/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConvertExpression.h"

#include "../CodeGenerator_p.h"
#include "../ExpressionResult_p.h"
#include "../ExpressionGenerationContext_p.h"

using namespace GTLCore;
using namespace GTLCore::AST;

ConvertExpression::ConvertExpression( AST::Expression* _value ) : m_value(_value)
{
}

ConvertExpression::~ConvertExpression()
{
  delete m_value;
}

void ConvertExpression::markAsReturnExpression()
{
  m_value->markAsReturnExpression();
}

bool ConvertExpression::isConstant() const
{
  return m_value->isConstant();
}

const AST::Expression* ConvertExpression::value() const
{
  return m_value;
}

DefaultConvertExpression::DefaultConvertExpression( AST::Expression* _value,  const GTLCore::Type* _dstType ) : ConvertExpression(_value), m_dstType(_dstType)
{
}

DefaultConvertExpression::~DefaultConvertExpression()
{
}

const GTLCore::Type* DefaultConvertExpression::type() const
{
  return m_dstType;
}

GTLCore::ExpressionResult DefaultConvertExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  GTLCore::ExpressionResult result = value()->generateValue( _gc, _egc );
  if( result.isConstant() )
  {
    return GTLCore::ExpressionResult( CodeGenerator::convertConstantTo( result.constant(), result.type(), m_dstType ), m_dstType);
  } else {
    return GTLCore::ExpressionResult( CodeGenerator::convertValueTo( _egc.currentBasicBlock(),  result.value(), result.type(), m_dstType ), m_dstType);
  }
}

ArrayToVectorConvertExpression::ArrayToVectorConvertExpression( AST::Expression* _value, const GTLCore::Type* _dstType ) : ConvertExpression(_value), m_dstType(_dstType)
{
}

ArrayToVectorConvertExpression::~ArrayToVectorConvertExpression()
{
}

const GTLCore::Type* ArrayToVectorConvertExpression::type() const
{
  return m_dstType;
}

GTLCore::ExpressionResult ArrayToVectorConvertExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  GTLCore::ExpressionResult result = value()->generateValue( _gc, _egc );
  if( result.isConstant() )
  {
    return GTLCore::ExpressionResult( CodeGenerator::convertConstantArrayToVector( result.constant(), result.type(), m_dstType ), m_dstType);
  } else {
    return GTLCore::ExpressionResult( CodeGenerator::convertArrayToVector( _gc, _egc,  result.value(), result.type(), m_dstType ), m_dstType);
  }
}
