/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "FunctionDeclaration.h"

#include <llvm/DerivedTypes.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Type.h>

#include <GTLCore/CodeGenerator_p.h>
#include <GTLCore/ExpressionResult_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/Type_p.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Function_p.h>
#include <GTLCore/Utils_p.h>
#include <GTLCore/ModuleData_p.h>
#include <GTLCore/ExpressionGenerationContext_p.h>

#include "Expression.h"
#include "Statement.h"
#include <complex>

using namespace GTLCore::AST;

FunctionParameter::~FunctionParameter()
{
  delete m_initialiser;
}

FunctionDeclaration::~FunctionDeclaration()
{
  deleteAll( m_parameters );
  deleteAll( m_parametersVariable );
  delete m_statement;
}

FunctionDeclaration::FunctionDeclaration(const GTLCore::ScopedName& _name, const GTLCore::Type* _returnType, std::vector< FunctionParameter* > _parameters ) : m_parameters(_parameters), m_statement(0)
{
  int minimumParameters = -1; //_parameters.size() - 1;
  std::vector<Parameter> parameters;
  for( unsigned int i = 0; i < m_parameters.size(); ++i)
  {
    // Compute the minimum number of parameters
    if( m_parameters[i]->initialiser() and minimumParameters == -1 )
    {
      minimumParameters = i;
    }
    parameters.push_back( m_parameters[i]->parameter() );
    // Create parameters variable NG
    const GTLCore::Type* type = m_parameters[i]->parameter().type();
    m_parametersVariable.push_back( new GTLCore::VariableNG( type,
                                    ( type->dataType() == Type::STRUCTURE or type->dataType() == Type::ARRAY ) and ( not m_parameters[i]->parameter().isOutput() ), false ) );
  }
  m_functionData = new Function::Data( parameters, minimumParameters );
  m_function = new Function( _name, _returnType, m_functionData );
}

GTLCore::Function* FunctionDeclaration::function()
{
  return m_function;
}

void FunctionDeclaration::setStatement( Statement* statement )
{
  GTL_ASSERT( not m_statement);
  m_statement = statement;
}

void FunctionDeclaration::generate( ModuleData* _module, GTLCore::CodeGenerator* _codeGenerator, llvm::LLVMContext* _llvmContext)
{
  GTL_DEBUG("Generate function " << m_function->name() );
  m_functionData->setModule( _module );
  // Create the list of parameters as in llvm
  std::vector<const llvm::Type*> params;
  
  if( m_function->d->isReturnedAsPointer() )
  {
    params.push_back( llvm::PointerType::get( m_function->returnType()->d->type(*_llvmContext), 0 ) );
  }
  
  
  for(std::vector< Parameter >::const_iterator it = m_function->parameters().begin();
      it != m_function->parameters().end(); ++it)
  {
    if( it->isOutput()  )
    {
      params.push_back( llvm::PointerType::get( it->type()->d->type(*_llvmContext), 0 ) );
    } else {
      params.push_back( it->type()->d->asArgumentType(*_llvmContext) );
    }
  }
    
  // Generate the full function
  const llvm::Type* returnType = m_function->d->isReturnedAsPointer() ? llvm::Type::getVoidTy(*_llvmContext) : m_function->returnType()->d->asArgumentType(*_llvmContext);
  llvm::FunctionType* definitionType = llvm::FunctionType::get( returnType, params, false );
  GTLCore::String fullFunctionName = GTLCore::Function::Data::symbolName( m_function->name(), m_function->returnType(), m_function->parameters());
  llvm::Function* fullFunction = _codeGenerator->createFunction(definitionType, fullFunctionName );
  
  // Set the list of functions
  m_functionData->setFunction( fullFunction );
  // Generate the code of the function
  llvm::BasicBlock* firstBlock = llvm::BasicBlock::Create(*_llvmContext);
  fullFunction->getBasicBlockList().push_back( firstBlock );
  GTL_DEBUG(*fullFunction);
  // Initialise the context
  GenerationContext generationContext( _codeGenerator, _llvmContext, fullFunction, m_function, _module);
  // Initialize the arguments
  {
    llvm::Function::arg_iterator arg_it = fullFunction->arg_begin();   // Get the arg.
    
    if( m_function->d->isReturnedAsPointer() )
    {
      generationContext.setReturnPointer(arg_it);
      ++arg_it;
    }
    
    std::vector< GTLCore::VariableNG* >::iterator pv_it = m_parametersVariable.begin();
    for(std::vector< Parameter >::const_iterator it = m_function->parameters().begin();
        it != m_function->parameters().end(); ++it, ++arg_it, ++pv_it)
    {
      if( it->isOutput())
      {
        (*pv_it)->initialise( generationContext, firstBlock, arg_it );
      } else {
        if( it->type()->dataType() == Type::STRUCTURE or it->type()->dataType() == Type::ARRAY )
        {
          (*pv_it)->initialise( generationContext, firstBlock, arg_it );
         } else {
          firstBlock = (*pv_it)->initialise( generationContext, firstBlock, ExpressionResult( arg_it, it->type()), std::list<llvm::Value*>() );
        }
      }
    }
  }
  // Call the first statement
  llvm::BasicBlock* lastBlock = m_statement->generateStatement( generationContext, firstBlock);
  if(not lastBlock->getTerminator() )
  {
    GTL_DEBUG("No return instruction for function : " << m_function->name() << " creating one");
    if( m_function->returnType() == GTLCore::Type::Void )
    {
      llvm::ReturnInst::Create(*_llvmContext, lastBlock);
    } else
    {
      llvm::ReturnInst::Create( *_llvmContext, _codeGenerator->convertConstantTo( _codeGenerator->integerToConstant( *_llvmContext, 0), GTLCore::Type::Integer32, m_function->returnType() ), lastBlock  );
    }
  }
}
