/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_MACROS_H_
#define _GTLCORE_MACROS_H_

#cmakedefine __GTL_HAVE_GCC_VISIBILITY

/**
 * @def GTL_NO_EXPORT
 * @ingroup GTLCore
 *
 * The GTL_NO_EXPORT macro marks the symbol of the given variable
 * to be hidden. A hidden symbol is stripped during the linking step,
 * so it can't be used from outside the resulting library, which is similar
 * to static. However, static limits the visibility to the current
 * compilation unit. Hidden symbols can still be used in multiple compilation
 * units.
 *
 * \code
 * int GTL_NO_EXPORT foo;
 * int GTL_EXPORT bar;
 * \endcode
 *
 * @sa GTL_EXPORT
 */

/**
 * @def GTL_EXPORT
 * @ingroup GTLCore
 *
 * The GTL_EXPORT macro marks the symbol of the given variable
 * to be visible, so it can be used from outside the resulting library.
 *
 * \code
 * int GTL_NO_EXPORT foo;
 * int GTL_EXPORT bar;
 * \endcode
 *
 * @sa GTL_NO_EXPORT
 */

/**
 * @def GTL_IMPORT
 * @ingroup GTLCore
 */

#ifdef __GTL_HAVE_GCC_VISIBILITY
#define GTL_NO_EXPORT __attribute__ ((visibility("hidden")))
#define GTL_EXPORT __attribute__ ((visibility("default")))
#define GTL_IMPORT __attribute__ ((visibility("default")))
#elif defined(_WIN32) || defined(_WIN64)
#define GTL_NO_EXPORT
#define GTL_EXPORT __declspec(dllexport)
#define GTL_IMPORT __declspec(dllimport)
#else
#define GTL_NO_EXPORT
#define GTL_EXPORT
#define GTL_IMPORT
#endif

/**
 * The GTL_DEPRECATED macro can be used to trigger compile-time warnings when a deprecated
 * functions are used.
 * 
 * For non-inline functions, the macro has to be inserted at the end of the declaration like in :
 * @code
 * DeprecatedConstructor() GTL_DEPRECATED;
 * void deprecatedFunction() const GTL_DEPRECATED;
 * @endcode
 * 
 * For inline functions, the macro has to be inserted before the declartion but after virtual or
 * static keywoard, like in :
 * @code
 * GTL_DEPRECATED void deprecatedInline() { ... }
 * virtual GTL_DEPRECATED int depreactedInline() { ... }
 * static GTL_DEPRECATED char* deprecatedInline() { ... }
 * @endcode
 * 
 * You can declare a class or struct to be deprecated :
 * @code
 * class GTL_DEPRECATED deprecatedClass { };
 * struct GTL_DEPRECATED deprecatedStruct { };
 * @endcode
 * 
 * @ingroup GTLCore
 */

#if defined(__GNUC__) && !defined(__INTEL_COMPILER) && (__GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 >= 2))
#  define GTL_DEPRECATED __attribute__ ((__deprecated__))
#elif defined(_MSC_VER) && (_MSC_VER >= 1300)
#  define GTL_DEPRECATED __declspec(deprecated)
#else
#  define GTL_DEPRECATED
#endif

#endif
