/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Kernel.h"

#include <sstream>
#include <fstream>

#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/wrappers/Allocate.h"

#include "Debug.h"
#include "CodeGenerator_p.h"
#include "Compiler_p.h"
#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"
#include "wrappers/RegionWrap_p.h"
#include "Library_p.h"

#include "Kernel_p.h"

using namespace OpenShiva;

Kernel::Kernel(const GTLCore::String& _name, int _channelsNb ) : Library( true, _channelsNb), d(new KernelPrivate )
{
  d->self = this;
  d->evaluatePixelesFunction = 0;
  d->wrapper = 0;
  Library::d->libraryCompilation = d;
}

Kernel::Kernel(int _channelsNb ) : Library( true, _channelsNb), d(new KernelPrivate )
{
  d->self = this;
  d->evaluatePixelesFunction = 0;
  d->wrapper = 0;
  Library::d->libraryCompilation = d;
}

Kernel::~Kernel()
{
  cleanup();
  delete d->wrapper;
  delete d;
}

void Kernel::cleanup()
{
  Library::cleanup();
  d->evaluatePixelesFunction = 0; // It's deleted by the moduleData
}

void Kernel::postCompilation()
{
  // Create a wrapper
  d->wrapper = new Wrapper(this, Library::d->m_moduleData);
  d->determineTypes();
  // Create the generateEvaluatePixeles LLVM function
  d->evaluatePixelesFunction = CodeGenerator::generateEvaluatePixeles( this, Library::d->count_channels_generic );
  // Call evaluateDepends as needed
  const std::list<GTLCore::Function*>* evaluateDependentsFuncs = Library::d->m_moduleData->function( name(), "evaluateDependents");
  if( evaluateDependentsFuncs )
  {
    GTLCore::Function* evaluateDependentsFunc = evaluateDependentsFuncs->front();
    void (*func)() = (void (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( evaluateDependentsFunc);
    (*func)();
  }
}

void Kernel::evaluatePixeles( const GTLCore::Region& _region, const std::list< GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage) const
{
  evaluatePixeles(_region, _inputImages, _outputImage, 0);
}

void Kernel::evaluatePixeles( const GTLCore::Region& _region, const std::list< GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage, GTLCore::ProgressReport* report) const
{
  SHIVA_DEBUG( _region.x() << " " << _region.y() << " " << _region.width() << " " << _region.height());
  SHIVA_ASSERT( d->evaluatePixelesFunction );
  
  // Wrap input images
  SHIVA_ASSERT( d->wrapper );
  const void** inputImages = new const void*[ _inputImages.size() ];
  int i = 0;
  for( std::list< GTLCore::AbstractImage* >::const_iterator it = _inputImages.begin();
       it != _inputImages.end(); ++it)
  {
    inputImages[i] = (const void*)d->wrapper->wrapImage( *it );
  }
  
  // Wrap output image
  ImageWrap* owrap = d->wrapper->wrapImage( _outputImage );
  
  // Call function
  SHIVA_DEBUG("Call function");
  void (*func)( int, int, int, int, const void**, void*, void*) = ( void(*)(int, int, int, int, const void**, void*, void* ))GTLCore::VirtualMachine::instance()->getPointerToFunction( d->evaluatePixelesFunction);
  SHIVA_ASSERT(func);
  (*func)( _region.x(), _region.y(), _region.width(), _region.height(), inputImages, owrap, report);
  for( std::size_t i = 0; i < _inputImages.size(); ++i)
  {
    delete (ImageWrap*)inputImages[i];
  }
  delete[] inputImages;
  delete owrap;
  SHIVA_DEBUG( "done" );
}

int Kernel::runTest() const
{
  SHIVA_ASSERT( isCompiled() );
  const std::list<GTLCore::Function*>* fs = Library::d->m_moduleData->function( name(), "runTest");
  SHIVA_ASSERT( fs );
  GTLCore::Function* f = fs->front();
  GTLCore::Value v = f->call( std::vector< GTLCore::Value >() );
  return v.asInt32();
}

bool Kernel::hasTestFunction() const
{
  return Library::d->m_moduleData->function( name(), "runTest");
}

GTLCore::Region Kernel::needed( GTLCore::Region output_region, int input_index, const std::list< GTLCore::Region>& input_DOD)
{
  const std::list<GTLCore::Function*>* neededFunctions = Library::d->m_moduleData->function( name(),"needed");
  GTL_ASSERT( neededFunctions );
  GTLCore::Function* neededFunction = neededFunctions->front();
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( neededFunction);
  RegionWrap* rwrap = (*func)( regionToRegionWrap( output_region ), input_index, regionListToArrayWrap( input_DOD ) );
  
  GTLCore::Region region = regionWrapToRegion( rwrap );
  gtlFree( rwrap );
  return region;
}

bool Kernel::hasNeededFunction() const
{
  return Library::d->m_moduleData->function( name(), "needed");
}

GTLCore::Region Kernel::changed( GTLCore::Region changed_input_region, int input_index, const std::list< GTLCore::Region>& input_DOD)
{
  const std::list<GTLCore::Function*>* changedFunctions = Library::d->m_moduleData->function( name(), "changed");
  GTL_ASSERT(changedFunctions);
  GTLCore::Function* changedFunction = changedFunctions->front();
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( changedFunction);
  ArrayWrap* aw = regionListToArrayWrap( input_DOD );
  RegionWrap* rwrap = (*func)( regionToRegionWrap( changed_input_region ), input_index, aw ); // TODO leak
  
  GTLCore::Region region = regionWrapToRegion( rwrap );
  gtlFree( rwrap );
  gtlFreeAllArray<RegionWrap*>( aw );
  return region;
}

bool Kernel::hasChangedFunction() const
{
  return Library::d->m_moduleData->function( name(), "changed");
}

GTLCore::Region Kernel::generated()
{
  const std::list<GTLCore::Function*>*  fs = Library::d->m_moduleData->function( name(), "generated");
  SHIVA_ASSERT( fs );
  GTLCore::Function* f = fs->front();

  RegionWrap* (*func)() = (RegionWrap* (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( f);
  
  RegionWrap* rwrap = (*func)();
  GTLCore::Region region = regionWrapToRegion( rwrap );
  gtlFree( rwrap );
  return region;
}

bool Kernel::hasGeneratedFunction() const
{
  return Library::d->m_moduleData->function( name(), "generated");
}

void Kernel::setHint( Hint _hint, const GTLCore::Value& _value)
{
  switch( _hint )
  {
    case IMAGE_WIDTH:
    {
      GTL_ASSERT( _value.type()->dataType() == GTLCore::Type::FLOAT );
      d->m_imageWidthHint = _value;
    }
      break;
    case IMAGE_HEIGHT:
    {
      GTL_ASSERT( _value.type()->dataType() == GTLCore::Type::FLOAT );
      d->m_imageHeightHint = _value;
    }
      break;
  }
}
