/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <GTLCore/AbstractImage.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Region.h>
#include <OpenShiva/Kernel.h>

class TestCallConstData : public GTLTest::Case {
  public:
    class FakeImage : public GTLCore::AbstractImage {
      public:
        FakeImage() : GTLCore::AbstractImage(GTLCore::PixelDescription(GTLCore::Type::UnsignedInteger8, 1)), pixel(new char[1]), m_dataAccess(0), m_constDataAccess(0)
        {}
        ~FakeImage()
        {
          delete pixel;
        }
        virtual char* data( int _x, int _y ) 
        {
          ++m_dataAccess;
          return pixel;
        }
        virtual const char* data( int _x, int _y ) const
        {
          ++m_constDataAccess;
          return pixel;
        }
        int dataAccess() const
        {
          return m_dataAccess;
        }
        int constDataAccess() const
        {
          return m_constDataAccess;
        }
        virtual GTLCore::AbstractImage::ConstIterator* createIterator() const { return 0; }
        virtual GTLCore::AbstractImage::Iterator* createIterator() { return 0; }
      private:
        char* pixel;
        mutable int m_dataAccess, m_constDataAccess;
    };
    TestCallConstData() :  GTLTest::Case("CallConstData") {}
    virtual void runTest()
    {
      {
        FakeImage img;
        const FakeImage* img2 = &img;
        GTLTEST_CHECK_EQUAL(img.dataAccess(), 0);
        GTLTEST_CHECK_EQUAL(img.constDataAccess(), 0);
        img.data(0,0);
        GTLTEST_CHECK_EQUAL(img.dataAccess(), 1);
        GTLTEST_CHECK_EQUAL(img.constDataAccess(), 0);
        img2->data(0,0);
        GTLTEST_CHECK_EQUAL(img.dataAccess(), 1);
        GTLTEST_CHECK_EQUAL(img.constDataAccess(), 1);
      }
      {
        FakeImage img;
        GTLCore::String source = "kernel test \
{ \
  void evaluatePixel(out pixel result) \
  { \
      } \
    }";
        OpenShiva::Kernel kernel(1);
        kernel.setSource(source);
        kernel.compile();
        GTLTEST_CHECK_REQUIRED(kernel.isCompiled());
        kernel.evaluatePixeles( GTLCore::Region(0,0,100,50), std::list< GTLCore::AbstractImage* >(), &img);
        GTLTEST_CHECK_EQUAL(img.dataAccess(), 5000);
        GTLTEST_CHECK_EQUAL(img.constDataAccess(), 0);
      }
      {
        FakeImage img;
        GTLCore::String source = "kernel test \
{ \
  void evaluatePixel(in image img, out pixel result) \
  { \
    img.sampleNearest( result.coord ) + img.sampleNearest( result.coord - 1.0 );\
      } \
    }";
        OpenShiva::Kernel kernel(1);
        kernel.setSource(source);
        kernel.compile();
        GTLTEST_CHECK_REQUIRED(kernel.isCompiled());
        std::list< GTLCore::AbstractImage* > imgs;
        imgs.push_back(&img);
        kernel.evaluatePixeles( GTLCore::Region(0,0,100,50), imgs, &img);
        GTLTEST_CHECK_EQUAL(img.dataAccess(), 5000);
        GTLTEST_CHECK_EQUAL(img.constDataAccess(), 10000);
      }
    }
};


class TestWrapper : public GTLTest::Suite {
  public:
    TestWrapper() : GTLTest::Suite("Wrapper")
    {
      addCase(new TestCallConstData );
    }
};
