/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VariablesManager_p.h"

#include <map>
#include <list>

#include "Debug.h"
#include "ScopedName.h"
#include "AST/GarbageCollectionStatement.h"

using namespace GTLCore;

struct VariablesManager::Context
{
  std::map< ScopedName, VariableNG* > variables;
};

struct VariablesManager::Private {
  std::list< Context > contextes;
  std::map< ScopedName, VariableNG* > parameters;
  std::map< ScopedName, VariableNG* > constants;
  String nameSpace;
  
  VariableNG* getVariableInMap( const std::map< ScopedName, VariableNG* >&, const ScopedName& );
  void fillList( std::list< VariableNG* >& list, const std::map< ScopedName, VariableNG* >& map );
};


VariableNG* VariablesManager::Private::getVariableInMap( const std::map< ScopedName, VariableNG* >& map, const ScopedName& n)
{
  for( std::map<ScopedName, VariableNG*>::const_iterator it = map.begin();
        it != map.end(); ++it)
  {
    GTL_DEBUG( " storage: " << it->first << " namespace: " << nameSpace );
    if( it->first == n
        or ( it->first.nameSpace() == nameSpace and it->first.name() == n.name() ) )
    {
      return it->second;
    }
  }
  return 0;
}

void VariablesManager::Private::fillList( std::list< VariableNG* >& list, const std::map< ScopedName, VariableNG* >& map )
{
  for( std::map<ScopedName, VariableNG*>::const_iterator it = map.begin();
        it != map.end(); ++it)
  {
    list.push_back( it->second );
  }
}

VariablesManager::VariablesManager() : d(new Private)
{
}

VariablesManager::~VariablesManager()
{
  delete d;
}

void VariablesManager::setNameSpace( const GTLCore::String& _nameSpace)
{
  d->nameSpace = _nameSpace;
}

void VariablesManager::declareVariable( const ScopedName& _scopedName, VariableNG* _variable)
{
  d->contextes.begin()->variables[_scopedName] = _variable;
}


VariableNG* VariablesManager::getVariable( const ScopedName& n ) const
{
  GTL_DEBUG("getVariable " << n );
  for( std::list<Context>::const_iterator cit = d->contextes.begin();
       cit != d->contextes.end(); cit++)
  {
    VariableNG* var = d->getVariableInMap( cit->variables, n );
    if(var)
    {
      return var;
    }
  }
  VariableNG* var = d->getVariableInMap( d->parameters, n );
  if(var)
  {
    return var;
  }
  if(d->nameSpace != "" and n.nameSpace() == "")
  {
    ScopedName n2(d->nameSpace, n.name());
    VariableNG* var = d->getVariableInMap( d->constants, n2 );
    if(var)
    {
      return var;
    }
  }  
  return d->getVariableInMap( d->constants, n );
}

bool VariablesManager::hasVariableInCurrentContext( const ScopedName&  _scopedName ) const
{
  return (d->contextes.begin()->variables.find( _scopedName ) != d->contextes.begin()->variables.end());
}

void VariablesManager::startContext()
{
  d->contextes.push_front( Context() );
}

void VariablesManager::endContext()
{
  d->contextes.pop_front();
}

void VariablesManager::declareParameter( const ScopedName& _scopedName, VariableNG* _variable)
{
  d->parameters[_scopedName] = _variable;
}

void VariablesManager::reset()
{
  d->parameters.clear();
}

AST::Statement* VariablesManager::garbageCollectCurrentContext() const
{
  std::list< VariableNG* > list;
  d->fillList( list, d->contextes.begin()->variables );
  return new AST::VariablesGarbageCollectionStatement( list );
}

AST::Statement* VariablesManager::garbageCollectEverything( ) const
{
  std::list< VariableNG* > list;
  for( std::list<Context>::const_iterator cit = d->contextes.begin();
       cit != d->contextes.end(); cit++)
  {
    d->fillList( list, cit->variables );
  }
  return new AST::VariablesGarbageCollectionStatement( list );
}

void VariablesManager::declareConstant( const ScopedName& _scopedName, VariableNG* _variable)
{
  d->constants[_scopedName] = _variable;
}
