/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Function_p.h"

#include <cstdarg>

#include <llvm/DerivedTypes.h>
#include <llvm/Module.h>
#include <llvm/Type.h>

#include "Parameter.h"

#include "Debug.h"
#include "ScopedName.h"
#include "Value.h"
#include "Type.h"
#include "Type_p.h"

using namespace GTLCore;

Function::Data::Data( const std::vector< Parameter >& _parameters, int minimumParameters) : m_parameters(_parameters), m_module(0), m_minimumParameters(minimumParameters), m_maximumParameters(m_parameters.size())
{
  if( minimumParameters == -1) m_minimumParameters = (unsigned int)m_maximumParameters;
}

void Function::Data::setFunctions( const std::vector<llvm::Function*>& _functions )
{
  m_functions = _functions;
  GTL_ASSERT(m_functions.size() == m_maximumParameters + 1);
#ifndef NDEBUG
  for(unsigned int i = 0; i < m_minimumParameters; ++i)
  {
    GTL_ASSERT( m_functions[i] == 0);
  }
  for(unsigned int i = m_minimumParameters; i <= m_maximumParameters; ++i)
  {
    GTL_ASSERT( m_functions[i] );
  }
#endif
}
void Function::Data::setModule( llvm::Module* _module )
{
  m_module = _module;
}

llvm::Function* Function::Data::function(unsigned int count)
{
  if( count < m_minimumParameters or count > m_maximumParameters) return 0;
  return m_functions[count];
}
llvm::Function* Function::Data::function(unsigned int count) const
{
  if( count < m_minimumParameters or count > m_maximumParameters) return 0;
  return m_functions[count];
}

GTLCore::String Function::Data::symbolName( const ScopedName& _functionName, const std::vector< Parameter >& _parameters )
{
  return _functionName.nameSpace() + "_" + _functionName.name() + GTLCore::String::number( (int)_parameters.size() );
}

GTLCore::Function* Function::Private::createExternalFunction(llvm::Module* _module, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, int _count, ...)
{
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, _count);
  for(int i = 0; i < _count; ++i)
  {
    const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
    arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
  }
  va_end(argp);
  return createExternalFunction(_module, _name, _symbolName, retType, arguments );
}

GTLCore::Function* Function::Private::createExternalFunction(llvm::Module* _module, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, const std::vector<GTLCore::Parameter>& arguments)
{
  std::vector<const llvm::Type*> llvmArguments;
  for(unsigned int i = 0; i < arguments.size(); ++i)
  {
    llvmArguments.push_back( arguments[i].type()->d->type());
  }
  std::vector<llvm::Function*> functions(arguments.size() + 1);
  functions[arguments.size() ] = dynamic_cast<llvm::Function*>( _module->getOrInsertFunction( _symbolName, llvm::FunctionType::get(retType->d->type(), llvmArguments, false ) ) );
  GTLCore::Function::Data* data = new GTLCore::Function::Data(arguments, arguments.size() );
  data->setFunctions( functions );
  data->setModule( _module );
  return new GTLCore::Function( GTLCore::ScopedName("", _name), retType, data );
}

GTLCore::Function* Function::Private::createInternalFunction(llvm::Module* _module, const GTLCore::String& _name, llvm::Function* _function, const GTLCore::Type* retType, int _count, ...)
{
  GTL_ASSERT( _function );
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, _count);
#ifndef NDEBUG
  const llvm::FunctionType *FTy =
      llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>(_function->getType())->getElementType());
#endif
  for(int i = 0; i < _count; ++i)
  {
    const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
    arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
#ifndef NDEBUG
    GTL_DEBUG( *type->d->asArgumentType() << " == " << *FTy->getParamType(i) );
    GTL_ASSERT( type->d->asArgumentType() == FTy->getParamType(i) );
#endif
  }
  va_end(argp);
  std::vector<llvm::Function*> functions(arguments.size() + 1);
  functions[arguments.size() ] = _function;
  GTLCore::Function::Data* data = new GTLCore::Function::Data(arguments, arguments.size() );
  data->setFunctions( functions );
  data->setModule( _module );
  return new GTLCore::Function( GTLCore::ScopedName("", _name), retType, data );
}
