/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Module.h"

#include <sstream>
#include <fstream>

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Type.h"

#include "Compiler.h"
#include "Debug.h"

using namespace OpenCTL;

// LLVM
#include <llvm/Module.h>
#include <llvm/Bitcode/ReaderWriter.h>
#include <llvm/System/Path.h>
#include <llvm/ModuleProvider.h>
#include <llvm/System/DynamicLibrary.h>

#include <iostream>

struct Module::Private {
  Private() : moduleData(0), moduleProvider(0) {}
  GTLCore::String name;
  GTLCore::String nameSpace;
  GTLCore::String source;
  bool compiled;
  GTLCore::ModuleData* moduleData;
  llvm::ModuleProvider* moduleProvider;
  std::list<GTLCore::ErrorMessage> compilationErrors;
};

Module::Module(const GTLCore::String& name) : d(new Private)
{
  d->compiled = false;
  d->name = llvm::sys::Path(name).getBasename();
}

Module::Module() : d(new Private)
{
  d->compiled = false;
  d->name = "";
}

Module::~Module()
{
  delete d->moduleData;
  delete d;
}

const GTLCore::String& Module::name() const
{
  return d->name;
}

const GTLCore::String& Module::nameSpace() const
{
  return d->nameSpace;
}

void Module::setSource(const GTLCore::String& str)
{
  d->source = str;
}

void Module::loadFromFile(const GTLCore::String& fileName)
{
  d->source = "";
  std::ifstream in;
  in.open(fileName.c_str() );
  if(not in)
  {
    OCTL_DEBUG( "Impossible to open file " << fileName );
    return;
  }
  GTLCore::String str;
  std::getline(in,str);
  while ( in ) {
    d->source += str;
    d->source += "\n";
    std::getline(in,str);
  }
}

const GTLCore::Function* Module::function(const GTLCore::String& name) const
{
  if( d->moduleData )
  {
    return d->moduleData->function( nameSpace(), name);
  }
  return 0;
}

void Module::compile()
{
  if(d->source.empty()) return;
  if(d->moduleProvider)
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->moduleProvider);
    delete d->moduleProvider;
  }
  delete d->moduleData;
  d->moduleData = new GTLCore::ModuleData(new llvm::Module(d->name));

  Compiler c;
  bool result = c.compile( d->source, d->name, d->moduleData );

  if(result)
  {
    d->compiled = true;
    llvm::sys::DynamicLibrary::LoadLibraryPermanently( _OPENCTL_LIB_, 0 ); // This needed because when OpenCTL is used in a plugins, OpenCTL symbols aren't loaded globally and then llvm can't find them
    d->moduleData->doLink();
    d->moduleProvider = new llvm::ExistingModuleProvider( d->moduleData->llvmLinkedModule() );
    GTLCore::VirtualMachine::instance()->registerModule( d->moduleProvider );
  } else {
    delete d->moduleData;
    d->moduleData = 0;
    d->compilationErrors = c.errorMessages();
  }
}

GTLCore::String Module::compilationErrorsMessage() const
{
  std::ostringstream os;
  for( std::list<GTLCore::ErrorMessage>::iterator it = d->compilationErrors.begin();
       it != d->compilationErrors.end(); ++it)
  {
    os << it->fileName() << " at " << it->line() << " : " << it->errorMessage()  << std::endl;
  }
  return os.str();
}


bool Module::isCompiled() const
{
  return d->compiled;
}

const std::list<GTLCore::ErrorMessage>& Module::compilationErrors() const
{
  return d->compilationErrors;
}

GTLCore::String Module::asmSourceCode() const
{
  std::ostringstream os;
  os << *d->moduleData->llvmLinkedModule() << std::endl;
  return os.str();
}

std::list<GTLCore::Function*> Module::functions()
{
  return d->moduleData->functions();
}

const GTLCore::ModuleData* Module::data() const
{
  return d->moduleData;
}

const GTLCore::TypesManager* Module::typesManager() const
{
  return d->moduleData->typesManager();
}
