/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConstantCoumpoundExpression.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "../CodeGenerator_p.h"
#include "../Debug.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../ExpressionResult_p.h"

using namespace GTLCore::AST;

ConstantCoumpoundExpression::ConstantCoumpoundExpression( const GTLCore::Type* _type, const std::vector<Expression*>& _expressions) : m_expressions(_expressions), m_type(_type)
{
  GTL_ASSERT(_type);
}

ConstantCoumpoundExpression::~ConstantCoumpoundExpression( )
{
  for( std::vector<Expression*>::iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
  {
    delete *it;
  }
}

const GTLCore::Type* ConstantCoumpoundExpression::type() const
{
  return m_type;
}

bool ConstantCoumpoundExpression::isConstant() const
{
  return m_constant;
}

GTLCore::ExpressionResult ConstantCoumpoundExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* _bb ) const
{
  GTL_DEBUG( m_type->dataType() << " " << Type::ARRAY << " " << Type::STRUCTURE );
  if( m_type->dataType() == Type::ARRAY or m_type->dataType() == Type::VECTOR)
  {
    const GTLCore::Type* arrayType = m_type->embeddedType();
    std::vector< llvm::Constant* > members;
    for(  std::vector<Expression*>::const_iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( (*it)->generateValue( _gc, _bb).constant(), (*it)->type(), arrayType ) );
    }
    if( m_type->dataType() == Type::ARRAY )
    {
      std::vector<llvm::Constant*> arrayStruct;
      arrayStruct.push_back( _gc.codeGenerator()->integerToConstant( 0 ) );
      arrayStruct.push_back( _gc.codeGenerator()->integerToConstant( members.size() ) );
      llvm::Constant* constant = llvm::ConstantArray::get(
                    llvm::ArrayType::get( arrayType->d->type(), members.size()), members );
      llvm::GlobalVariable* gvar = new llvm::GlobalVariable( constant->getType(), true, llvm::GlobalValue::ExternalLinkage, constant, "", _gc.llvmModule() );
      
      llvm::Constant* zero = llvm::ConstantInt::get(llvm::Type::Int32Ty, 0); // Access the structure of the array
      llvm::Constant *Ops[] = {zero, zero };
      llvm::Constant* iptr = llvm::ConstantExpr::getGetElementPtr( gvar, Ops, 2);
      
      arrayStruct.push_back( iptr );
      
      const llvm::StructType* structType = dynamic_cast<const llvm::StructType*>( m_type->d->type() );
      GTL_ASSERT( structType );
      
      return GTLCore::ExpressionResult(
            llvm::ConstantStruct::get( structType, arrayStruct ), type() );
    } else if( m_type->dataType() == Type::VECTOR )
    {
      const llvm::VectorType* vectorType = dynamic_cast<const llvm::VectorType*>( m_type->d->type() );
      GTL_ASSERT( vectorType );
      llvm::Constant* constant = llvm::ConstantVector::get( vectorType, members );
      GTL_DEBUG( "Coumpound constant = " << *constant << " type = " << *m_type );
      return GTLCore::ExpressionResult( constant, m_type );
    }
  } else if( m_type->dataType() == Type::STRUCTURE )
  {
    std::vector<llvm::Constant*> members;
    members.push_back( CodeGenerator::integerToConstant( 0 ) ); // GC constant
    for( uint i = 0; i < m_expressions.size(); ++i)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( m_expressions[i]->generateValue( _gc, _bb).constant(), m_expressions[i]->type(), m_type->structDataMember(i).type() ) );
    }
    const llvm::StructType* structType = dynamic_cast<const llvm::StructType*>( m_type->d->type() );
    return GTLCore::ExpressionResult(
          llvm::ConstantStruct::get( structType, members ), type() );
  }
  GTL_ABORT("Can't generate coumpound value, neither array, neither vector.");
  return GTLCore::ExpressionResult();
}

void ConstantCoumpoundExpression::markAsReturnExpression()
{
  GTL_ABORT("ConstantCoumpoundExpression can't be in a return statement");
}

Expression* ConstantCoumpoundExpression::expressionAt(unsigned int _idx )
{
  return m_expressions[ _idx ];
}
