/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConvertCenter_p.h"

#include <list>
#include <map>

#include "AST/ConvertExpression.h"
#include "Debug.h"
#include "Type_p.h"

using namespace GTLCore;

//--------------------------- ConvertExpressionFactory ---------------------------//

struct ConvertExpressionFactory::Private {
};

ConvertExpressionFactory::ConvertExpressionFactory( ) : d(new Private)
{
}

ConvertExpressionFactory::~ConvertExpressionFactory()
{
  delete d;
}


//------------------------ DefaultConvertExpressionFactory -----------------------//

class DefaultConvertExpressionFactory : public ConvertExpressionFactory {
  public:
    virtual AST::ConvertExpression* create( AST::Expression* value, const GTLCore::Type* _dstType ) const
    {
      GTL_ASSERT( canConvertBetween( value->type(), _dstType ));
      return new AST::DefaultConvertExpression( value, _dstType );
    }
    virtual bool canConvertBetween( const GTLCore::Type* srcType, const GTLCore::Type* dstType) const
    {
      return ( srcType == dstType)
          or ( dstType->dataType() == Type::VECTOR and canConvertBetween( srcType, dstType->embeddedType() ) )
          or ( srcType->isNumber() and dstType->isNumber() );
    }
};

//--------------------------------- ConvertCenter --------------------------------//

struct ConvertCenter::Private {
  std::list< ConvertExpressionFactory* > factories;
  std::map< const GTLCore::Type*, const GTLCore::Type* > autoconversion;
};

ConvertCenter::ConvertCenter() : d(new Private)
{
  addConvertExpressionFactory( new DefaultConvertExpressionFactory );
}

ConvertCenter::~ConvertCenter()
{
  for( std::list< ConvertExpressionFactory* >::iterator it = d->factories.begin(); it != d->factories.end(); ++it)
  {
    delete *it;
  }
  delete d;
}

AST::Expression* ConvertCenter::createConvertExpression( AST::Expression* _value, const GTLCore::Type* _dstType ) const
{
  if( _value->type() == _dstType )
  {
    return _value;
  }
  for( std::list< ConvertExpressionFactory* >::iterator it = d->factories.begin(); it != d->factories.end(); ++it)
  {
    if( (*it)->canConvertBetween( _value->type(), _dstType ) )
    {
      return (*it)->create( _value, _dstType );
    }
  }
  return 0;
}

std::pair<AST::Expression*, AST::Expression*> ConvertCenter::createConvertExpressions( AST::Expression* value1, AST::Expression* value2 ) const
{
  GTL_ASSERT(value1);
  GTL_ASSERT(value2);
  // Auto conversion
  value1 = createConvertExpression( value1, autoConvertType( value1->type() ) );
  value2 = createConvertExpression( value2, autoConvertType( value2->type() ) );
  GTL_ASSERT(value1);
  GTL_ASSERT(value2);
  // If values are of the same type, there is no need to convert
  if( value1->type() == value2->type() )
  {
    return std::pair<AST::Expression*, AST::Expression*>( value1, value2 );
  }
  // Select the conversion with the highest priority
  const Type* type = Type::Private::selectType(value1->type(), value2->type());
  GTL_DEBUG( "Selected " << *type << " for " << *value1->type() << " and " << *value2->type());
  return std::pair<AST::Expression*, AST::Expression*>( createConvertExpression(value1, type), createConvertExpression(value2, type)) ;
}

void ConvertCenter::addConvertExpressionFactory( ConvertExpressionFactory* cef)
{
  d->factories.push_back( cef );
}

void ConvertCenter::addAutoConversion( const GTLCore::Type* _srcType, const GTLCore::Type* _dstType )
{
  d->autoconversion[ _srcType ] = _dstType;
}

const GTLCore::Type* ConvertCenter::autoConvertType( const GTLCore::Type* _srcType ) const
{
  if( d->autoconversion.find( _srcType ) == d->autoconversion.end() )
  {
    return _srcType;
  } else {
    return d->autoconversion[ _srcType ];
  }
}
