/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Value.h"

#include "Debug.h"
#include "Type.h"
#include "TypesManager_p.h"

using namespace GTLCore;

struct Value::Private {
  Private() : type(0) {}
  union {
    bool b;
    float f;
    int i;
    unsigned int ui;
    std::vector< Value >* array;
  } value;
  const Type* type;
  void cleanup()
  {
    if( type and (type->dataType() == Type::ARRAY or type->dataType() == Type::VECTOR ) )
    {
      delete value.array;
    }
  }
};

Value::Value() : d(new Private)
{
  d->type = Type::Undefined;
}

Value::Value(float v) : d(new Private)
{
  d->value.f = v;
  d->type = Type::Float;
}

Value::Value(bool v) : d(new Private)
{
  d->value.b = v;
  d->type = Type::Boolean;
}

Value::Value(int v) : d(new Private)
{
  d->value.i = v;
  d->type = Type::Integer32;
}

Value::Value(unsigned int v) : d(new Private)
{
  d->value.ui = v;
  d->type = Type::UnsignedInteger32;
}

Value::Value(const std::vector< Value >& v, const GTLCore::Type* _type ) : d(new Private)
{
  d->value.array = new std::vector< Value >( v );
  if( _type )
  {
    d->type = _type;
  } else {
    d->type = TypesManager::getArray( v[0].type() );
  }
}

Value::Value(const Value& rhs) : d(new Private(*rhs.d))
{
  if( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::VECTOR )
  {
    d->value.array = new std::vector< Value >( *rhs.d->value.array );
  }
}

Value Value::operator=(const Value& rhs)
{
  d->cleanup();
  *d = *rhs.d;
  if( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::VECTOR )
  {
    d->value.array = new std::vector< Value >( *rhs.d->value.array );
  }
  return *this;
}

bool Value::operator==(const Value& rhs) const
{
  if( d->type == rhs.d->type )
  {
    switch( d->type->dataType())
    {
      case Type::BOOLEAN:
        return d->value.b == rhs.d->value.b;
      case Type::FLOAT:
        return d->value.f == rhs.d->value.f;
      case Type::INTEGER32:
        return d->value.i == rhs.d->value.i;
      case Type::UNSIGNED_INTEGER32:
        return d->value.ui == rhs.d->value.ui;
      case Type::ARRAY:
      case Type::VECTOR:
      {
        if( d->value.array->size() == rhs.d->value.array->size() )
        {
          for( std::size_t i = 0; i < d->value.array->size(); ++i)
          {
            if( not( (*d->value.array)[i] == (*rhs.d->value.array)[i] ) )
            {
              return false;
            }
          }
          return true;
        }
      }
      default:
        return false;
    }
  }
  return false;
}

Value::~Value()
{
  d->cleanup();
  delete d;
}

#define RETURN_AS( _type_ ) \
  switch( d->type->dataType()) \
  { \
    case Type::BOOLEAN: \
      return (_type_)d->value.b; \
    case Type::FLOAT: \
      return (_type_)d->value.f; \
    case Type::INTEGER32: \
      return (_type_)d->value.i; \
    case Type::UNSIGNED_INTEGER32: \
      return (_type_)d->value.ui; \
    default: \
      GTL_ABORT("Can't convert to that type."); \
      return (_type_)0; \
 }

float Value::asFloat() const
{
  RETURN_AS(float);
}

void Value::setFloat( float _v )
{
  d->value.f = _v;
  d->type = Type::Float;
}

bool Value::asBoolean() const
{
  RETURN_AS(bool);
}

int Value::asInt32() const
{
  RETURN_AS(int);
}

unsigned int Value::asUnsignedInt32() const
{
  RETURN_AS(unsigned int);
}

const Type* Value::type() const
{
  return d->type;
}

void Value::setInt32(int v)
{
  d->value.i = v;
  d->type = Type::Integer32;
}

const std::vector< Value >* Value::asArray() const
{
  if( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::VECTOR )
  {
    return d->value.array;
  } else {
    return 0;
  }
}

void Value::setArray( const std::vector< Value >& _array, const GTLCore::Type* _type )
{
  d->cleanup();
  d->value.array = new std::vector< Value >( _array );
  if( _type )
  {
    d->type = _type;
  } else {
    d->type = TypesManager::getArray( _array[0].type() );
  }
}
