/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/AbstractImage.h>
#include <GTLCore/Debug.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Region.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>
#include <GTLImageIO/Version.h>

void printVersion()
{
  std::cout << GTLImageIO::LibraryShortName << " - " << GTLImageIO::LibraryName << " - " << GTLImageIO::LibraryVersionString << std::endl;
  std::cout << GTLImageIO::LibraryCopyright << std::endl;
}
void printHelp()
{
  std::cout << "Usage : imagecompare file1 file2" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else {
      break;
    }
  }
  if( argc != 3)
  {
    printHelp();
  } else {
    GTLCore::String fileName1 = argv[1]; 
    GTLCore::String fileName2 = argv[2];
    const GTLImageIO::ImageDC* decoder1 = GTLImageIO::ImageDCRegistry::instance()->decoder( fileName1 );
    if(not decoder1 )
    {
      std::cerr << "No decoder for " << fileName1 << std::endl;
      return EXIT_FAILURE;
    }
    const GTLImageIO::ImageDC* decoder2 = GTLImageIO::ImageDCRegistry::instance()->decoder( fileName2 );
    if(not decoder2 )
    {
      std::cerr << "No encoder for " << fileName2 << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::String errMsg;
    GTLCore::Region region1;
    GTLCore::AbstractImage* image1 = decoder1->decode( fileName1, &region1, &errMsg );
    if( not image1 )
    {
      std::cerr << "Error while decoding " << fileName1 << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::Region region2;
    GTLCore::AbstractImage* image2 = decoder2->decode( fileName2, &region2, &errMsg );
    if(not image2 )
    {
      std::cerr << "Error while decoding " << fileName2 << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
    // Test size
    if( region1 != region2 )
    {
      std::cout << "Different image size : " << region1 << " != " << region2 << std::endl;
      return EXIT_FAILURE;
    }
    int errorCount = image1->compare( image2, region1 );
    if( errorCount != 0 )
    {
      std::cout << errorCount << " different pixels" << std::endl;
      return EXIT_FAILURE;
    }
  }
  return EXIT_SUCCESS;
}
