;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html


(in-package :ccl)


;;;;;;;;;;;;;;;;;;; from lispequ - put in clos-macros or something

;; Generic Function Dispatch tables.
;; These accessors are at the beginning of the table.
;; rest of the table is alternating wrappers & combined-methods.
;



(defun make-gf (name bits &optional (dcode #'%%0-arg-dcode))
  (let* ((dt (make-gf-dispatch-table))
         (gf (%cons-gf name dt dcode bits)))
    (setf (%gf-dispatch-table-gf dt) gf)
    gf))

(defun make-n+1th-arg-combined-method (methods gf argnum)
  (let ((table (make-gf-dispatch-table)))
    (setf (%gf-dispatch-table-methods table) methods
          (%gf-dispatch-table-argnum table) (%i+ 1 argnum))
    (let ((self (%cons-combined-method gf table #'%%nth-arg-dcode t))) ; <<
      (setf (%gf-dispatch-table-gf table) self)
      self)))

;Bring the generic function to the smallest possible size by removing
;any cached recomputable info.  Currently this means clearing out the
;combined methods from the dispatch table.

(defun clear-gf-cache (gf)
  #-bccl (unless t (typep gf 'standard-generic-function) 
           (report-bad-arg gf 'standard-generic-function))
  (let ((dt (%gf-dispatch-table gf)))
    (if (eq (%gf-dispatch-table-size dt) *min-gf-dispatch-table-size*)
      (clear-gf-dispatch-table dt)
      (let ((new (make-gf-dispatch-table)))
        (setf (%gf-dispatch-table-methods new) (%gf-dispatch-table-methods dt))
        (setf (%gf-dispatch-table-precedence-list new)
              (%gf-dispatch-table-precedence-list dt))
        (setf (%gf-dispatch-table-gf new) gf)
        (setf (%gf-dispatch-table-instance new)
              (%gf-dispatch-table-instance dt))
        (setf (%gf-dispatch-table-argnum new) (%gf-dispatch-table-argnum dt))
        (setf (%gf-dispatch-table gf) new)))))

(defun grow-gf-dispatch-table (gf-or-cm wrapper table-entry &optional obsolete-wrappers-p)
  ; Grow the table associated with gf and insert table-entry as the value for
  ; wrapper.  Wrapper is a class-wrapper.  Assumes that it is not obsolete.
  (let* ((dt (if (standard-generic-function-p gf-or-cm)
               (%gf-dispatch-table gf-or-cm)
               (%combined-method-methods gf-or-cm)))  ; huh
         (size (%gf-dispatch-table-size dt))
         (new-size (if obsolete-wrappers-p
                     size
                     (%i+ size size)))
         new-dt)
    (if (> new-size *max-gf-dispatch-table-size*)
      (progn 
        (when (not (fixnump (%gf-dispatch-table-mask dt)))(bug "906")) ; cant be right that its so big
        (setq new-dt (clear-gf-dispatch-table dt)
                   *gf-dt-ovf-cnt* (%i+ *gf-dt-ovf-cnt* 1))
        (when (not (fixnump (%gf-dispatch-table-mask new-dt)))(bug "903")))
      (progn
        (setq new-dt (make-gf-dispatch-table new-size))
        (setf (%gf-dispatch-table-methods new-dt) (%gf-dispatch-table-methods dt)
              (%gf-dispatch-table-precedence-list new-dt) (%gf-dispatch-table-precedence-list dt)
              (%gf-dispatch-table-instance new-dt) (%gf-dispatch-table-instance dt)
              (%gf-dispatch-table-gf new-dt) gf-or-cm
              (%gf-dispatch-table-argnum new-dt) (%gf-dispatch-table-argnum dt))
        (let ((i 0) index w cm)
          (dotimes (j (%ilsr 1 (%gf-dispatch-table-size dt)))
	    (declare (fixnum j))
            (unless (or (null (setq w (%gf-dispatch-table-ref dt i)))
                        (eql 0 (%wrapper-hash-index w))
                        (no-applicable-method-cm-p
                         (setq cm (%gf-dispatch-table-ref dt (%i+ i 1)))))
              (setq index (find-gf-dispatch-table-index new-dt w t))
              (setf (%gf-dispatch-table-ref new-dt index) w)
              (setf (%gf-dispatch-table-ref new-dt (%i+ index 1)) cm))
            (setq i (%i+ i 2))))))
    (let ((index (find-gf-dispatch-table-index new-dt wrapper t)))
      (setf (%gf-dispatch-table-ref new-dt index) wrapper)
      (setf (%gf-dispatch-table-ref new-dt (%i+ index 1)) table-entry))
    (if (standard-generic-function-p gf-or-cm)
      (setf (%gf-dispatch-table gf-or-cm) new-dt)
      (setf (%combined-method-methods gf-or-cm) new-dt))))

;; also need to nuke :method-class standard-reader-method from defclass in level-2


(defun inner-lfun-bits (function &optional value)
  (lfun-bits (closure-function function) value))



; probably want to use alists vs. hash-tables initially


; only used if error - well not really
(defun collect-lexpr-args (args first &optional last) 
  (if (listp args)
    (subseq args first (or last (length args)))
    (let ((res nil))
      (when (not last)(setq last (%lexpr-count args)))
      (dotimes (i (- last first))
        (setq res (push (%lexpr-ref args last (+ first i)) res)))
      (nreverse res))))




(defmacro with-list-from-lexpr ((list lexpr) &body body)
  (let ((len (gensym)))
    `(let* ((,len (%lexpr-count ,lexpr))
            (,list  (make-list ,len)))
       (declare (dynamic-extent ,list) (fixnum ,len))       
       (do* ((i 0 (1+ i))
             (ls ,list (cdr ls)))
            ((= i ,len) ,list)
         (declare (fixnum i) (list ls))
         (declare (optimize (speed 3)(safety 0)))
         (%rplaca ls (%lexpr-ref ,lexpr ,len i)))
       ,@body)))



(defmacro %standard-instance-p (i)
  `(eq (typecode ,i) arch::subtag-instance))

#+ppc-target
(defppclapfunction %apply-lexpr-with-method-context ((magic arg_x)
                                                   (function arg_y)
                                                   (args arg_z))
  ; Somebody's called (or tail-called) us.
  ; Put magic arg in ppc::next-method-context (= ppc::temp1).
  ; Put function in ppc::nfn (= ppc::temp2).
  ; Set nargs to 0, then spread "args" on stack (clobbers arg_x, arg_y, arg_z,
  ;   but preserves ppc::nfn/ppc::next-method-context.
  ; Jump to the function in ppc::nfn.
  (mr ppc::next-method-context magic)
  (mr ppc::nfn function)
  (set-nargs 0)
  (mflr loc-pc)
  (bla .SPspread-lexpr-z)
  (mtlr loc-pc)
  (lwz temp0 arch::misc-data-offset nfn)
  (mtctr temp0)
  (bctr))

#+sparc-target
(defsparclapfunction %apply-lexpr-with-method-context ((magic %arg_x)
                                                       (function %arg_y)
                                                       (args %arg_z))
  ; Somebody's called (or tail-called) us.
  ; Put magic arg in sparc::next-method-context (= sparc::temp1).
  ; Put function in sparc::nfn (= sparc::temp2).
  ; Set nargs to 0, then spread "args" on stack (clobbers arg_x, arg_y, arg_z,
  ;   but preserves sparc::nfn/sparc::next-method-context.
  ; Jump to the function in sparc::nfn.
  (mov magic %next-method-context)
  (mov function %nfn)
  (mov %ra0 %ra1)
  (call-subprim .SPspread-lexpr-z)
    (set-nargs 0)
  (ld (%nfn arch::misc-data-offset) %temp0)
  (jmp %temp0 arch::misc-data-offset)
    (mov %ra1 %ra0))


#+ppc-target
(defppclapfunction %apply-with-method-context ((magic arg_x)
                                                   (function arg_y)
                                                   (args arg_z))
  ; Somebody's called (or tail-called) us.
  ; Put magic arg in ppc::next-method-context (= ppc::temp1).
  ; Put function in ppc::nfn (= ppc::temp2).
  ; Set nargs to 0, then spread "args" on stack (clobbers arg_x, arg_y, arg_z,
  ;   but preserves ppc::nfn/ppc::next-method-context.
  ; Jump to the function in ppc::nfn.
  (mr ppc::next-method-context magic)
  (mr ppc::nfn function)
  (set-nargs 0)
  (mflr loc-pc)
  (bla .SPspreadargZ)
  (mtlr loc-pc)
  (lwz temp0 arch::misc-data-offset nfn)
  (mtctr temp0)
  (bctr))

#+sparc-target
(defsparclapfunction %apply-with-method-context ((magic %arg_x)
						 (function %arg_y)
						 (args %arg_z))
  ; Somebody's called (or tail-called) us.
  ; Put magic arg in sparc::next-method-context (= sparc::temp1).
  ; Put function in sparc::nfn (= sparc::temp2).
  ; Set nargs to 0, then spread "args" on stack (clobbers arg_x, arg_y, arg_z,
  ;   but preserves sparc::nfn/sparc::next-method-context.
  ; Jump to the function in sparc::nfn.
  (mov magic %next-method-context)
  (mov function %nfn)
  (mov %ra0 %ra1)
  (call-subprim .SPspreadargz)
    (set-nargs 0)
  (ld (%nfn arch::misc-data-offset) %temp0)
  (jmp %temp0 arch::misc-data-offset)
    (mov %ra1 %ra0))




(declaim (inline %find-1st-arg-combined-method))
(declaim (inline %find-nth-arg-combined-method))


; for calls from outside - e.g. stream-reader
(defun find-1st-arg-combined-method (gf arg)
  (declare (optimize (speed 3)(safety 0)))
  (%find-1st-arg-combined-method (%gf-dispatch-table gf) arg))

(defun %find-1st-arg-combined-method (dt arg)
  (declare (optimize (speed 3)(safety 0)))
  (flet ((get-wrapper (arg)
           (if (not (%standard-instance-p arg))
             (let* ((class (class-of arg)))
               (or (%class-own-wrapper class)
                   (initialize-class-and-wrapper class)))
             (%instance-class-wrapper arg))))
    (declare (inline get-wrapper))
    (let ((wrapper (get-wrapper arg)))
      (when (eql 0 (%wrapper-hash-index wrapper))
        (update-obsolete-instance arg)
        (setq wrapper (get-wrapper arg)))
      (let* ((mask (%gf-dispatch-table-mask dt))
             (index (%ilsl 1 (%ilogand mask (%wrapper-hash-index wrapper))))
             table-wrapper flag)
        (declare (fixnum index mask))
        (loop 
          (if (eq (setq table-wrapper (%gf-dispatch-table-ref dt index)) wrapper)
            (return (%gf-dispatch-table-ref dt  (the fixnum (1+ index))))
            (progn
              (when (null (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                (if (or (neq table-wrapper (%unbound-marker-8))
                        (eql 0 flag))
                  (without-interrupts ; why?
                   (return (1st-arg-combined-method-trap (%gf-dispatch-table-gf dt) wrapper arg))) ; the only difference?
                  (setq flag 0 index -2)))
              (setq index (+ 2 index)))))))))

; more PC - it it possible one needs to go round more than once? - seems unlikely
(defun %find-nth-arg-combined-method (dt arg args)  
  (declare (optimize (speed 3)(safety 0)))
  (flet ((get-wrapper (arg)
           (if (not (%standard-instance-p arg))
             (let* ((class (class-of arg)))
               (or (%class-own-wrapper class)
                   (initialize-class-and-wrapper class)))
             (%instance-class-wrapper arg))))
    (declare (inline get-wrapper))
    (let ((wrapper (get-wrapper arg)))
      (when (eql 0 (%wrapper-hash-index wrapper))
        (update-obsolete-instance arg)
        (setq wrapper (get-wrapper arg)))
      (let* ((mask (%gf-dispatch-table-mask dt))
             (index (%ilsl 1 (%ilogand mask (%wrapper-hash-index wrapper))))
             table-wrapper flag)
        (declare (fixnum index mask))
        (loop 
          (if (eq (setq table-wrapper (%gf-dispatch-table-ref dt index)) wrapper)
            (return (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
            (progn
              (when (null (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                (if (or (neq table-wrapper (%unbound-marker-8))
                        (eql 0 flag))
                  (without-interrupts ; why?
                   (let ((gf (%gf-dispatch-table-gf dt)))
                     (if (listp args)
                       (return (nth-arg-combined-method-trap-0 gf dt wrapper args))
                       (with-list-from-lexpr (args-list args)
                         (return (nth-arg-combined-method-trap-0 gf dt wrapper args-list))))))
                  (setq flag 0 index -2)))
              (setq index (+ 2 index)))))))))

#| ; not used today
(defun %find-2nd-arg-combined-method (dt arg1 arg)  
  (declare (optimize (speed 3)(safety 0)))
  (flet ((get-wrapper (arg)
           (if (not (%standard-instance-p arg))
             (let* ((class (class-of arg)))
               (or (%class-own-wrapper class)
                   (initialize-class-and-wrapper class)))
             (%instance-class-wrapper arg))))
    (declare (inline get-wrapper))
    (let ((wrapper (get-wrapper arg)))
      (when (eql 0 (%wrapper-hash-index wrapper))
        (update-obsolete-instance arg)
        (setq wrapper (get-wrapper arg)))
      (let* ((mask (%gf-dispatch-table-mask dt))
             (index (%ilsl 1 (%ilogand mask (%wrapper-hash-index wrapper))))
             table-wrapper flag)
        (declare (fixnum index mask))
        (loop 
          (if (eq (setq table-wrapper (%gf-dispatch-table-ref dt index)) wrapper)
            (return (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
            (progn
              (when (null (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                (if (or (neq table-wrapper (%unbound-marker-8))
                        (eql 0 flag))
                  (without-interrupts ; why?
                   (let ((gf (%gf-dispatch-table-gf dt)))
                     (let ((args-list (make-list 2)))
                       (declare (dynamic-extent args-list))
                       (%rplaca args-list arg1)
                       (%rplaca (cdr args-list) arg)
                       (return (nth-arg-combined-method-trap-0 gf dt wrapper args-list))
                       )))
                  (setq flag 0 index -2)))
              (setq index (+ 2 index)))))))))
|#


;;;;;;;;;;;;;;;;;;;;;;;;;;; Generic functions and methods ;;;;;;;;;;;;;;;;;;;;


(defun standard-method-p (thing)
  (when (%standard-instance-p thing)
    (let* ((cpl (%class-cpl (%wrapper-class (%instance-class-wrapper thing))))
           (smc *standard-method-class*))
      (dolist (c cpl)
        (if (eq c smc)(return t))))))



(defun %method-function-p (thing)
  (when (functionp thing)
    (let ((bits (lfun-bits thing)))
      (declare (fixnum bits))
      (logbitp $lfbits-method-bit bits))))


(setf (type-predicate 'standard-generic-function) 'standard-generic-function-p)
(setf (type-predicate 'combined-method) 'combined-method-p)

(setf (type-predicate 'standard-method) 'standard-method-p)

;; Maybe we shouldn't make this a real type...
(setf (type-predicate 'method-function) '%method-function-p)

(defvar %all-gfs% (%cons-population nil))

(eval-when (:compile-toplevel :execute)
(defconstant $lfbits-numinh-mask (logior (dpb -1 $lfbits-numinh 0)
                                         (%ilsl $lfbits-nonnullenv-bit 1)))
)

; if we do this then %gf-dispatch-table and  friends are different too.
; its still the case that the code vectors for these things are all the
; same so could (should) be shared



; #'it - copy code-vect, 2 literals, name, bits = 5
; with closures we just copy code-vect name bits = 3 




#+ppc-target
(defparameter *gf-proto*
  (nfunction
   gag
   (lambda (&lap &lexpr args)
     (ppc-lap-function 
      gag 
      ()
      (mflr loc-pc)
      (vpush-argregs)
      (vpush nargs)
      (add imm0 vsp nargs)
      (la imm0 4 imm0)                  ; caller's vsp
      (bla .SPlexpr-entry)
      (mtlr loc-pc)                     ; return to kernel
      (mr arg_z vsp)                    ; lexpr
      (lwz arg_y '#(nil nil) nfn)       ; dispatch table
      (set-nargs 2)
      (lwz nfn '#(a b) nfn)             ; dcode function
      (lwz temp0 arch::misc-data-offset nfn)
      (mtctr temp0)
      (bctr)))))

#+sparc-target
(defparameter *gf-proto*
  (nfunction
   gag
   (lambda (&lap &lexpr args)
     (sparc-lap-function 
      gag 
      ()
      (vpush-argregs)
      (add %vsp %nargs %imm0)           ; caller's vsp
      (vpush %nargs)
      (call-subprim* .SPlexpr-entry)
        (nop)
      (mov %vsp %arg_z)                 ; lexpr
      (ld (%nfn '#(nil nil)) %arg_y)    ; dispatch table
      (ld (%nfn '#(a b)) %nfn)          ; dcode function
      (ld (%nfn arch::misc-data-offset) %temp0)
      (jmp %temp0 arch::misc-data-offset)
        (set-nargs 2)))))

  

;; is a winner - saves ~15%
#+ppc-target
(defppclapfunction gag-one-arg ((arg arg_z))
  (twnei nargs 4)  
  (lwz arg_y '#(nil nil) nfn) ; mention dt first
  (set-nargs 2)
  (lwz nfn '#(a b) nfn)
  (lwz temp0 arch::misc-data-offset nfn)
  (mtctr temp0)
  (bctr))

#+sparc-target
(defsparclapfunction gag-one-arg ((arg %arg_z))
  (check-nargs 1)
  (ld (%nfn '#(nil nil)) %arg_y) ; mention dt first
  (ld (%nfn '#(a b)) %nfn)
  (ld (%nfn arch::misc-data-offset) %temp0)
  (jmp %temp0 arch::misc-data-offset)
    (set-nargs 2))

#+ppc-target
(defppclapfunction gag-two-arg ((arg0 arg_y) (arg1 arg_z))
  (twnei nargs 8)  
  (lwz arg_x '#(nil nil) nfn) ; mention dt first
  (set-nargs 3)
  (lwz nfn '#(a b) nfn)
  (lwz temp0 arch::misc-data-offset nfn)
  (mtctr temp0)
  (bctr))
  
#+sparc-target
(defsparclapfunction gag-two-arg ((arg0 %arg_y) (arg1 %arg_z))
  (check-nargs 2)
  (ld (%nfn '#(nil nil)) %arg_x) ; mention dt first
  (ld (%nfn '#(a b)) %nfn)
  (ld (%nfn arch::misc-data-offset) %temp0)
  (jmp %temp0 arch::misc-data-offset)
    (set-nargs 3))

(defparameter *gf-proto-one-arg*  #'gag-one-arg)
(defparameter *gf-proto-two-arg*  #'gag-two-arg)




(defun %cons-gf (name dispatch-table dcode bits)
  (setq bits (%ilogior (%ilsl $lfbits-gfn-bit 1) (%ilogand $lfbits-args-mask bits)))
  (let* ((vect (copy-uvector (%lfun-vector *gf-proto*)))
         (fn (%lfun-vector-lfun vect)))
    (set-imm0 fn dispatch-table) ; 6 off fn
    (set-imm1 fn dcode)
    (lfun-name fn name)
    (lfun-bits fn bits)
    (push fn (population-data %all-gfs%))
    fn))

(defun %cons-combined-method (gf thing dcode &optional jsr?)
  (declare (ignore jsr?))
  ;(dbg 24)
  (let ((bits (lfun-bits gf)))
    (setq bits (%ilogior (%ilsl $lfbits-cm-bit 1) (%ilogand $lfbits-args-mask bits)))
    ; set bits and name = gf
    (let* ((vect (copy-uvector (%lfun-vector *gf-proto*)))
           (fn (%lfun-vector-lfun vect)))
      (set-imm0 fn thing) ; 6 off fn
      (set-imm1 fn dcode)
      (lfun-bits fn bits)
      (lfun-name fn gf)
      fn)))

(defun %gf-dispatch-table (gf)
  ;(require-type gf 'standard-generic-function)
  ; in ppc land this would be imm 0 - or %svref 1 - or 2 if name is first ?
  (%nth-immediate (%lfun-vector gf) 0))

(defun %gf-dcode (gf)
  ;(require-type gf 'standard-generic-function)
  (%nth-immediate (%lfun-vector gf) 1))

(defun %set-gf-dcode (gf val)
  (set-imm1 gf val))

(defun %set-gf-dispatch-table (gf val)
  (set-imm0 gf val))


(defun %combined-method-methods  (cm)
  ;(require-type cm 'combined-method)
  ; in ppc land this would be imm 0 - or %svref 1 - or 2 if name is first ?
  (%nth-immediate (%lfun-vector cm) 0))

(defun %combined-method-dcode (cm)
  ;(require-type cm 'combined-method)
  (%nth-immediate (%lfun-vector cm) 1))

(defun %set-combined-method-methods (cm val)
  (set-imm0 cm val))

(defun %set-combined-method-dcode (cm val)
  (set-imm1 cm val))



(defun set-imm0 (thing value)
  (setf (%svref thing 1) value))
(defun set-imm1 (thing value)
  (setf (%svref thing 2) value))

; this screws up all the class-of/type-of stuff
(defun standard-generic-function-p (thing)
  (when (functionp thing)
    (let () ;(outer-bits (lfun-bits thing)))
      (when t ;(%ilogbitp $lfbits-trampoline-bit outer-bits)  ; <<
        (let ((bits (lfun-bits thing)))
          (declare (fixnum bits))
          (and (logbitp $lfbits-gfn-bit bits)
               (not (logbitp $lfbits-method-bit bits))))))))



(defun combined-method-p (thing)
  (when (functionp thing)
    (let () ;(outer-bits (lfun-bits thing)))
      (when t ;(%ilogbitp $lfbits-trampoline-bit outer-bits)  ; <<
        (let ((bits (lfun-bits thing)))
          (declare (fixnum bits))
          (and (logbitp $lfbits-cm-bit bits)
               (not (logbitp  $lfbits-method-bit bits))))))))

(setf (type-predicate 'standard-generic-function) 'standard-generic-function-p)
(setf (type-predicate 'combined-method) 'combined-method-p)











;;; A generic-function looks like:
;;; 
;;; header | trampoline | apply-dcode-fn | dispatch-table | dcode | name | bits
;;; %svref :    0              1                  2           3       4     5
;;;
;;; The trampoline is a code vector that's used to call closures.
;;; Actually today its *gf-proto*'s code vector that gins up a lexpr
;;; The "apply-dcode-fn" is a function that APPLYs the dcode function to 
;;;  the dispatch-table and some args.  
;;; The dispatch-table and dcode are sort of settable closed-over variables.


(defun set-immediate-0 (thing value)
  (setf (%svref thing 1) value))
(defun set-immediate-1 (thing value)
  (setf (%svref thing 2) value))
(defun set-immediate-2 (thing value)
  (setf (%svref thing 3) value))

(defun %gf-dcode (gf)
  ;(require-type gf 'standard-generic-function)
  (%svref gf 2))

(defun %gf-dispatch-table (gf)
  (%svref gf 1))





(defun copy-closure-inner (fun)
  ;(dbg 1026)  ; first here  2010
  (let* ((inner (%lfun-vector (closure-function fun)))
         (new-inner (copy-uvector inner)))
    (when (eq fun (closure-function fun)) (bug "1027"))  ; 2018
    ;(cerror "a" "b")
    ; aka set-closure-lfun
    (set-immediate-0 fun (%lfun-vector-lfun new-inner))

    fun))


(defsetf %gf-dispatch-table %set-gf-dispatch-table)

(defun %gf-methods (gf)
  (%gf-dispatch-table-methods (%gf-dispatch-table gf)))

(defun %gf-precedence-list (gf)
  (%gf-dispatch-table-precedence-list (%gf-dispatch-table gf)))

(defun %gf-instance (gf)
  (%gf-dispatch-table-instance (%gf-dispatch-table gf)))

; Returns the %gf-instance of a STANDARD-GENERIC-FUNCTION
; If INSTANCE is not a STANDARD-GENERIC-FUNCTION, returns INSTANCE if
; it is a STANDARD-INSTANCE, otherwise NIL.
; If INSTANCE is a STANDARD-GENERIC-FUNCTION, but has no %gf-instance,
; returns NIL.
(defun %maybe-gf-instance (instance)
  (if (functionp instance)
    (if (standard-generic-function-p instance)
      (%gf-instance instance))
    (and (%standard-instance-p instance)
         instance)))

(defun %set-gf-instance (gf instance)
  (setf (%gf-dispatch-table-instance (%gf-dispatch-table gf)) instance))

(defsetf %gf-instance %set-gf-instance)


(defsetf %gf-dcode %set-gf-dcode)

(defun %gf-method-class (gf)
  (let ((instance (%gf-instance gf)))
    (if instance
      (slot-value gf 'method-class)  ; huh
      *standard-method-class*)))

; initialized for real in l1-clos
(defvar *standard-method-combination* nil)

(defun %gf-method-combination (gf)
  (let ((instance (%gf-instance gf)))
    (if instance
      (slot-value gf 'method-combination)
      *standard-method-combination*)))
                                                        


; ultimately its an svref

(defun combined-method-or-gf-first-literal (gf)
  (if (combined-method-p gf)
    (%combined-method-methods gf)
    (%gf-dispatch-table gf)))


(defun %combined-method-methods  (cm)
  ;(require-type cm 'combined-method)
  ; in ppc land this would be imm 0 - or %svref 1 - or 2 if name is first ?
  (%svref cm 1))

(defun %combined-method-dcode (cm)
  ;(require-type cm 'combined-method)
  (%svref cm 2))


; need setters too

(defsetf %combined-method-methods %set-combined-method-methods)

(defparameter *min-gf-dispatch-table-size* 2
  "The minimum size of a generic-function dispatch table")

(defun make-gf-dispatch-table (&optional (size *min-gf-dispatch-table-size*))
  (when (<= size 0) (report-bad-arg size '(integer 1)))
  (setq size (%imax (%ilsl (%i- (integer-length (%i+ size size -1))
                                1)
                           1)           ; next power of 2
                    *min-gf-dispatch-table-size*))
  (let ((res (%cons-gf-dispatch-table size)))
    (setf (%gf-dispatch-table-mask res) (%i- (%ilsr 1 size) 1)
          (%gf-dispatch-table-argnum res) 0
          (%gf-dispatch-table-ref res size) (%unbound-marker-8))
    res))

; I wanted this to be faster - I didn't
(defun clear-gf-dispatch-table (dt)
  (let ((i %gf-dispatch-table-first-data))
    (dotimes (j (%gf-dispatch-table-size dt))
      (declare (fixnum j))
      (setf (%svref dt i) nil               ; svref is for debugging - nil not 0 is right
            i (%i+ i 1)))
    (setf (%svref dt i) (%unbound-marker-8))   ; paranoia...
    (setf (svref dt (%i+ 1 i)) nil))
  dt)    ;  ok did we run off the end


; Remove all combined-methods from the world
(defun clear-all-gf-caches ()
  (dolist (f (population-data %all-gfs%))
    (clear-gf-cache f))
  (clrhash *combined-methods*)
  nil)



;  Lap fever strikes again... is this still correct? - seems not - maybe ok now
(defun find-gf-dispatch-table-index (dt wrapper &optional skip-full-check?)
  ;searches for an empty slot in dt at the hash-index for wrapper.
  ;returns nil if the table was full.
  (let ((contains-obsolete-wrappers-p nil)
        (mask (%gf-dispatch-table-mask dt)))
    (declare (fixnum mask))
    (unless skip-full-check?
      (let* ((size (1+ mask))
             (max-count (- size (the fixnum (ash (the fixnum (+ size 3)) -2))))
             (index 0)
             (count 0))
        (declare (fixnum size max-count index count))
        (dotimes (i size)
          (declare (fixnum i))
          (let ((wrapper (%gf-dispatch-table-ref dt index)))
            (if wrapper
              (if (eql 0 (%wrapper-hash-index wrapper))
                (setf contains-obsolete-wrappers-p t
                      (%gf-dispatch-table-ref dt index) *obsolete-wrapper*
                      (%gf-dispatch-table-ref dt (1+ index))
                      #'(lambda (&rest rest) 
                          (declare (ignore rest))
                          (error "Generic-function dispatch bug.")))
                (setq count (%i+ count 1)))))
          (setq index (%i+ index 2)))
        (when (> count max-count)
          (return-from find-gf-dispatch-table-index (values nil contains-obsolete-wrappers-p)))))
    (let* ((index (ash (logand mask (%wrapper-hash-index wrapper)) 1)) ; * 2 ??
           (flag nil)
           table-wrapper)      
      (values
       (loop
         (while (and (neq wrapper
                          (setq table-wrapper (%gf-dispatch-table-ref dt index)))
                     (%gf-dispatch-table-ref dt (1+ index))
                     (neq 0 (%wrapper-hash-index table-wrapper)))
           (setq index (%i+ index 2)))
         (if (eq (%unbound-marker-8) table-wrapper)
           (if flag
             (return nil)         ; table full
             (setq flag 1
                   index 0))
           (return index)))
       contains-obsolete-wrappers-p))))


(defvar *obsolete-wrapper* #(obsolete-wrapper 0))
(defvar *gf-dispatch-bug*
  #'(lambda (&rest rest)
      (declare (ignore rest))
      (error "Generic-function dispatch bug!")))

  
; This maximum is necessary because of the 32 bit arithmetic in
; find-gf-dispatch-table-index.
(defparameter *max-gf-dispatch-table-size* (expt 2 16))
(defvar *gf-dt-ovf-cnt* 0)              ; overflow count

(defvar *no-applicable-method-hash* nil)



(setq *no-applicable-method-hash* (make-hash-table :test 'eq :size 0 :weak :key))


(defun make-no-applicable-method-function (gf)
  ;(dbg 99)  ; we get here 318  - why??
  (if *no-applicable-method-hash*
    (progn
      (or (gethash gf *no-applicable-method-hash*))
      (setf (gethash gf *no-applicable-method-hash*)
            (%cons-no-applicable-method gf)))
    (%cons-no-applicable-method gf)))

(defun %cons-no-applicable-method (gf)
  (%cons-combined-method gf gf #'%%no-applicable-method))

; Returns true if F is a combined-method that calls no-applicable-method
(defun no-applicable-method-cm-p (f)
  (and (typep f 'combined-method)
       (eq '%%no-applicable-method
           (function-name (%combined-method-dcode f)))))


(defun %%no-applicable-method (gf args)
  ; do we really need this? - now we do
  ;(declare (dynamic-extent args)) ; today caller does the &rest
  (if (listp args)
    (apply #'no-applicable-method gf args)
    (%apply-lexpr #'no-applicable-method gf args )))

; if obsolete-wrappers-p is true, will rehash instead of grow.
; It would be better to do the rehash in place, but I'm lazy today.


(defun arg-wrapper (arg)
  (or (standard-object-p arg)
      (%class-own-wrapper (class-of arg))
      (error "~a has no wrapper" arg)))

;;;;;;;;;;;;;;;;;;;;;;;;; generic-function dcode ;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Simple case for generic-functions with no specializers
;; Why anyone would want to do this I can't imagine.

(defun %%0-arg-dcode (dispatch-table args) ; need to get gf from table
  (let ((method (or (%gf-dispatch-table-ref dispatch-table 1)
                    (0-arg-combined-method-trap
                     (%gf-dispatch-table-gf dispatch-table)))))
    (if (not (listp args))
      (progn
        (%apply-lexpr-tail-wise method args))
      (apply method args))))


; arg passed is dispatch table -  add a slot to it containing gf? - later
; or pass the gf instead of the dispatch table 
; (means adding another constant to gf to contain the dispatch table- above is clearer)

(defun %%1st-arg-dcode (dt  args)
  ;(declare (dynamic-extent args))
  (if (not (listp args))
    (let* ((args-len (%lexpr-count args)))
      (if (neq 0 args-len) 
        (let ((method (%find-1st-arg-combined-method dt (%lexpr-ref args args-len 0))))
	  (%apply-lexpr-tail-wise method args))
        (error "0 args to ~s" (%gf-dispatch-table-gf dt))))
    (let* ()  ; happens if traced
      (when (null args) (error "0 args to ~s" (%gf-dispatch-table-gf dt)))
      (let ((method (%find-1st-arg-combined-method dt (%car args))))
        (apply method args)))))

; oh damn this screws up trace and advise big time - fixed we think
(defun %%one-arg-dcode (dt  arg)
  (let ((method (%find-1st-arg-combined-method dt arg)))
    (funcall method arg)))

; two args - specialized on first
(defun %%1st-two-arg-dcode (dt arg1 arg2)
  (let ((method (%find-1st-arg-combined-method dt arg1)))
    (funcall method arg1 arg2)))

#|
; two args - specialized on second - worth the trouble? maybe not since most are writers anyway
; we have 43 callers of nth-arg-dcode out of 1100 gf's
; I think we have only one multimethod = convert-scrap
(defun %%2nd-two-arg-dcode (dt arg1 arg2)
  (let ((method (%find-2nd-arg-combined-method dt arg1 arg2)))
    (funcall method arg1 arg2)))
|#



;  arg is dispatch-table and argnum is in the dispatch table
(defun %%nth-arg-dcode (dt args)
  ;(declare (dynamic-extent args))
  (if (listp args)
    (let* ((args-len (list-length args))
           (argnum (%gf-dispatch-table-argnum dt)))
      (when (< args-len argnum) (error "Too few args ~s to ~s." args-len (%gf-dispatch-table-gf dt)))
      (let ((method (%find-nth-arg-combined-method dt (nth argnum args) args)))
        (apply method args)))
    (let* ((args-len (%lexpr-count args))
           (argnum (%gf-dispatch-table-argnum dt)))
      (when (< args-len argnum) (error "Too few args ~s to ~s." args-len (%gf-dispatch-table-gf dt)))
      (let ((method (%find-nth-arg-combined-method dt (%lexpr-ref args args-len argnum) args)))
	(%apply-lexpr-tail-wise method args)))))

#+ppc-target
(defppclapfunction %apply-lexpr-tail-wise ((method arg_y) (args arg_z))
  ; This assumes
  ; a) that "args" is a lexpr made via the .SPlexpr-entry mechanism
  ; b) That the LR on entry to this function points to the lexpr-cleanup
  ;    code that .SPlexpr-entry set up
  ; c) That there weren't any required args to the lexpr, e.g. that
  ;    (%lexpr-ref args (%lexpr-count args) 0) was the first arg to the gf.
  ; The lexpr-cleanup code will be EQ to either (lisp-global ret1valaddr)
  ; or (lisp-global lexpr-return1v).  In the former case, discard a frame
  ; from the cstack (multiple-value tossing).  Restore FN and LR from
  ; the first frame that .SPlexpr-entry pushed, restore vsp from (+ args 4),
  ; pop the argregs, and jump to the function.
  ; d) The lexpr args have not been modified since they were moved by a stack overflow
  (mflr loc-pc)
  (ref-global imm0 ret1valaddr)
  (cmpw cr2 loc-pc imm0)
  (lwz nargs 0 args)
  (cmpwi cr0 nargs 0)
  (cmpwi cr1 nargs '2)
  (mr nfn arg_y)
  (lwz temp0 arch::misc-data-offset nfn)
  (mtctr temp0)
  (if (:cr2 :eq)
    (la sp ppc::lisp-frame.size sp))
  (lwz loc-pc ppc::lisp-frame.savelr sp)
  (lwz fn ppc::lisp-frame.savefn sp)
  (lwz imm0 ppc::lisp-frame.savevsp sp)
  (sub vsp imm0 nargs)
  (mtlr loc-pc)
  (la sp ppc::lisp-frame.size sp)
  (beqctr)
  (vpop arg_z)
  (bltctr cr1)
  (vpop arg_y)
  (beqctr cr1)
  (vpop arg_x)
  (bctr))

#+sparc-target
(defsparclapfunction %apply-lexpr-tail-wise ((method %arg_y) (args %arg_z))
  ; This assumes
  ; a) that "args" is a lexpr made via the .SPlexpr-entry mechanism
  ; b) That the LR on entry to this function points to the lexpr-cleanup
  ;    code that .SPlexpr-entry set up
  ; c) That there weren't any required args to the lexpr, e.g. that
  ;    (%lexpr-ref args (%lexpr-count args) 0) was the first arg to the gf.
  ; The lexpr-cleanup code will be EQ to either (lisp-global ret1valaddr)
  ; or (lisp-global lexpr-return1v).  In the former case, discard a frame
  ; from the cstack (multiple-value tossing).  Restore FN and LR from
  ; the first frame that .SPlexpr-entry pushed, restore vsp from (+ args 4),
  ; pop the argregs, and jump to the function.
  ; d) The lexpr args have not been modified since they were moved by a stack overflow
  (ref-global %imm0 ret1valaddr)
  (cmp %ra0 %imm0)
  (ld (args) %nargs)
  (mov method %nfn)
  (ld (%nfn arch::misc-data-offset) %temp0)
  (be.a @1)
   (inc sparc::lisp-frame.size %lsp)
  @1
  (ld (%lsp sparc::lisp-frame.savelr) %ra0)
  (ld (%lsp sparc::lisp-frame.savefn) %fn)
  (ld (%lsp sparc::lisp-frame.savevsp) %imm0)
  (sub %imm0 %nargs %vsp)
  (discard-lisp-frame)
  (tst %nargs)
  (be @go)
   (cmp %nargs '2)
  (vpop %arg_z)
  (bl @go)
   (nop)
  (vpop %arg_y)
  (be @go)
   (nop)
  (vpop %arg_x)
  @go
  (jmp %temp0 arch::misc-data-offset)
    (nop))


(defun 0-arg-combined-method-trap (gf)
  (let* ((methods (%gf-methods gf))
         (mc (%gf-method-combination gf))
         (cm (if (eq mc *standard-method-combination*)
               (make-standard-combined-method methods nil gf)
               (compute-effective-method-function 
                gf 
                mc
                (sort-methods (copy-list methods) nil)))))
    (setf (%gf-dispatch-table-ref (%gf-dispatch-table gf) 1) cm)
    cm))

(defun compute-effective-method-function (gf mc methods)  
  (if methods
    (compute-effective-method gf mc methods)
    (make-no-applicable-method-function gf)))

(defun 1st-arg-combined-method-trap (gf wrapper arg)
  ; Here when we can't find the method in the dispatch table.
  ; Compute it and add it to the table.  This code will remain in Lisp.
  ;In case pointing to the lfun-vector of a swappable - punt swapping
  (declare (resident))
  ;#-bccl (setq gf (require-type gf 'standard-generic-function))  
  (let ((table (%gf-dispatch-table gf))
        (combined-method (compute-1st-arg-combined-method gf arg wrapper)))
    (multiple-value-bind (index obsolete-wrappers-p)
                         (find-gf-dispatch-table-index table wrapper)
      (if index
        (setf (%gf-dispatch-table-ref table index) wrapper
              (%gf-dispatch-table-ref table (%i+ index 1)) combined-method)
        (grow-gf-dispatch-table gf wrapper combined-method obsolete-wrappers-p)))
    combined-method))

(defvar *cpl-classes* nil)

(defun %inited-class-cpl (class &optional initialize-can-fail)
  (or (%class-cpl class)
      (if (memq class *cpl-classes*)
        (compute-cpl class)
        (let ((*cpl-classes* (cons class *cpl-classes*)))
          (declare (dynamic-extent *cpl-classes*))
          (initialize-class class initialize-can-fail)
          (%class-cpl class)))))


(defun compute-1st-arg-combined-method (gf arg &optional 
                                           (wrapper (arg-wrapper arg)))
  (declare (resident))
  ;#-bccl (setq gf (require-type gf 'standard-generic-function))  
  (let* ((methods (%gf-dispatch-table-methods (%gf-dispatch-table gf)))
         (cpl (%inited-class-cpl (%wrapper-class wrapper)))
         (method-combination (%gf-method-combination gf))
         applicable-methods eql-methods specializer)
    (dolist (method methods)
      ;#-bccl (setq method (require-type method 'standard-method))   ; for debugging.
      (setq specializer (%car (%method-specializers method)))
      ;(DBG 449)(DBG SPECIALIZER)(dbg cpl)(dbg (length cpl))
      (if (listp specializer)
        (when (cpl-memq (%wrapper-class (arg-wrapper (cadr specializer))) cpl)
          (push method eql-methods))
        (when (cpl-memq specializer cpl)
          (push method applicable-methods))))
    ;(WHEN (NULL APPLICABLE-METHODS)(DBG 448)(DBG METHODS)(DBG GF)(DBG CPL))    
    (if (null eql-methods)
      (if (eq method-combination *standard-method-combination*)
        (make-standard-combined-method applicable-methods (list cpl) gf)
        (compute-effective-method-function 
         gf 
         method-combination
         (sort-methods applicable-methods
                       (list cpl)
                       (%gf-precedence-list gf))))
      (make-eql-combined-method  
       eql-methods applicable-methods (list cpl) gf 0 nil method-combination))))
      


(defvar *combined-methods* (make-hash-table  :test 'equal :weak :value))                          

(defun gethash-combined-method (key)
  (gethash key *combined-methods*))

(defun puthash-combined-method (key value)
  (setf (gethash key *combined-methods*) value))

;; Some statistics on the hash table above
(defvar *returned-combined-methods* 0)
(defvar *consed-combined-methods* 0)

;; Assumes methods are already sorted if cpls is nil
(defun make-standard-combined-method (methods cpls gf &optional
                                              (ok-if-no-primaries (null methods)))
  (unless (null cpls)
    (setq methods (sort-methods 
                   methods cpls (%gf-precedence-list (combined-method-gf gf)))))
  (let* ((keywords (compute-allowable-keywords-vector gf methods))
         (combined-method (make-standard-combined-method-internal
                           methods gf keywords ok-if-no-primaries)))
    (if (and keywords methods)
      (make-keyword-checking-combined-method gf combined-method keywords)
      combined-method)))


; Initialized below after the functions exist.
(defvar *clos-initialization-functions* nil)

; Returns NIL if all keywords allowed, or a vector of the allowable ones.
(defun compute-allowable-keywords-vector (gf methods)
  (setq gf (combined-method-gf gf))
  (unless (memq gf *clos-initialization-functions*)
    (let* ((gbits (inner-lfun-bits gf))
           (&key-mentioned-p (logbitp $lfbits-keys-bit gbits)))
      (unless (or (logbitp $lfbits-aok-bit gbits)
                  (dolist (method methods)
                    (let ((mbits (lfun-bits (%method-function method))))
                      (when (logbitp $lfbits-keys-bit mbits)
                        (setq &key-mentioned-p t)
                        (if (logbitp $lfbits-aok-bit mbits)
                          (return t)))))
                  (not &key-mentioned-p))
        (let (keys)
          (flet ((adjoin-keys (keyvect keys)
                              (when keyvect
                                (dovector (key keyvect) (pushnew key keys)))
                              keys))
            (when (logbitp $lfbits-keys-bit gbits)
              (setq keys (adjoin-keys (%defgeneric-keys gf) keys)))
            (dolist (method methods)
              (let ((f (%inner-method-function method)))
                (when (logbitp $lfbits-keys-bit (lfun-bits f))
                  (setq keys (adjoin-keys (lfun-keyvect f) keys))))))
          (apply #'vector keys))))))

; The aux arg is used by keyword checking for %call-next-method-with-args - it is?
(defun make-keyword-checking-combined-method (gf combined-method keyvect)
  (let* ((bits (inner-lfun-bits gf))
         (numreq (ldb $lfbits-numreq bits))
         (key-index (+ numreq (ldb $lfbits-numopt bits))))
    ;(dbg 500)
    (%cons-combined-method 
     gf       
     (vector key-index keyvect combined-method)
     #'%%check-keywords
     t)))
; ok

; #(keyvect key-index combined-method) in atemp1 - actually key-index keyvect today


(defun odd-keys-error (varg l) 
  (let ((gf (combined-method-gf (%svref varg 2))))
    (error "Odd number of keyword args to ~s~%keyargs: ~s" gf l)))


(defun bad-key-error (key varg l)
  (let* ((keys (%svref varg 1))
         (gf (combined-method-gf (%svref varg 2)))
         (*print-array* t)
         (*print-readably* t)
         (readable-keys (format nil "~s" keys)))
    (error "Bad keyword ~s to ~s.~%keyargs: ~s~%allowable keys are ~a." key gf l readable-keys)))

; vector arg is (vector key-index keyvect combined-method) ; the next combined method
(defun %%check-keywords (vector-arg args)
  (flet ((do-it (vector-arg args)
           (let* ((args-len (length args))
                  (keyvect (%svref vector-arg 1))
                  (keyvect-len (length keyvect))
                  (key-index (%svref vector-arg 0)))
             ; vector arg is (vector key-index keyvect combined-method) ; the next combined method
             (declare (fixnum args-len key-index keyvect-len))
             (when (>= args-len key-index)
               (let* ((keys-in (- args-len key-index))
                      aok)  ; actually * 2
                 (declare (fixnum  key-index keys-in keyvect-len))
                 (when (logbitp 0 keys-in) (odd-keys-error vector-arg (collect-lexpr-args args key-index args-len)))
                 (do ((i key-index (+ i 2))
                      (kargs (nthcdr key-index args) (cddr kargs)))
                     ((eq i args-len))
                   (declare (fixnum i))
                   (when aok (return))
                   (let ((key (car kargs)))
                     (when (and (eq key :allow-other-keys)
                                (cadr kargs))
                       (return))
                     (when (not (dotimes (i keyvect-len nil)
                                  (if (eq key (%svref keyvect i))
                                    (return t))))
                       ; not found - is :allow-other-keys t in rest of user args
                       (when (not (do ((remargs kargs (cddr remargs)))
                                      ((null remargs) nil)
                                    (when (and (eq (car remargs) :allow-other-keys)
                                               (cadr remargs))
                                      (setq aok t)
                                      (return t))))              
                         (bad-key-error key vector-arg (collect-lexpr-args args key-index args-len))))))))
             (let ((method (%svref vector-arg 2)))
               ; magic here ?? not needed
               (apply method args)))))
    (if (listp args)
      (do-it vector-arg args)
      (with-list-from-lexpr (args-list args)
        (do-it vector-arg args-list)))))

; called from %%call-next-method-with-args - its the key-or-init-fn 
; called from call-next-method-with-args - just check the blooming keys
; dont invoke any methods - maybe use x%%check-keywords with last vector elt nil
; means dont call any methods - but need the gf or method for error message
(defun x-%%check-keywords (vector-arg ARGS)
  ;(declare (dynamic-extent args))
    ; vector arg is (vector key-index keyvect unused)
  (let* ((ARGS-LEN (length args))
         (keyvect (%svref vector-arg 1))
         (keyvect-len (length keyvect))
         (key-index (%svref vector-arg 0))
         (keys-in (- args-len key-index))
         aok)  ; actually * 2
    (declare (fixnum args-len key-index keys-in keyvect-len))
    
    (when (logbitp 0 keys-in) (odd-keys-error vector-arg (collect-lexpr-args args key-index args-len)))
    (do ((i key-index (+ i 2))
         (kargs (nthcdr key-index args) (cddr kargs)))
        ((eq i args-len))
      (declare (fixnum i))
      (when aok (return))
      (let ((key (car kargs)))
        (when (and (eq key :allow-other-keys)
                   (cadr kargs))
          (return))
        (when (not (dotimes (i keyvect-len nil)
                     (if (eq key (%svref keyvect i))
                       (return t))))
          ; not found - is :allow-other-keys t in rest of user args
          (when (not (do ((remargs kargs (cddr remargs)))
                         ((null remargs) nil)
                       (when (and (eq (car remargs) :allow-other-keys)
                                  (cadr remargs))
                         (setq aok t)
                         (return t))))              
            (bad-key-error key vector-arg 
                           (collect-lexpr-args args key-index args-len))))))))
#| ; testing
(setq keyvect  #(:a :b ))
(setq foo (make-array 3))
(setf (aref foo 0) keyvect (aref foo 1) 2)
(setf (aref foo 2)(method window-close (window)))
( %%check-keywords 1 2 :a 3 :c 4 foo)
( %%check-keywords 1 2 :a 3 :b 4 :d foo)
|#
 
    



; Map an effective-method to it's generic-function.
; This is only used for effective-method's which are not combined-method's
; (e.g. those created by non-STANDARD method-combination)
(defvar *effective-method-gfs* (make-hash-table :test 'eq :weak :key))


(defun get-combined-method (method-list gf)
  (let ((cm (gethash-combined-method method-list)))
    (when cm
      (setq gf (combined-method-gf gf))
      (if (combined-method-p cm)
        (and (eq (combined-method-gf cm) gf) cm)
        (and (eq (gethash cm *effective-method-gfs*) gf) cm)))))

(defun put-combined-method (method-list cm gf)
  (unless (%method-function-p cm)       ; don't bother with non-combined methods
    (puthash-combined-method method-list cm)
    (unless (combined-method-p cm)
      (setf (gethash cm *effective-method-gfs*) (combined-method-gf gf))))
  cm)

(defun make-standard-combined-method-internal (methods gf &optional 
                                                       keywords
                                                       (ok-if-no-primaries
                                                        (null methods)))
  ;(DBG 699)  ; 15d8
  (let ((method-list (and methods (compute-method-list methods))))
    (if method-list                 ; no applicable primary methods
      (if (atom method-list)
        (%method-function method-list)    ; can jump right to the method-function
        (progn
          (incf *returned-combined-methods*)  ; dont need this
          (if (contains-call-next-method-with-args-p method-list)
            (make-cnm-combined-method gf methods method-list keywords)
            (or (get-combined-method method-list gf)
                (progn
                  (incf *consed-combined-methods*)  ; dont need this
                  (puthash-combined-method
                   method-list
                   (%cons-combined-method
                    gf method-list #'%%standard-combined-method-dcode)))))))
      (if ok-if-no-primaries
        (make-no-applicable-method-function (combined-method-gf gf))
        (no-applicable-primary-method gf methods)))))

; Initialized after the initialization (generic) functions exist.
(defvar *initialization-functions-alist* nil)

; This could be in-line above, but I was getting confused.

; ok
(defun make-cnm-combined-method (gf methods method-list keywords)
  (setq gf (combined-method-gf gf))
  (let ((key (cons methods method-list)))
    (or (get-combined-method key gf)
        (let* (key-or-init-arg
               key-or-init-fn)
          (if keywords
            (let* ((bits (inner-lfun-bits gf))
                   (numreq (ldb $lfbits-numreq bits))
                   (key-index (+ numreq (ldb $lfbits-numopt bits))))
              (setq key-or-init-arg (vector key-index keywords gf))
              (setq key-or-init-fn #'x-%%check-keywords))
            (let ((init-cell (assq gf *initialization-functions-alist*)))
              (when init-cell                
                (setq key-or-init-arg init-cell)
                (setq key-or-init-fn #'%%cnm-with-args-check-initargs))))
          (incf *consed-combined-methods*)
          (let* ((vect (vector gf methods key-or-init-arg key-or-init-fn method-list))
                 (self (%cons-combined-method
                        gf vect #'%%cnm-with-args-combined-method-dcode)))
            ;(setf (svref vect 4) self)
            (puthash-combined-method ; if  testing 1 2 3 dont put in our real table
             key
             self))))))


(defparameter *check-call-next-method-with-args* t)

(defun contains-call-next-method-with-args-p (method-list)
  (when *check-call-next-method-with-args*
    (let ((methods method-list)
          method)
      (loop
        (setq method (pop methods))
        (unless methods (return nil))
        (unless (listp method)
          (if (logbitp $lfbits-nextmeth-with-args-bit
                       (lfun-bits (%method-function method)))
            (return t)))))))

; The METHODS arg is a sorted list of applicable methods.
; Returns the method-list expected by %%before-and-after-combined-method-dcode
; or a single method, or NIL if there are no applicable primaries
(defun compute-method-list (methods)
  (let (arounds befores primaries afters qs)
    (dolist (m methods)
      (setq qs (%method-qualifiers m))
      (if qs
        (if (cdr qs)
          (%invalid-method-error
           m "Multiple method qualifiers not allowed in ~s method combination"
           'standard)
          (case (car qs)
            (:before (push m befores))
            (:after (push m afters))
            (:around (push m arounds))
            (t (%invalid-method-error m "~s is not one of ~s, ~s, and ~s."
                                      (car qs) :before :after :around))))
        (push m primaries)))
    (setq primaries (nremove-uncallable-next-methods (nreverse primaries))
          arounds (nremove-uncallable-next-methods (nreverse arounds))
          befores (nreverse befores))      
    (flet ((next-method-bit-p (method)
                              (logbitp $lfbits-nextmeth-bit 
                                       (lfun-bits (%method-function method)))))
      (unless (null primaries)            ; return NIL if no applicable primary methods
        (when (and arounds (not (next-method-bit-p (car (last arounds)))))
          ; Arounds don't call-next-method, can't get to befores, afters, or primaries
          (setq primaries arounds
                arounds nil
                befores nil
                afters nil))
        (if (and (null befores) (null afters)
                 (progn
                   (when arounds
                     (setq primaries (nremove-uncallable-next-methods
                                      (nconc arounds primaries))
                           arounds nil))
                   t)
                 (null (cdr primaries))
                 (not (next-method-bit-p (car primaries))))
          (car primaries)                 ; single method, no call-next-method
          (let ((method-list primaries))
            (if (or befores afters)
              (setq method-list (cons befores (cons afters method-list))))
            (nconc arounds method-list)))))))


; ok 

(defun %invalid-method-error (method format-string &rest format-args)
  (error "~s is an invalid method.~%~?" method format-string format-args))

(defun %method-combination-error (format-string &rest args)
  (apply #'error format-string args))

; ok


(defun combined-method-gf (gf-or-cm)
  (let ((gf gf-or-cm))
    (while (combined-method-p gf)
      (setq gf (lfun-name gf)))
    gf))

(defun nth-arg-dcode-too-few-args (gf-or-cm)
  (error "Too few args to: ~s" (combined-method-gf gf-or-cm)))

(defun nth-arg-combined-method-trap-0 (gf-or-cm table wrapper args)
  (let* ((argnum (%gf-dispatch-table-argnum table))
         (arg (nth argnum args)))
    (nth-arg-combined-method-trap gf-or-cm table argnum args arg wrapper)))

; ok

(defun nth-arg-combined-method-trap (gf-or-cm table argnum args &optional
                                              (arg (nth-or-gf-error 
                                                    argnum args gf-or-cm))
                                              (wrapper (arg-wrapper arg)))
  ; Here when we can't find the method in the dispatch table.
  ; Compute it and add it to the table.  This code will remain in Lisp.
  (multiple-value-bind (combined-method sub-dispatch?)
                       (compute-nth-arg-combined-method
                        gf-or-cm (%gf-dispatch-table-methods table) argnum args
                        wrapper)
    (multiple-value-bind (index obsolete-wrappers-p)
                         ( find-gf-dispatch-table-index table wrapper)
      (if index
        (setf (%gf-dispatch-table-ref table index) wrapper
              (%gf-dispatch-table-ref table (%i+ index 1)) combined-method)
        (grow-gf-dispatch-table gf-or-cm wrapper combined-method obsolete-wrappers-p)))
    (if sub-dispatch?
      (let ((table (%combined-method-methods combined-method)))
        (nth-arg-combined-method-trap
         combined-method
         table
         (%gf-dispatch-table-argnum table)
         args))
      combined-method)))

;; Returns (values combined-method sub-dispatch?)
;; If sub-dispatch? is true, need to compute a combined-method on the
;; next arg.
(defun compute-nth-arg-combined-method (gf methods argnum args &optional 
                                           (wrapper (arg-wrapper
                                                     (nth-or-gf-error
                                                      argnum args gf))))
  (let* ((cpl (%inited-class-cpl (%wrapper-class wrapper)))
         (real-gf (combined-method-gf gf))
         (mc (%gf-method-combination real-gf))
         (standard-mc? (eq mc *standard-method-combination*))
         applicable-methods eql-methods specializers specializer sub-dispatch?)
    (dolist (method methods)
      ;(require-type method 'standard-method)   ; for debugging.
      (setq specializers (nthcdr argnum (%method-specializers method))
            specializer (%car specializers))
      (when (if (listp specializer)
              (when (cpl-memq (%wrapper-class
                                (arg-wrapper (cadr specializer))) cpl)
                (push method eql-methods))
              (when (cpl-memq specializer cpl)
                (push method applicable-methods)))
        (if (contains-non-t-specializer? (%cdr specializers))
          (setq sub-dispatch? t))))
    (if (or eql-methods applicable-methods)
      (if (or (not standard-mc?)
            (contains-primary-method? applicable-methods)
            (contains-primary-method? eql-methods))
        (let ((cpls (args-cpls args)))
          (if eql-methods
            (make-eql-combined-method
             eql-methods applicable-methods cpls gf argnum sub-dispatch? mc)
            (if sub-dispatch?
              (values (make-n+1th-arg-combined-method applicable-methods gf argnum)
                      t)
              (if standard-mc?
                (make-standard-combined-method applicable-methods cpls gf)
                (compute-effective-method-function
                 real-gf mc (sort-methods applicable-methods
                                          (args-cpls args)
                                          (%gf-precedence-list real-gf)))))))
        (no-applicable-primary-method
         real-gf
         (sort-methods (append eql-methods applicable-methods)
                       (args-cpls args)
                       (%gf-precedence-list real-gf))))
       (make-no-applicable-method-function real-gf))))



(defun nth-or-gf-error (n l gf)
  (dotimes (i n) (declare (fixnum i)) (setf l (cdr l)))
  (if (null l)
    (nth-arg-dcode-too-few-args gf))
  (car l))

(defun contains-non-t-specializer? (specializer-list)
  (dolist (s specializer-list nil)
    (unless (eq *t-class* s)
      (return t))))

(defun contains-primary-method? (method-list)
  (dolist (m method-list nil)
    (if (null (%method-qualifiers m))
      (return t))))

(defun args-cpls (args &aux res)
  (dolist (arg args)
    (push (%inited-class-cpl (%wrapper-class (arg-wrapper arg))) res))
  (nreverse res))



;; This needs to be updated to use a linear search in a vector changing to
;; a hash table when the number of entries crosses some threshold.
(defun make-eql-combined-method (eql-methods methods cpls gf argnum sub-dispatch? &optional
                                             (method-combination *standard-method-combination*))
  (let ((eql-ms (copy-list eql-methods))
        (precedence-list (%gf-precedence-list (combined-method-gf gf)))
        (standard-mc? (eq method-combination *standard-method-combination*))
        (real-gf (combined-method-gf gf))
        eql-method-alist
        (can-use-eq? t))
    (unless sub-dispatch?
      (setq methods (sort-methods methods cpls precedence-list)))
    (while eql-ms
      (let ((eql-element (cadr (nth argnum (%method-specializers (car eql-ms)))))
            (this-element-methods eql-ms)
            cell last-cell)
        (if (or (and (numberp eql-element) (not (fixnump eql-element)))
                (macptrp eql-element))
          (setq can-use-eq? nil))
        (setf eql-ms (%cdr eql-ms)
              (%cdr this-element-methods) nil
              cell eql-ms)
        (while cell
          (if (eql eql-element
                     (cadr (nth argnum (%method-specializers (car cell)))))
            (let ((cell-save cell))
              (if last-cell
                (setf (%cdr last-cell) (cdr cell))
                (setq eql-ms (cdr eql-ms)))
              (setf cell (cdr cell)
                    (%cdr cell-save) this-element-methods
                    this-element-methods cell-save))
            (setq last-cell cell
                  cell (cdr cell))))
        (let* ((sorted-methods
                (sort-methods (nreconc (copy-list this-element-methods)
                                       (copy-list methods))
                              cpls
                              precedence-list))
               (method-list (and standard-mc? (compute-method-list sorted-methods))))
          (when (or (not standard-mc?)
                    (memq method-list this-element-methods)
                    (and (consp method-list)
                         (labels ((member-anywhere (tem mlist)
                                    (member tem mlist
                                            :test #'(lambda (tem el)
                                                      (if (listp el)
                                                        (member-anywhere tem el)
                                                        (member el tem))))))
                           (member-anywhere this-element-methods method-list))))
            ; Do EQL comparison only if the EQL methods can run
            ; (e.g. does not come after a primary method that does not call-next-method)
            (push (cons eql-element
                        (if sub-dispatch?
                          (make-n+1th-arg-combined-method
                           sorted-methods gf argnum)
                          (if standard-mc?
                            (make-standard-combined-method sorted-methods nil gf)
                            (compute-effective-method-function
                             real-gf method-combination sorted-methods))))
                  eql-method-alist)))))
    ;;eql-method-alist has (element . combined-method) pairs.
    ;;for now, we're going to use assq or assoc
    (let ((default-method (if sub-dispatch?
                            (make-n+1th-arg-combined-method
                             methods gf argnum)
                            (if standard-mc?
                              (make-standard-combined-method methods nil gf t)
                              (compute-effective-method-function
                               real-gf method-combination methods)))))
      (if eql-method-alist
        (%cons-combined-method 
         gf (cons argnum (cons eql-method-alist default-method))
         (if can-use-eq? 
           #'%%assq-combined-method-dcode
           #'%%assoc-combined-method-dcode)
         t)
        default-method))))

; ok



(DEFun %%assq-combined-method-dcode (stuff args)
  ;; stuff is (argnum eql-method-list . default-method)
  ;(declare (dynamic-extent args))
  (if (listp args)
    (let* ((args-len (list-length args))
           (argnum (car stuff)))
      (when (>= argnum args-len)(Error "Too few args to ~s." (%method-gf (cddr stuff))))
      (let* ((arg (nth argnum args))
             (thing (assq arg (cadr stuff)))) ; are these things methods or method-functions? - fns    
        (if thing 
          (apply (cdr thing) args)
          (apply (cddr stuff) args))))
    (let* ((args-len (%lexpr-count args))
           (argnum (car stuff)))
      (when (>= argnum args-len)(Error "Too few args to ~s." (%method-gf (cddr stuff))))
      (let* ((arg (%lexpr-ref args args-len argnum))
             (thing (assq arg (cadr stuff)))) ; are these things methods or method-functions? - fns    
        (if thing 
          (%apply-lexpr (cdr thing) args)
          (%apply-lexpr (cddr stuff) args))))))
  

(DEFun %%assoc-combined-method-dcode (stuff args)
  ;; stuff is (argnum eql-method-list . default-method)
  ;(declare (dynamic-extent args))
  (if (listp args)
    (let* ((args-len (list-length args))
           (argnum (car stuff)))
      (when (>= argnum args-len)(Error "Too few args to ~s." (%method-gf (cddr stuff))))
      (let* ((arg (nth argnum args))
             (thing (assoc arg (cadr stuff)))) ; are these things methods or method-functions?    
        (if thing 
          (apply (cdr thing) args)
          (apply (cddr stuff) args))))
    (let* ((args-len (%lexpr-count args))
           (argnum (car stuff)))
      (when (>= argnum args-len)(Error "Too few args to ~s." (%method-gf (cddr stuff))))
      (let* ((arg (%lexpr-ref args args-len argnum))
             (thing (assoc arg (cadr stuff)))) ; are these things methods or method-functions?    
        (if thing 
          (%apply-lexpr (cdr thing) args)
          (%apply-lexpr (cddr stuff) args))))))


; Assumes the two methods have the same number of specializers and that
; each specializer of each method is in the corresponding element of cpls
; (e.g. cpls is a list of the cpl's for the classes of args for which both
; method1 & method2 are applicable.
(defun %method< (method1 method2 cpls)
  (let ((s1s (%method-specializers method1))
        (s2s (%method-specializers method2))
        s1 s2 cpl)
    (loop
      (if (null s1s)
        (return (method-qualifiers< method1 method2)))
      (setq s1 (%pop s1s)
            s2 (%pop s2s)
            cpl (%pop cpls))
      (cond ((listp s1) 
             (unless (and (listp s2) (eql (%cadr s1) (%cadr s2)))
               (return t)))
            ((listp s2) (return nil))
            ((eq s1 s2))
            (t (return (%i< (cpl-index s1 cpl) (cpl-index s2 cpl))))))))

(defun %simple-method< (method1 method2 cpl)
  (let ((s1 (%car (%method-specializers method1)))
        (s2 (%car (%method-specializers method2))))
    (cond ((listp s1) 
           (if (and (listp s2) (eql (%cadr s1) (%cadr s2)))
             (method-qualifiers< method1 method2)
             t))
          ((listp s2) nil)
          ((eq s1 s2) (method-qualifiers< method1 method2))
          (t (%i< (cpl-index s1 cpl) (cpl-index s2 cpl))))))

; Sort methods with argument-precedence-order
(defun %hairy-method< (method1 method2 cpls apo)
  (let ((s1s (%method-specializers method1))
        (s2s (%method-specializers method2))
        s1 s2 cpl index)
    (loop
      (if (null apo)
        (return (method-qualifiers< method1 method2)))
      (setq index (pop apo))
      (setq s1 (nth index s1s)
            s2 (nth index s2s)
            cpl (nth index cpls))
      (cond ((listp s1) 
             (unless (and (listp s2) (eql (%cadr s1) (%cadr s2)))
               (return t)))
            ((listp s2) (return nil))
            ((eq s1 s2))
            (t (return (%i< (cpl-index s1 cpl) (cpl-index s2 cpl))))))))

; This can matter if the user removes & reinstalls methods between
; invoking a generic-function and doing call-next-method with args.
; Hence, we need a truly canonical sort order for the methods
; (or a smarter comparison than EQUAL in %%cnm-with-args-check-methods).
(defun method-qualifiers< (method1 method2)
  (labels ((qualifier-list< (ql1 ql2 &aux q1 q2)
              (cond ((null ql1) (not (null ql2)))
                    ((null ql2) nil)
                    ((eq (setq q1 (car ql1)) (setq q2 (car ql2)))
                     (qualifier-list< (cdr ql1) (cdr ql2)))
                    ((string-lessp q1 q2) t)
                    ; This isn't entirely correct.
                    ; two qualifiers with the same pname in different packages
                    ; are not comparable here.
                    ; Unfortunately, users can change package names, hence,
                    ; comparing the package names doesn't work either.
                    (t nil))))
    (qualifier-list< (%method-qualifiers method1) (%method-qualifiers method2))))
       
(defun sort-methods (methods cpls &optional apo)
  (cond ((null cpls) methods)
        ((null (%cdr cpls))
         (setq cpls (%car cpls))
         (flet ((simple-sort-fn (m1 m2)
                  (%simple-method< m1 m2 cpls)))
           (declare (dynamic-extent #'simple-sort-fn))
           (%sort-list-no-key methods #'simple-sort-fn)))
        ((null apo)                     ; no unusual argument-precedence-order
         (flet ((sort-fn (m1 m2) 
                  (%method< m1 m2 cpls)))
           (declare (dynamic-extent #'sort-fn))
           (%sort-list-no-key methods #'sort-fn)))
        (t                              ; I guess some people are just plain rude
         (flet ((hairy-sort-fn (m1 m2)
                  (%hairy-method< m1 m2 cpls apo)))
           (declare (dynamic-extent #'hairy-sort-fn))
           (%sort-list-no-key methods #'hairy-sort-fn)))))

(defun nremove-uncallable-next-methods (methods)
  (do ((m methods (%cdr m))
       mbits)
      ((null m))
    (setq mbits (lfun-bits (%method-function (%car m))))
    (unless (logbitp $lfbits-nextmeth-bit mbits)
      (setf (%cdr m) nil)
      (return)))
  methods)

; ok
; Lap mania struck again - I'm immune
; Often used as a predicate - dont need index
(defun cpl-index (superclass cpl)
  ;; This will be table lookup later.  Also we'll prelookup the tables
  ;; in compute-1st-arg-combined-methods above.
  (locally (declare (optimize (speed 3)(safety 0)))
    (do ((i 0 (%i+ i 1))
         (cpl cpl (%cdr cpl)))
        ((null cpl) nil)
      (if (eq superclass (%car cpl))
        (return i)))))

(defun cpl-memq (superclass cpl)
  (locally (declare (optimize (speed 3)(safety 0)))
    (do ((cpl cpl (%cdr cpl)))
        ((null cpl) nil)
      (if (eq superclass (%car cpl))
        (return cpl)))))

;; Combined method interpretation


; magic is a list of (cnm-cm (methods) . args)
; cnm-cm is the argument checker for call-next-method-with-args or nil
; could make it be a cons as a flag that magic has been heap consed - done
; could also switch car and cadr
; if we do &lexpr business then if cddr is  lexpr-p (aka (not listp)) thats the clue
;  also would need to do lexpr-apply or apply depending on the state.


; per gb - use cons vs. make-list - untested - shorter tho
(defun %%standard-combined-method-dcode (methods  args)
  ; combined-methods as made by make-combined-method are in methods
  ; args are as put there by the caller of the gf.
  ;(declare (dynamic-extent args))
  (let* ((car-meths (car methods))
         (cell-2 (cons methods args))
         (magic (cons nil cell-2)))
    ; i.e. magic is nil methods . args
    (declare (dynamic-extent magic)
             (dynamic-extent cell-2))    
    ;(%rplaca magic nil) ; not needed ? 
    ;(setf (cadr magic) methods)
    ;(%rplaca (cdr magic) methods)
    ;(setf (cddr magic) args)
    ;(%rplacd (cdr magic) args)
    (if (listp car-meths)
      (progn
        (%%before-and-after-combined-method-dcode magic))
      (progn       
        (if (not (cdr methods))
          (%rplaca (cdr magic) car-meths)
          (%rplaca (cdr magic) (cdr methods)))
        ; so maybe its a combined-method ?? - no
        (apply-with-method-context magic (%method-function car-meths) args)))))

; args is list, old-args may be lexpr
(defun cmp-args-old-args (args old-args numreq)
  (declare (optimize (speed 3)(safety 0)))
  (if (listp old-args)
    (do ((newl args (cdr newl))
         (oldl old-args (cdr oldl))
         (i 0 (1+ i)))
        ((eql i numreq) t)
      (when (neq (car newl)(car oldl))(return nil)))
    (let ((len (%lexpr-count old-args)))
      (do ((newl args (cdr newl))
           (i 0 (1+ i)))
          ((eql i numreq) t)
        (when (neq (car newl)(%lexpr-ref old-args len i))(return nil))))))        


; called from call-next-method-with-args with magic supplied and 1st time around with not
(defun %%cnm-with-args-combined-method-dcode (thing args &optional magic) ; was &rest args
  ;(declare (dynamic-extent args))
  ; now thing is vector of gf orig methods, arg for key or initarg check, key or initarg fnction
  ; and our job is to do all the arg checking
  ;(dbg 2048)  ; #x4000
  (let ()
    (when magic
      ;(DBG 2054)  ; #x4030
      (flet ((do-it (thing args)
               (let* ((args-len (length args))
                      (gf (svref thing 0))  ; could get this from a method
                      (numreq (ldb $lfbits-numreq (inner-lfun-bits gf)))
                      (next-methods (cadr magic)))
                 ;(when (null self)(error "Next method with args context error"))
                 (when (neq 0 numreq)
                   ; oh screw it - old-args may be lexpr too
                   (let ((old-args (cddr magic)))
                     (when (< args-len numreq) (error "Too few args to ~S" gf))
                     (when (null (cmp-args-old-args args old-args numreq))
                       ; required args not eq - usually true, we expect
                       (let ((new-methods (%compute-applicable-methods* gf args))
                             (old-methods (svref thing 1)))
                         (when (not (equal new-methods old-methods))
                           (error '"Applicable-methods changed in call-next-method.~%~
                                    Should be: ~s~%Was: ~s~%Next-methods: ~s"
                                  old-methods new-methods next-methods))))))
                 (let ((key-or-init-fn (svref thing 3)))
                   (when key-or-init-fn 
                     ; was apply
                     (funcall key-or-init-fn (svref thing 2) args))))))
        (if (listp args)
          (do-it thing args)
          (with-list-from-lexpr (args-list args)
            (do-it thing args-list)))))
    ; ok done checking - lets do it 
    ;(dbg 2051)(dbg magic)  ; #x4018  ; 1st time - magic is nil
    (let* ((methods (if magic (cadr magic)(svref thing 4)))  ;<< was 5 this is nil unless cnm with args
           ; was if magic
           (car-meths (car methods))
           (cell-2 (cons methods args))
           (magic (cons thing cell-2)))
      (declare (dynamic-extent magic cell-2))
      ; i.e. magic is thing methods . args
      ;(DBG METHODS)
      ;(%rplaca magic thing)
      ;(setf (cadr magic) methods)
      ;(%rplaca (cdr magic) methods)
      ;(setf (cddr magic) args)
      ;(%rplacd (cdr magic) args)
      (if (listp car-meths)
        (progn
          (%%before-and-after-combined-method-dcode magic))
        (progn       
          (if (not (cdr methods))
            (%rplaca (cdr magic) car-meths)
            (%rplaca (cdr magic) (cdr methods)))
          ; so maybe its a combined-method ?? - no
          (apply-with-method-context magic (%method-function car-meths) args))))))



; here if car of methods is listp. methods = (befores afters . primaries)
(defun %%before-and-after-combined-method-dcode (magic) 
  (declare (list magic))
  ;(dbg 4104)
  (let* ((methods (cadr magic))         
         (befores (car methods))         
         (cdr-meths (cdr methods))
         (primaries (cdr cdr-meths))
         (afters (car cdr-meths))
         (args (cddr magic)))
    (declare (list befores afters primaries))
    (when befores 
      (dolist (method befores)
        (rplaca (cdr magic) method)
        (apply-with-method-context magic (%method-function method) args)))
    (let* ((cdr (cdr primaries))
           (method-function (%method-function (car primaries))))   ; guaranteed non nil?
      (rplaca (cdr magic) (if (null cdr)(car primaries) cdr))      
      (if (null afters)
        (apply-with-method-context magic method-function args)  ; tail call if possible
        (multiple-value-prog1
          (apply-with-method-context magic method-function args)        
          (dolist (method afters)
            (rplaca (cdr magic) method)
            (apply-with-method-context magic (%method-function method) args)))))))


; This is called by the compiler expansion of next-method-p
; I think there's a bug going around... LAP fever! I'm immune
(defun %next-method-p (magic)
  (let ((methods (%cadr magic)))
    (consp methods)))


(defun %call-next-method (magic &rest args) ; if args supplied they are new ones
  (declare (dynamic-extent args)) 
  ;(dbg-paws 2045 magic args)
  (if args
    (apply #'%call-next-method-with-args magic args)
    (let* ((next-methods (%cadr magic))) ; don't get this closed magic stuff      
      (if (not (consp next-methods))
        ( %no-next-method  magic)            
        (let ((args (%cddr magic)))  ; get original args
          ;The unwind-protect is needed in case some hacker in his/her wisdom decides to:
          ; (defmethod foo (x) (catch 'foo (call-next-method)) (call-next-method))
          ; where the next-method throws to 'foo.
          ; The alternative is to make a new magic var with args
          ; actually not that fancy (call-next-method)(call-next-method) is same problem
          (let ()
            ;(dbg-paws 2046 magic args)
            (unwind-protect
              (if (listp (car next-methods))
                ( %%before-and-after-combined-method-dcode magic)
                (let ((cdr (cdr next-methods)))
                  (rplaca (cdr magic)(if (not cdr)(car next-methods) cdr))
                  (let ((method-function (%method-function (car next-methods))))
                    (apply-with-method-context magic method-function args))))
              (rplaca (cdr magic) next-methods))))))))

;; Note: we need to change the compiler to call this when it can prove that
;; call-next-method cannot be called a second time. I believe thats done.


(defun %tail-call-next-method (magic)
  (let* ((next-methods (%cadr magic))  ; or make it car
         (args (%cddr magic))) ; get original args        
    (if (not (consp next-methods)) ; or consp?
      ( %no-next-method magic)
      (if (listp (car next-methods))
        ( %%before-and-after-combined-method-dcode magic)
        (let ((cdr (cdr next-methods)))
          (rplaca (cdr magic) (if (not cdr)(car next-methods) cdr))
          (apply-with-method-context magic (%method-function (car next-methods)) args))))))

; may be simpler to blow another cell so magic looks like
; (cnm-cm/nil next-methods . args) - done
; and also use first cell to mean heap-consed if itsa cons

(defun %call-next-method-with-args (magic &rest args)
  (declare (dynamic-extent args))
  ;(dbg 2050)  ; #x4010
  (if (null args)
    (%call-next-method magic)
    (let* ((methods (%cadr magic)))
      (if (not (consp methods))
        (%no-next-method  magic)
        (let* ((cnm-cm (car magic)))
          ; a combined method
          (when (consp cnm-cm)(setq cnm-cm (car cnm-cm)))
          ;(dbg 2049)(dbg cnm-cm)  ; #x4008
          ; could just put the vector in car magic & no self needed in vector?
          (let ((the-vect cnm-cm)) ;  <<
            (funcall #'%%cnm-with-args-combined-method-dcode ;(%combined-method-dcode cnm-cm)
                     the-vect
                     args
                     magic)))))))



; called from x%%call-next-method-with-args - its the key-or-init-fn 
(defun %%cnm-with-args-check-initargs (init-cell args)
  ; here we forget the lexpr idea because it wants to cdr
  ;(declare (dynamic-extent args))
  (let* ((rest (cdr args))
         (first-arg (car args)))
    (declare (list rest))
    (let* ((initargs rest)
           (init-function (car init-cell))
           (instance (cond ((eq init-function #'update-instance-for-different-class)
                            (setq initargs (cdr rest))
                            (car rest))
                           ((eq init-function #'shared-initialize)
                            (setq initargs (cdr rest))
                            first-arg)
                           ((eq init-function #'update-instance-for-redefined-class)
                            (setq initargs (%cdddr rest))
                            first-arg)
                           (t first-arg)))
           (class (class-of instance))
           bad-initarg)
      (dolist (functions (cdr init-cell)
                         (error "Bad initarg: ~s to call-next-method for ~s~%on ~s"
                                bad-initarg instance (car init-cell)))
        (multiple-value-bind 
          (errorp bad-key)
          (if (eq (car functions) #'initialize-instance)
            (apply #'check-initargs instance class initargs nil
                   #'initialize-instance #'allocate-instance #'shared-initialize
                   (aux-init-functions class))
            (apply #'check-initargs instance class initargs nil functions))
          (if errorp
            (unless bad-initarg (setq bad-initarg bad-key))
            (return t)))))))



(defun %no-next-method (magic)
  (let* ((method (%cadr magic)))
    ;(dbg-paws 333)
    (if (consp method) (setq method (car method)))
    (unless (typep method 'standard-method)
      (error "call-next-method called outside of generic-function dispatch context.~@
              Usually indicates an error in a define-method-combination form."))
    (let ((args (cddr magic))
          (gf (%method-gf method)))
      (if (listp args)
        (apply #'no-next-method gf method args)
        (%apply-lexpr #'no-next-method gf method args)))))




;; This makes a consed version of the magic first arg to a method.
;; Called when someone closes over the magic arg. (i.e. does (george #'call-next-method))

(defun %cons-magic-next-method-arg (magic)
  ; car is a cons as a flag that its already heap-consed! - else cnm-cm or nil
  (if (consp (car magic))
    magic
    (list* (list (car magic))
           (if (consp (%cadr magic))
             (copy-list (%cadr magic)) ; is this copy needed - probably not
             (cadr magic))
           (let ((args (%cddr magic)))
             (if (listp args)
               (copy-list args)
               (let* ((len (%lexpr-count args))
                      (l (make-list len)))
                 (do ((i 0 (1+ i))
                      (list l (cdr list)))
                     ((null list))
                   (%rplaca list (%lexpr-ref args len i)))
                 l))))))





(defun %%reader-dcode-no-lexpr (dt arg)
  (locally (declare (optimize (speed 3)(safety 0)))
    (let* ((gf (%gf-dispatch-table-gf dt))
           (instance (if (%standard-instance-p arg) arg (%maybe-gf-instance arg))))
      (unless instance
        (let ((args-list (make-list 1 :initial-element arg)))
          (declare (dynamic-extent args-list))
          (reader-trap-no-method gf args-list)))
      (let* ((wrapper (%instance-class-wrapper instance)))
        (when (eql 0 (%wrapper-hash-index wrapper))
          (update-obsolete-instance arg)
          (setq wrapper (%instance-class-wrapper instance))
          (setq dt (%gf-dispatch-table gf))) ; may have changed
        (let* ((mask (%gf-dispatch-table-mask dt))
               (index (%ilogand mask (%wrapper-hash-index wrapper)))
               table-wrapper flag)
          (declare (fixnum index mask))
          ;(print (list 'first-index index wrapper))
          (setq index (+ index index)) ; +2 ??
          (loop 
            (if (eq (setq table-wrapper (%gf-dispatch-table-ref dt index)) wrapper)
              (let* ((the-pos (%gf-dispatch-table-ref dt (the fixnum (1+ index)))))
                (if (fixnump the-pos)
                  (locally (declare (fixnum the-pos))
                    (let* ((the-val (%svref (%forwarded-instance instance) (the fixnum (+ the-pos 2)))))
                      (if (eq the-val (%unbound-marker-8))
                        (return (slot-unbound 
                                 (%wrapper-class wrapper) 
                                 arg 
                                 (%svref (%wrapper-instance-slots wrapper) the-pos)))
                        (return the-val))))
                  (let ((the-val (cadr the-pos)))
                    (if (eq the-val (%unbound-marker-8))
                      (return (slot-unbound (%wrapper-class wrapper) arg (car the-pos)))
                      (return the-val)))))
              ; shit after here re class slots etc., forwarded instance
              (progn
                (when (null (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                  (if (or (neq table-wrapper (%unbound-marker-8))
                          (eq 0 flag))
                    (without-interrupts
                     (return
                      ; this is silly
                      (let ((args-list (make-list 1 :initial-element arg)))
                        (declare (dynamic-extent args-list))
                        (reader-trap-2 gf wrapper arg args-list))))
                    (setq flag 0 index -2)))
                (setq index (+ 2 index))))))))))

(defun %%writer-dcode-no-lexpr (dt arg0 arg)
  (locally (declare (optimize (speed 3)(safety 0)))
    (let* ((gf (%gf-dispatch-table-gf dt))
           (instance (if (%standard-instance-p arg) arg (%maybe-gf-instance arg))))
      (unless instance
        (let ((args-list (make-list 2)))
          (declare (dynamic-extent args-list))
          (%rplaca args-list arg0)
          (%rplaca (cdr args-list) arg)
          (reader-trap-no-method gf args-list)))
      (let* ((wrapper (%instance-class-wrapper instance)))
        (when (eql 0 (%wrapper-hash-index wrapper))
          (update-obsolete-instance arg)
          (setq wrapper (%instance-class-wrapper instance))
          (setq dt (%gf-dispatch-table gf))) ; may have changed
        (let* ((mask (%gf-dispatch-table-mask dt))
               (index (%ilsl 1 (%ilogand mask (%wrapper-hash-index wrapper))))
               table-wrapper flag)
          (declare (fixnum index mask))
          (loop 
            (if (neq (setq table-wrapper (%gf-dispatch-table-ref dt index)) wrapper)
              (progn
                (when (null (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                  (if (or (neq table-wrapper (%unbound-marker-8))
                          (eql 0 flag))
                    (without-interrupts
                     (return
                      (let ((args-list (make-list 2)))
                        (declare (dynamic-extent args-list))
                        (%rplaca args-list arg0)
                        (%rplaca (cdr args-list) arg)
                        (reader-trap-2 gf wrapper arg args-list))))
                    (setq flag 0 index -2)))
                (setq index (+ 2 index)))
              (let* ((the-pos (%gf-dispatch-table-ref dt (the fixnum (1+ index))))
                     (the-val arg0))
                (if (fixnump the-pos)
                  (locally (declare (fixnum the-pos))
                    (return (setf (%svref (%forwarded-instance instance) (the fixnum (+ the-pos 2)))
                                  the-val)))
                  (return (setf (%cadr the-pos) the-val)))))))))))

(defun reader-trap-no-method (gf args)
  (if (listp args)
    (apply #'no-applicable-method gf args)
    (apply #'no-applicable-method gf (collect-lexpr-args args 0))))
    

(defun reader-trap-2 (gf wrapper instance args)
  ;(declare (dynamic-extent args))
  (let ((method (and wrapper (car (compute-applicable-methods gf args)))))
    (if (not method)
      (apply #'no-applicable-method gf args)
      (let ((slots (%wrapper-instance-slots wrapper))
            (slot-name (method-slot-name method)))
        (when (eql 0 slots)
          (error "Obsolete instance in reader-trap-2"))
        (let ((idx (or (%vector-member slot-name slots)
                       (assq slot-name (%wrapper-class-slots wrapper)))))
          (unless idx
            (error "~s has no slot named ~s" instance slot-name))
          (let ((table (%gf-dispatch-table gf)))
            (multiple-value-bind (index obsolete-wrappers-p)
                                 (find-gf-dispatch-table-index table wrapper)
              (if index
                (setf (%gf-dispatch-table-ref table index) wrapper
                      (%gf-dispatch-table-ref table (%i+ index 1)) idx)
                (grow-gf-dispatch-table gf wrapper idx obsolete-wrappers-p))))
          (apply (%method-function method) args))))))


; Support CALL-METHOD in DEFINE-METHOD-COMBINATION
(defun %%call-method* (method next-methods args)
  (let* ((method-function (%method-function method))
         (bits (lfun-bits method-function)))
    (declare (fixnum bits))
    (if (not (and (logbitp $lfbits-nextmeth-bit  bits)
                  (logbitp  $lfbits-method-bit bits)))
      (if (listp args)
        (apply method-function args)
        (%apply-lexpr method-function args))
      (let* ((cell-2 (cons next-methods args))
             (magic (cons nil cell-2)))
        (declare (dynamic-extent magic)
                 (dynamic-extent cell-2))  
        (if (null next-methods)
          (%rplaca (cdr magic) method))
        (apply-with-method-context magic method-function args)))))

; Error checking version for user's to call
(defun %call-method* (method next-methods args)
  (let* ((method-function (%method-function method))
         (bits (lfun-bits method-function)))
    (declare (fixnum bits))
    (if (not (and (logbitp $lfbits-nextmeth-bit  bits)
                  (logbitp  $lfbits-method-bit bits)))
      (progn
        (require-type method 'standard-method)
        (if (listp args)
          (apply method-function args)
          (%apply-lexpr method-function args)))
      (progn
        (do* ((list next-methods (cdr list)))
             ((null list))
          (when (not (listp list))
            (%err-disp $XIMPROPERLIST next-methods))
          (when (not (standard-method-p (car list)))
            (report-bad-arg (car list) 'standard-method))) 
        (let* ((cell-2 (cons next-methods args))
               (magic (cons nil cell-2)))
          (declare (dynamic-extent magic)
                   (dynamic-extent cell-2))  
          (if (null next-methods)
            (%rplaca (cdr magic) method))
          (apply-with-method-context magic method-function args))))))



