;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  
;;;
;;;   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
;;;   License , known as the LLGPL and distributed with OpenMCL as the
;;;   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
;;;   which is distributed with OpenMCL as the file "LGPL".  Where these
;;;   conflict, the preamble takes precedence.  
;;;
;;;   OpenMCL is referenced in the preamble as the "LIBRARY."
;;;
;;;   The LLGPL is also available online at
;;;   http://opensource.franz.com/preamble.html

; l1-readloop-lds.lisp

(in-package "CCL")



  




(defun continuable-process-p ()
  (without-interrupts
   (dolist (p *active-processes*)
     (let ((sg (process-stack-group p)))
       (when (symbol-value-in-stack-group '*continuablep* sg)
         (return p))))))

(defun continuable-processes ()
  (without-interrupts
   (let ((result))
     (dolist (process *active-processes*)
         (when  (symbol-value-in-process '*continuablep* process)
           (push process result)))
     result)))

; this isnt quite right but requiring *in-read-loop* isnt right either.
(defun restartable-process-p ()
  (without-interrupts
   (dolist (p *active-processes*)
     (when (neq 0 (symbol-value-in-process '*break-level* p))
       (return p)))))

(defun restartable-processes ()
  (without-interrupts
   (let ((result))
     (dolist (process *active-processes*)
       (when (neq 0 (symbol-value-in-process '*break-level* process))
         (push process result)))
     result)))

(defun process-to-continue ()
  (let ((ps (continuable-processes)))
    (case (length ps)
      (0 nil)
      (1 (car ps))
      (t (let ((p (catch-cancel (select-item-from-list ps :window-title "Continue Process"))))
           (if (neq p :cancel) (car p)))))))

(defun process-to-restart (&optional (string "Restart Process"))
  (let ((ps (restartable-processes)))
    (case (length ps)
      (0 nil)
      (1 (car ps))
      (t (let ((p (catch-cancel (select-item-from-list ps :window-title string))))
           (if (neq p :cancel) (car p)))))))




(defun toplevel-loop ()
  (loop
    (if (eq (catch :toplevel 
              (read-loop :break-level 0)) $xstkover)
      (format t "~&;[Stacks reset due to overflow.]")
      (when (eq *current-process* *initial-process*)
        (toplevel)))))


(defvar *defined-toplevel-commands* ())
(defvar *active-toplevel-commands* ())

(defun %define-toplevel-command (group-name key name fn doc args)
  (let* ((group (or (assoc group-name *defined-toplevel-commands*)
		    (car (push (list group-name)
			       *defined-toplevel-commands*))))
	 (pair (assoc key (cdr group) :test #'eq)))
    (if pair
      (rplacd pair (list* fn doc args))
      (push (cons key (list* fn doc args)) (cdr group))))
  name)

(define-toplevel-command 
    :global y (p) "Yield control of terminal-input to process whose name or ID is <p>"
    (if (not (eq *current-process* *initial-process*))
      (format t "~&;; only the initial procss can yield control of the terminal.")
      (let* ((proc (find-process p)))
	(if (null proc)
	  (format t "~&;; not found - ~s" p)
	  (if (not (member proc *terminal-input-requests*))
	    (format t "~&;; process not requesting control of terminal input")
	    (%%yield-terminal-to proc))))))

(define-toplevel-command
    :global kill (p) "Kill process whose name or ID matches <p>"
    (let* ((proc (find-process p)))
      (if p
	(process-kill proc))))

(define-toplevel-command 
    :global proc (&optional p) "Show information about specified process <p>/all processes"
    (flet ((show-process-info (proc)
	     (format t "~&~d : ~a ~a ~20t[~a] "
		     (process-serial-number proc)
		     (if (eq proc *current-process*)
		       "->"
		       "  ")
		     (process-name proc)
		     (process-whostate proc))
	     (if (process-active-p proc)
	       (format t " (Active)"))
	     (if (member proc *terminal-input-requests*)
	       (format t " (Requesting terminal input)"))
	     (fresh-line)))
      (if p
	(let* ((proc (find-process p)))
	  (if (null proc)
	    (format t "~&;; not found - ~s" p)
	    (show-process-info proc)))
	(dolist (proc *all-processes* (values))
	  (show-process-info proc)))))

(define-toplevel-command :break pop () "exit current break loop" (abort-break))
(define-toplevel-command :break go () "continue" (continue))
(define-toplevel-command :break q () "return to toplevel" (toplevel))
(define-toplevel-command :break r () "list restarts"
  (let* ((r (apply #'vector (compute-restarts *break-condition*))))
    (dotimes (i (length r) (terpri))
      (format t "~&~d. ~a" i (svref r i)))))

;;; From Marco Baringer 2003/03/18
(define-toplevel-command :break set (n frame value) "Set <n>th item of frame <frame> to <value>"
  (let* ((frame-sp (nth-raw-frame frame *break-frame* *current-stack-group*)))
    (if frame-sp
        (toplevel-print (list (set-nth-value-in-frame frame-sp n *current-stack-group* value)))
        (format *debug-io* "No frame with number ~D~%" frame))))

;;;; this is the one i've found the most usefull since from within
;;;; the inspector you can change the data on the stack
(define-toplevel-command :break iv (n frame) "Inspect the <n>th item of frame <frame>"
  (let* ((frame-sp (nth-raw-frame frame *break-frame* *current-stack-group*)))
    (if frame-sp
        (inspect (nth-value-in-frame frame-sp n *current-stack-group*))
        (format *debug-io* "No frame with number ~D~%" frame))))

(define-toplevel-command :break d (n frame) "Describe the <n>th item of frame number <frame>"
  (let* ((frame-sp (nth-raw-frame frame *break-frame* *current-stack-group*)))
    (if frame-sp
        (describe (nth-value-in-frame frame-sp n *current-stack-group*))
        (format *debug-io* "No frame with number ~D~%" frame))))


(define-toplevel-command :global ? () "help"
  (dolist (g *active-toplevel-commands*)
    (dolist (c (cdr g))
      (let* ((command (car c))
	     (doc (caddr c))
	     (args (cdddr c)))
	(if args
	  (format t "~& (~S~{ ~A~}) ~8T~A" command args doc)
	  (format t "~& ~S  ~8T~A" command doc))))))


(define-toplevel-command :break b (&optional show-frame-contents) "backtrace"
  (when *break-frame*
    (print-call-history :detailed-p show-frame-contents
                        :start-frame *break-frame*)))

(define-toplevel-command :break c (n) "Choose restart <n>"
   (select-restart n))

(define-toplevel-command :break f (n) "Show backtrace frame <n>"
   (print-call-history :start-frame *break-frame*
                       :detailed-p n))

(define-toplevel-command :break v (n frame-number) "Return value <n> in frame <frame-number>"
  (let* ((frame-sp (nth-raw-frame frame-number *break-frame* *current-stack-group*)))
    (if frame-sp
      (toplevel-print (list (nth-value-in-frame frame-sp n *current-stack-group*))))))

(defun %use-toplevel-commands (group-name)
  ;; Push the whole group
  (pushnew (assoc group-name *defined-toplevel-commands*)
	   *active-toplevel-commands*
	   :key #'(lambda (x) (car x))))  ; #'car not defined yet ...

(%use-toplevel-commands :global)

(defun check-toplevel-command (form)
  (let* ((cmd (if (consp form) (car form) form))
         (args (if (consp form) (cdr form))))
    (if (keywordp cmd)
      (dolist (g *active-toplevel-commands*)
	(when
	    (let* ((pair (assoc cmd (cdr g))))
	      (if pair 
		(progn (apply (cadr pair) args)
		       t)))
	  (return t))))))

;This is the part common to toplevel loop and inner break loops.
(defun read-loop (&key (break-level *break-level*)
		       (prompt-function #'(lambda () (print-listener-prompt t)))
		       (input-stream *terminal-io*)
		       (output-stream *terminal-io*))
  (let* ((*break-level* break-level)
         (*last-break-level* break-level)
         *loading-file-source-file*
         *in-read-loop*
         (*listener-p* t)
         *** ** * +++ ++ + /// // / -
         form)
    (loop
      (restart-case
        (catch :abort ;last resort...
          (loop
            (catch-cancel
              (loop                
                (setq *loading-file-source-file* nil
                      *in-read-loop* nil
                      *break-level* break-level)
                (setq form (toplevel-read :input-stream input-stream
					  :output-stream output-stream
					  :prompt-function prompt-function))
                (if (eq form *eof-value*)
		  (if (eof-transient-p (stream-device *stdin* :input))
		    (progn
		      (stream-clear-input *terminal-io*)
		      (abort-break))
		    (quit))
                  (with-non-background-process
		    (or (check-toplevel-command form)
			(toplevel-print
			 (toplevel-eval form)))))))
            (format *terminal-io* "~&Cancelled")))
        (abort () :report (lambda (stream)
                            (if (eq break-level 0)
                              (format stream "Return to toplevel.")
                              (format stream "Return to break level ~D." break-level)))
               #| ; Handled by interactive-abort
                ; go up one more if abort occurred while awaiting/reading input               
                (when (and *in-read-loop* (neq break-level 0))
                  (abort))
                |#
               )
        (abort-break () 
                     (unless (eq break-level 0)
                       (abort))))
      (clear-input *terminal-io*)
      (format *terminal-io* "~%"))))

;Read a form from *terminal-io*.
(defun toplevel-read (&key (input-stream *standard-input*)
			   (output-stream *standard-output*)
			   (prompt-function #'print-listener-prompt))
  (let* ((listener input-stream))
    (force-output output-stream)
    (funcall prompt-function)
    (loop
        (let* ((*in-read-loop* nil)  ;So can abort out of buggy reader macros...
               (form))
          (catch '%re-read            
            (if (eq (setq form (read listener nil *eof-value*)) *eof-value*)
              (return form)
	      (progn
		(let ((ch)) ;Trim whitespace
		  (while (and (listen listener)
			      (setq ch (tyi listener))
			      (whitespacep cH))
		    (setq ch nil))
		  (when ch (untyi ch listener)))
		(when *listener-indent* 
		  (tyo #\space listener)
		  (tyo #\space listener))
		(return (process-single-selection form)))))))))

(defvar *always-eval-user-defvars* nil)

(defun process-single-selection (form)
  (if (and *always-eval-user-defvars*
           (listp form) (eq (car form) 'defvar) (cddr form))
    `(defparameter ,@(cdr form))
    form))

(defun toplevel-eval (form &optional env &aux values)
   (declare (resident))
  (setq +++ ++ ++ + + - - form)
  (setq values (multiple-value-list (cheap-eval-in-environment form env)))
  values)

(defun toplevel-print (values)
  (declare (resident))
  (setq /// // // / / values)
  (setq *** ** ** * * (if (neq (%car values) (%unbound-marker-8)) (%car values)))
  (when values
    (fresh-line)
    (dolist (val values) (write val) (terpri))))

(defun print-listener-prompt (&optional (force t))
  (when (or force (neq *break-level* *last-break-level*))
    (let* ((*listener-indent* nil))
      (fresh-line *terminal-io*)            
      (if (%izerop *break-level*)
        (%write-string "?" *terminal-io*)
        (format *terminal-io* "~s >" *break-level*)))        
    (write-string " " *terminal-io*)        
    (setq *last-break-level* *break-level*))
      (force-output *terminal-io*))


;;; Fairly crude default error-handlingbehavior, and a fairly crude mechanism
;;; for customizing it.

(defvar *app-error-handler-mode* :quit
  "one of :quit, :quit-quietly, :listener might be useful.")

(defmethod application-error ((a application) condition error-pointer)
  (case *app-error-handler-mode*
    (:listener   (break-loop-handle-error condition error-pointer))
    (:quit-quietly (quit -1))
    (:quit  (format t "~&Fatal error in ~s : ~a"
                    (pathname-name (car *command-line-argument-list*))
                    condition)
                    (quit -1))))

(defun make-application-error-handler (app mode)
  (declare (ignore app))
  (setq *app-error-handler-mode* mode))


; You may want to do this anyway even if your application
; does not otherwise wish to be a "lisp-development-system"
(defmethod application-error ((a lisp-development-system) condition error-pointer)
  (break-loop-handle-error condition error-pointer))

(defun break-loop-handle-error (condition error-pointer)
  (multiple-value-bind (bogus-globals newvals oldvals) (%check-error-globals)
    (dolist (x bogus-globals)
      (set x (funcall (pop newvals))))
    (when (and *debugger-hook* *break-on-errors*)
      (let ((hook *debugger-hook*)
            (*debugger-hook* nil))
        (funcall hook condition hook)))
    (with-terminal-input
	(%break-message (error-header "Error") condition error-pointer)
      (let* ((s *error-output*))
	(dolist (bogusness bogus-globals)
	  (let ((oldval (pop oldvals)))
	    (format s "~&;  NOTE: ~S was " bogusness)
	    (if (eq oldval (%unbound-marker-8))
	      (format s "unbound")
	      (format s "~s" oldval))
	    (format s ", was reset to ~s ." (symbol-value bogusness)))))
      (if *break-on-errors*
	(break-loop condition error-pointer)
	(abort)))))

(defun break (&optional string &rest args &aux (fp (%get-frame-ptr)))
  (flet ((do-break-loop ()
           (let ((c (make-condition 'simple-condition
                                    :format-control (or string "")
                                    :format-arguments args)))
             (cbreak-loop (error-header "Break") "Return from BREAK." c fp))))
    (cond ((%i> *interrupt-level* -1)
           (do-break-loop))
          (*break-loop-when-uninterruptable*
           (format *error-output* "Break while *interrupt-level* less than zero; binding to 0 during break-loop.")
           (let ((*interrupt-level* 0))
             (do-break-loop)))
          (t (format *error-output* "Break while *interrupt-level* less than zero; ignored.")))))

(defun invoke-debugger (&optional string &rest args &aux (fp (%get-frame-ptr)))
  (let ((c (condition-arg (or string "") args 'simple-condition)))
    (when *debugger-hook*
      (let ((hook *debugger-hook*)
            (*debugger-hook* nil))
        (funcall hook c hook)))
    (with-terminal-input
	(%break-message "Debug" c fp)
      (break-loop c fp))))

(defun %break-message (msg condition error-pointer &optional (prefixchar #\>))
  (let ((*print-circle* *error-print-circle*)
        ;(*print-pretty* nil)
        (*print-array* nil)
        (*print-escape* t)
        (*print-gensym* t)
        (*print-length* nil)  ; ?
        (*print-level* nil)   ; ?
        (*print-lines* nil)
        (*print-miser-width* nil)
        (*print-readably* nil)
        (*print-right-margin* nil)
        (*signal-printing-errors* nil)
        (s (make-indenting-string-output-stream prefixchar nil)))
    (format s "~A ~A: " prefixchar msg)
    (setf (indenting-string-output-stream-indent s) (column s))
    ;(format s "~A" condition) ; evil if circle
    (report-condition condition s)
    (if (not (and (typep condition 'simple-program-error)
                  (simple-program-error-context condition)))
      (format *error-output* "~&~A~%~A While executing: ~S~%"
              (get-output-stream-string s) prefixchar (%real-err-fn-name error-pointer))
      (format *error-output* "~&~A~%"
              (get-output-stream-string s)))
  (force-output *error-output*)))
					; returns NIL

(defun cbreak-loop (msg cont-string condition error-pointer)
  (let* ((*print-readably* nil))
    (with-terminal-input
	(%break-message msg condition error-pointer)
      (restart-case (break-loop condition error-pointer *backtrace-on-break*)
		    (continue () :report (lambda (stream) (write-string cont-string stream))))
      (fresh-line *error-output*)
      nil)))

(defun warn (format-string &rest args)
  (let ((fp (%get-frame-ptr))
        (c (require-type (condition-arg format-string args 'simple-warning) 'warning)))
    (when *break-on-warnings*
      (cbreak-loop "Warning" "Signal the warning." c fp))
    (restart-case (signal c)
      (muffle-warning () :report "Skip the warning" (return-from warn nil)))
    (%break-message (if (typep c 'compiler-warning) "Compiler warning" "Warning") c fp #\;)
    ))

(declaim (notinline select-backtrace))

(defmacro new-backtrace-info (dialog youngest oldest stack-group)
  `(vector ,dialog ,youngest ,oldest ,stack-group nil))

(defun select-backtrace ()
  (declare (notinline select-backtrace))
  (require 'new-backtrace)
  (require :inspector)
  (select-backtrace))

(defvar *break-condition* nil "condition argument to innermost break-loop.")
(defvar *break-frame* nil "frame-pointer arg to break-loop")
(defvar *break-loop-when-uninterruptable* t)




(eval-when (:compile-toplevel :execute :load-toplevel)

(unless (fboundp 'databases-locked-p)

; Redefined by Wood
(defun databases-locked-p (&optional by-locker)
  (declare (ignore by-locker))
  nil)

)  ; end of unless

(unless (fboundp 'funcall-with-databases-unlocked)

; Redefined by Wood
(defun funcall-with-databases-unlocked (thunk)
  (funcall thunk))

))  ; end of unless and eval-when

(defmacro with-databases-unlocked (&body body)
  (let ((thunk (gensym)))
    `(let ((,thunk #'(lambda () ,@body)))
       (declare (dynamic-extent ,thunk))
       (funcall-with-databases-unlocked ,thunk))))

(defvar %last-continue% nil)
(defun break-loop (condition frame-pointer
                             &optional (backtracep *backtrace-on-break*))
  "Never returns"
  (when (and (%i< *interrupt-level* 0) (not *break-loop-when-uninterruptable*))
    (abort))
  (let* ((%handlers% nil)		; firewall
         (databases-locked-p (databases-locked-p *current-process*))
         (*break-frame* frame-pointer)
         (*break-condition* condition)
         (*compiling-file* nil)
         (*backquote-stack* nil)
         (continue (find-restart 'continue))
         (*continuablep* (unless (eq %last-continue% continue) continue))
         (%last-continue% continue)
         (*standard-input* *debug-io*)
         (*standard-output* *debug-io*)
         (*interrupt-level* 0)
         (*signal-printing-errors* nil)
         (*read-suppress* nil)
         (*print-readably* nil))

    (with-databases-unlocked
	(with-toplevel-commands :break
	  (if *continuablep*
	    (let* ((*print-circle* *error-print-circle*)
					;(*print-pretty* nil)
		   (*print-array* nil))
	      (format t "~&> Type :GO to continue, :POP to abort.")
	      (format t "~&> If continued: ~A~%" continue))
	    (format t "~&> Type :POP to abort.~%"))
	  (format t "~&Type :? for other options.")
	  (terpri)
	  (when databases-locked-p
	    (format t "> WARNING: The WOOD database lock was locked.~@
                           > Use caution or you might damage your database.~%"))
	  (force-output)
	  (when backtracep
	    (select-backtrace))
	  (clear-input *debug-io*)
	  (setq *error-reentry-count* 0) ; succesfully reported error
	  (read-loop :break-level (1+ *break-level*))))))



(defun display-restarts (&optional (condition *break-condition*))
  (let ((i 0))
    (format t "~&[Pretend that these are buttons.]")
    (dolist (r (compute-restarts condition) i)
      (format t "~&~a : ~A" i r)
      (setq i (%i+ i 1)))
    (fresh-line nil)))

(defun select-restart (n &optional (condition *break-condition*))
  (let* ((restarts (compute-restarts condition)))
    (invoke-restart-interactively
     (nth (require-type n `(integer 0 (,(length restarts)))) restarts))))




; End of l1-readloop-lds.lisp
