/*
 * conference.cxx
 *
 * Conferencing functions for a simple MCU
 *
 * Copyright (c) 2000 Equivalence Pty. Ltd.
 * Copyright (c) 2004 Post Increment
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of ths code were written by by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Stonevoice, slc. http://www.stonevoice.com
 *
 * Portions of this code were written by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Citron Networks (http://www.citron.com.tw)
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 Craig Southeren (craig@postincrement.com)
 *
 * $Log: conference.cxx,v $
 * Revision 1.4  2009/05/26 18:16:04  willamowius
 * fix gcc 4.3.x warning
 *
 * Revision 1.3  2008/05/23 11:18:12  willamowius
 * switch BOOL to PBoolean to be able to compile with Ptlib 2.2.x
 *
 * Revision 1.2  2008/02/19 06:36:11  shorne
 * Added ability to have 16CIF & 4CIF size video. Thx Marcos Jardini for the initial 4CIF patch
 *
 * Revision 1.1  2007/10/17 19:44:28  shorne
 * Initial Commit
 *
 * Revision 2.12  2006/08/07 06:18:22  csoutheren
 * Fix detection of MCU members
 * Add auto-removal of conferences only containing MCU members
 *
 * Revision 2.11  2006/07/21 08:01:40  csoutheren
 * Fixed conference member detect
 * Re-factored video mixer code slightly
 *
 * Revision 2.10  2006/07/21 05:52:51  csoutheren
 * Add flag to indicate if remote endpoint is an MCU
 *
 * Revision 2.9  2006/07/21 05:08:03  csoutheren
 * Stability fixes and more inline documentation
 * Thanks to Paolo Amadini of Stonevoice
 *
 * Revision 2.8  2006/06/09 04:39:59  csoutheren
 * Migrated VideoBranch to main trunk
 *
 * Revision 2.7.2.18  2006/06/06 08:24:16  csoutheren
 * Added support for echo test room
 * Fix problem with using high frame rates
 *
 * Revision 2.7.2.17  2006/05/31 08:48:02  csoutheren
 * Fixed crash on second call when entry/exit files do not exist
 * Fix crash when Cisco HOLD used. Thanks to Christian Bongiovanni of Stonevoice
 *
 * Revision 2.7.2.16  2006/05/16 02:20:39  csoutheren
 * Fixed problems compiling without video
 *
 * Revision 1.8  2006/05/16 00:29:18  craigs
 * Fixed compilation when video disabled
 *
 * Revision 1.7  2006/04/26 16:02:21  craigs
 * Update to latest public sources
 * Added ability to have per-conference timeout and reminders
 *
 * Revision 2.7.2.15  2006/04/26 15:03:16  csoutheren
 * Added better override for creating conferenc
 *
 * Revision 2.7.2.14  2006/04/26 13:09:08  csoutheren
 * Fix problem when connecting file not available
 * Add optional time limit for rooms
 *
 * Revision 2.7.2.13  2006/04/18 03:05:09  csoutheren
 * Fix video mix problem with > 3 members
 * Add test room
 *
 * Revision 2.7.2.12  2006/04/12 22:58:16  csoutheren
 * Removed return in function returning void
 *
 * Revision 2.7.2.11  2006/04/12 04:54:18  csoutheren
 * Fix more problems with audio-only members
 *
 * Revision 2.7.2.10  2006/04/11 03:46:46  csoutheren
 * Fixed problem with hangup during entering wav file play
 *
 * Revision 2.7.2.9  2006/04/06 08:20:29  csoutheren
 * Retyped conference member identifier to epxlicit type
 * Added support for H.245 terminal added and terminal left
 *
 * Revision 2.7.2.8  2006/04/06 01:11:16  csoutheren
 * Latest sources include
 *   - premedia blanking and optional image display
 *   - ablity to defer conference join for authentication if required
 *   - more bulletproofing on conference join
 *   - new video copy/fill functions
 *
 * Revision 2.7.2.7  2006/04/06 00:50:28  csoutheren
 * Latest changes (more to come)
 *
 * Revision 2.7.2.6  2006/03/28 05:13:38  csoutheren
 * Normalised file headers
 * Fixed problem with QCIF video
 * Seperated H.323 and MCU process functions into seperate files
 *
 * Revision 2.7.2.5  2006/03/24 22:49:24  csoutheren
 * Video now working
 *
 * Revision 2.7.2.4  2006/03/21 14:02:49  csoutheren
 * More video mixiing.
 * Still some crashes in video mixer code when members exit conference
 *
 * Revision 2.7.2.3  2006/03/17 07:00:24  csoutheren
 * More video implementation
 * Video mixing now working except copying subimage UV components causes overrun
 * Y is OK, which is why the images are grey. Also need to work out flipping requirements
 *
 * Revision 2.7.2.2  2006/03/14 08:02:50  csoutheren
 * More implemenrtation of video
 * Video mixing infrastructure implemented but not fully working
 *
 * Revision 2.7.2.1  2006/03/06 05:45:30  csoutheren
 * Start of implementation for video. Still in progress
 *
 * Revision 2.7  2006/02/08 06:54:44  csoutheren
 * Added extra information to monitoring page
 *
 * Revision 2.6  2006/02/03 07:21:41  csoutheren
 * Added INVITE function via web interface
 *
 * Revision 2.5  2004/09/22 22:39:26  csoutheren
 * Fixed race condition, thanks to Neil McCurdy
 *
 * Revision 2.4  2004/05/26 06:54:31  csoutheren
 * Changed to be a PHTTPServiceProcess
 * Added ability to play WAV files on member entry and exit
 * Added additional documentation on all classes
 * Preparation for re-introducing video
 *
 * Revision 2.3  2004/03/31 03:36:38  csoutheren
 * Fixed problem with user indication messages
 * Fixed problems with room listener and unlisten
 *
 * Revision 2.2  2004/03/30 11:27:23  csoutheren
 * Reverted to old mixing algorithm
 *
 * Revision 2.1  2004/03/23 11:40:06  csoutheren
 * Fixed problem where deleting map element in-place causes crash at end of call
 * Fixed problem where referencing map by iterator rather than ID
 * Fixed code formatting problems
 *
 * Revision 2.0  2004/03/08 02:06:24  csoutheren
 * Totally rewritten to use new connection locking mecahnism
 * Added ability to monitor conferences
 * Added initial support for H.323 MCU messages
 * Thanks to Citron Networks for supporting this work
 *
 */

#include <ptlib.h>

#include "conference.h"
#include "mcu.h"

#if OPENMCU_VIDEO
#include <ptlib/vconvert.h>
#endif

// size of a PCM data packet, in samples
#define PCM_PACKET_LEN          240

// size of a PCM data buffer, in bytes
#define PCM_BUFFER_LEN          (PCM_PACKET_LEN * 2)

// number of PCM buffers to keep
#define PCM_BUFFER_COUNT        2

#define PCM_BUFFER_SIZE         (PCM_BUFFER_LEN * PCM_BUFFER_COUNT)

////////////////////////////////////////////////////////////////////////////////////

ConferenceManager::ConferenceManager()
{
  maxConferenceCount = 0;
  monitor  = new ConferenceMonitor(*this);
}

ConferenceManager::~ConferenceManager()
{
  monitor->running = FALSE;
  monitor->WaitForTermination();
  delete monitor;
}


Conference * ConferenceManager::MakeAndLockConference(const PString & roomToCreate, const PString & name)
{
  PWaitAndSignal m(conferenceListMutex);

  OpalGloballyUniqueID conferenceID;
  ConferenceListType::const_iterator r;
  for (r = conferenceList.begin(); r != conferenceList.end(); ++r) {
    if (roomToCreate == r->second->GetNumber()) {
      conferenceID = r->second->GetID();
      break;
    }
  }

  return MakeAndLockConference(conferenceID, roomToCreate, name);
}


Conference * ConferenceManager::MakeAndLockConference(const OpalGloballyUniqueID & conferenceID, 
                                                                   const PString & roomToCreate, 
                                                                   const PString & name)
{
  conferenceListMutex.Wait();

  Conference * conference = NULL;
  PBoolean newConference = FALSE;
  ConferenceListType::const_iterator r = conferenceList.find(conferenceID);
  if (r != conferenceList.end())
    conference = r->second;
  else {
    // create the conference
    conference = CreateConference(conferenceID, roomToCreate, name, mcuNumberMap.GetNumber(conferenceID));

    // insert conference into the map
    conferenceList.insert(ConferenceListType::value_type(conferenceID, conference));

    // set the conference count
    maxConferenceCount = PMAX(maxConferenceCount, (PINDEX)conferenceList.size());
    newConference = TRUE;
  }

  if (newConference)
    OnCreateConference(conference);

  return conference;
}

void ConferenceManager::OnCreateConference(Conference * conference)
{
  // set time limit, if there is one
  int timeLimit = OpenMCU::Current().GetRoomTimeLimit();
  if (timeLimit > 0)
    monitor->AddMonitorEvent(new ConferenceTimeLimitInfo(conference->GetID(), PTime() + timeLimit*1000));
}

Conference * ConferenceManager::CreateConference(const OpalGloballyUniqueID & _guid,
                                                              const PString & _number,
                                                              const PString & _name,
                                                                          int _mcuNumber)
{ 
#if OPENMCU_VIDEO
#if ENABLE_ECHO_MIXER
  if (_number *= "echo")
    return new Conference(*this, _guid, _guid.AsString(), _name, _mcuNumber, new EchoVideoMixer());
#endif
#if ENABLE_TEST_ROOMS
  if (_number.Left(8) == "testroom") {
    int count = _number.Mid(8).AsInteger();
    if (count > 0)
      return new Conference(*this, _guid, _number, _name, _mcuNumber, new TestVideoMixer(count));
  }
#endif
#endif

  return new Conference(*this, _guid, _number, _name, _mcuNumber
#if OPENMCU_VIDEO
                        ,OpenMCU::Current().CreateVideoMixer()
#endif
                        ); 
}

PBoolean ConferenceManager::HasConference(const OpalGloballyUniqueID & conferenceID, PString & number)
{
  PWaitAndSignal m(conferenceListMutex);
  ConferenceListType::const_iterator r = conferenceList.find(conferenceID);
  if (r == conferenceList.end())
    return FALSE;
  number = r->second->GetNumber();
  return TRUE;
}

PBoolean ConferenceManager::HasConference(const PString & number, OpalGloballyUniqueID & conferenceID)
{
  PWaitAndSignal m(conferenceListMutex);
  ConferenceListType::const_iterator r;
  for (r = conferenceList.begin(); r != conferenceList.end(); ++r) {
    if (r->second->GetNumber() == number) {
      conferenceID = r->second->GetID();
      return TRUE;
    }
  }
  return FALSE;
}

void ConferenceManager::RemoveConference(const OpalGloballyUniqueID & confId)
{
  PWaitAndSignal m(conferenceListMutex);
  ConferenceListType::iterator r = conferenceList.find(confId);
  if (r != conferenceList.end())  {
    Conference * conf = r->second;
    OnDestroyConference(conf);
    conferenceList.erase(confId);
    mcuNumberMap.RemoveNumber(conf->GetMCUNumber());
    monitor->RemoveForConference(conf->GetID());

    delete conf;
  }
}

void ConferenceManager::RemoveMember(const OpalGloballyUniqueID & confId, ConferenceMember * toRemove)
{
  PWaitAndSignal m(conferenceListMutex);
  Conference * conf = toRemove->GetConference();

  OpalGloballyUniqueID id = conf->GetID();  // make a copy of the ID because it may be about to disappear

  PBoolean removeConf = conf->RemoveMember(toRemove);
  delete toRemove;
  if (removeConf)
    RemoveConference(id);
}

void ConferenceManager::AddMonitorEvent(ConferenceMonitorInfo * info)
{
  monitor->AddMonitorEvent(info);
}


///////////////////////////////////////////////////////////////

void ConferenceMonitor::Main()
{
  running = TRUE;

  for (;;) {

    if (!running)
      break;

    Sleep(1000);

    if (!running)
      break;

    PWaitAndSignal m(mutex);

    PTime now;
    MonitorInfoList::iterator r = monitorList.begin();
    while (r != monitorList.end()) {
      ConferenceMonitorInfo & info = **r;
      if (now < info.timeToPerform)
        ++r;
      else {
        PBoolean deleteAfterPerform = TRUE;
        {
          PWaitAndSignal m2(manager.GetConferenceListMutex());
          ConferenceListType & confList = manager.GetConferenceList();
          ConferenceListType::iterator s = confList.find(info.guid);
          if (s != confList.end())
            deleteAfterPerform = info.Perform(*s->second);
        }
        if (!deleteAfterPerform)
          ++r;
        else {
          delete *r;
          monitorList.erase(r);
          r = monitorList.begin();
        }
      }
    }
  }
}

void ConferenceMonitor::AddMonitorEvent(ConferenceMonitorInfo * info)
{
  PWaitAndSignal m(mutex);
  monitorList.push_back(info);
}

void ConferenceMonitor::RemoveForConference(const OpalGloballyUniqueID & guid)
{
  PWaitAndSignal m(mutex);

  MonitorInfoList::iterator r = monitorList.begin();
  while (r != monitorList.end()) {
    ConferenceMonitorInfo & info = **r;
    if (info.guid != guid)
      ++r;
    else {
      delete *r;
      monitorList.erase(r);
      r = monitorList.begin();
    }
  }
}

PBoolean ConferenceTimeLimitInfo::Perform(Conference & conference)
{
  Conference::MemberList & list = conference.GetMemberList();
  Conference::MemberList::iterator r;
  for (r = list.begin(); r != list.end(); ++r)
    r->second->Close();
  return TRUE;
}

PBoolean ConferenceRepeatingInfo::Perform(Conference & conference)
{
  this->timeToPerform = PTime() + repeatTime;
  return FALSE;
}


PBoolean ConferenceMCUCheckInfo::Perform(Conference & conference)
{
  // see if any member of this conference is a not an MCU
  Conference::MemberList & list = conference.GetMemberList();
  Conference::MemberList::iterator r;
  for (r = list.begin(); r != list.end(); ++r)
    if (!r->second->IsMCU())
      break;

  // if there is any non-MCU member, check again later
  if (r != list.end())
    return ConferenceRepeatingInfo::Perform(conference);

  // else shut down the conference
  for (r = list.begin(); r != list.end(); ++r)
    r->second->Close();

  return TRUE;
}


////////////////////////////////////////////////////////////////////////////////////

Conference::Conference(        ConferenceManager & _manager,
                       const OpalGloballyUniqueID & _guid,
                                    const PString & _number,
                                    const PString & _name,
                                                int _mcuNumber
#if OPENMCU_VIDEO
                                    ,MCUVideoMixer * _videoMixer
#endif
                                    ,ConferenceType t
)
  : manager(_manager), guid(_guid), number(_number), name(_name), mcuNumber(_mcuNumber), mcuMonitorRunning(FALSE)
    ,type (t)
#if OPENMCU_VIDEO
    ,videoMixer(_videoMixer)
#endif
{ 
  maxMemberCount = 0;

  PTRACE(3, "Conference\tNew conference started: ID=" << guid << ", number = " << number);
}

Conference::~Conference()
{
#if OPENMCU_VIDEO
  delete videoMixer;
#endif
}

int Conference::GetVisibleMemberCount() const
{
  PWaitAndSignal m(memberListMutex);
  int visibleMembers = 0;
  std::map<void *, ConferenceMember *>::const_iterator r;
  for (r = memberList.begin(); r != memberList.end(); r++) {
    if (r->second->IsVisible())
      ++visibleMembers;
  }
  return visibleMembers;
}

void Conference::AddMonitorEvent(ConferenceMonitorInfo * info)
{ 
  manager.AddMonitorEvent(info); 
}


PBoolean Conference::AddMember(ConferenceMember * memberToAdd)
{
  PTRACE(3, "Conference\tAbout to add member " << memberToAdd->GetTitle() << " to conference " << guid);

  // see if the callback refuses the new member
  if (!BeforeMemberJoining(memberToAdd))
    return FALSE;

  // add the member to the conference
  if (!memberToAdd->AddToConference(this))
    return FALSE;

  {
    // lock the member list
    PWaitAndSignal m(memberListMutex);

#if OPENMCU_VIDEO
    if (UseSameVideoForAllMembers() && memberToAdd->IsVisible()) {
      if (!videoMixer->AddVideoSource(memberToAdd->GetID(), *memberToAdd)) 
        return FALSE;
    }
#endif

    PTRACE(3, "Conference\tAdding member " << memberToAdd->GetTitle() << " to conference " << guid);

    // add this member to the conference member list
    memberList.insert(MemberList::value_type(memberToAdd->GetID(), memberToAdd));

    int tid = terminalNumberMap.GetNumber(memberToAdd->GetID());
    memberToAdd->SetTerminalNumber(tid);

    // make sure each member has a connection created for the new member
    // make sure the new member has a connection created for each existing member
    PINDEX visibleMembers = 0;
    std::map<void *, ConferenceMember *>::const_iterator r;
    for (r = memberList.begin(); r != memberList.end(); r++) {
      ConferenceMember * conn = r->second;
      if (conn != memberToAdd) {
        conn->AddConnection(memberToAdd);
        memberToAdd->AddConnection(conn);
#if OPENMCU_VIDEO
        if (!UseSameVideoForAllMembers()) {
          if (conn->IsVisible())
            memberToAdd->AddVideoSource(conn->GetID());
          if (memberToAdd->IsVisible())
            conn->AddVideoSource(memberToAdd->GetID());
        }
#endif
      }
      if (conn->IsVisible())
        ++visibleMembers;
    }

    // update the statistics
    if (memberToAdd->IsVisible()) {
      maxMemberCount = PMAX(maxMemberCount, visibleMembers);

      // trigger H245 thread for join message
      new NotifyH245Thread(*this, TRUE, memberToAdd);
    }
  }

  // notify that member is joined
  memberToAdd->SetJoined(TRUE);

  // call the callback function
  OnMemberJoining(memberToAdd);

  if (memberToAdd->IsMCU() && !mcuMonitorRunning) {
    manager.AddMonitorEvent(new ConferenceMCUCheckInfo(GetID(), 1000));
    mcuMonitorRunning = TRUE;
  }

  return TRUE;
}

PBoolean Conference::RemoveMember(ConferenceMember * memberToRemove)
{
  if(!memberToRemove->IsJoined())
  {
    PTRACE(4, "Conference\tNo need to remove call " << memberToRemove->GetTitle() << " from conference " << guid);
    return (memberList.size() == 0);
  }

  PTRACE(3, "Conference\tRemoving call " << memberToRemove->GetTitle() << " from conference " << guid << " with size " << (PINDEX)memberList.size());
  cout << memberToRemove->GetTitle() << " leaving conference " << number << "(" << guid << ")" << endl;

  PBoolean closeConference;
  {
    PWaitAndSignal m(memberListMutex);

    // remove this member from the connection lists for all other members
    MemberList::iterator r;
    for (r = memberList.begin(); r != memberList.end(); r++) {
      ConferenceMember * conn = r->second;
      if (conn != memberToRemove) {
        r->second->RemoveConnection(memberToRemove->GetID());
#if OPENMCU_VIDEO
        if (!UseSameVideoForAllMembers()) {
          if (memberToRemove->IsVisible())
            conn->RemoveVideoSource(memberToRemove->GetID());
          if (conn->IsVisible())
            memberToRemove->RemoveVideoSource(conn->GetID());
        }
#endif
      }
    }

#if OPENMCU_VIDEO
    if (UseSameVideoForAllMembers() && memberToRemove->IsVisible())
      videoMixer->RemoveVideoSource(memberToRemove->GetID(), *memberToRemove);
#endif

    // trigger H245 thread for leave message
    if (memberToRemove->IsVisible())
      new NotifyH245Thread(*this, FALSE, memberToRemove);

    terminalNumberMap.RemoveNumber(memberToRemove->GetTerminalNumber());

    // remove this connection from the member list
    memberList.erase(memberToRemove->GetID());

    // return TRUE if conference is empty 
    closeConference = memberList.size() == 0;
  }

  // notify that member is not joined anymore
  memberToRemove->SetJoined(FALSE);

  // call the callback function
  if (!closeConference)
    OnMemberLeaving(memberToRemove);

  return closeConference;
}


void Conference::ReadMemberAudio(ConferenceMember * member, void * buffer, PINDEX amount)
{
  // get number of channels to mix
  ConferenceMember::ConnectionListType & connectionList = member->GetConnectionList();
  ConferenceMember::ConnectionListType::iterator r;
  for (r = connectionList.begin(); r != connectionList.end(); ++r) 
    if (r->second != NULL)
      r->second->ReadAndMixAudio((BYTE *)buffer, amount, (PINDEX)connectionList.size());
}

void Conference::WriteMemberAudioLevel(ConferenceMember * member, unsigned audioLevel)
{
#if OPENMCU_VIDEO
  if (UseSameVideoForAllMembers()) {
    if (videoMixer != NULL)
      videoMixer->SetAudioLevel(member->GetID(), audioLevel);
  }
  else 
#endif // OPENMCU_VIDEO
  {
    PWaitAndSignal m(memberListMutex);
    MemberList::iterator r;
    for (r = memberList.begin(); r != memberList.end(); r++)
      r->second->OnExternalSetAudioLevel(member->GetID(), audioLevel);
  }
}

#if OPENMCU_VIDEO

void Conference::ReadMemberVideo(ConferenceMember * member, void * buffer, int width, int height, PINDEX & amount)
{
  if (videoMixer == NULL)
    return;

  if (videoMixer->ReadFrame(*member, buffer, width, height, amount))
    return;

  // find the other member and copy it's video
  PWaitAndSignal m(memberListMutex);
  MemberList::iterator r;
  for (r = memberList.begin(); r != memberList.end(); r++) {
    if ((r->second != member) && r->second->IsVisible()) {
      void * frameStore = r->second->OnExternalReadVideo(member->GetID(), width, height, amount);  
      if (frameStore != NULL) {
        memcpy(buffer, frameStore, amount);
        r->second->UnlockExternalVideo();
      }
    }
  }
}

PBoolean Conference::WriteMemberVideo(ConferenceMember * member, const void * buffer, int width, int height, PINDEX amount)
{
  if (UseSameVideoForAllMembers()) {
    if (videoMixer != NULL)
      return videoMixer->WriteFrame(member->GetID(), buffer, width, height, amount);
  }
  else {
    PWaitAndSignal m(memberListMutex);
    MemberList::iterator r;
    for (r = memberList.begin(); r != memberList.end(); r++)
      r->second->OnExternalSendVideo(member->GetID(), buffer, width, height, amount);
  }
  return TRUE;
}

#endif

PBoolean Conference::BeforeMemberJoining(ConferenceMember * member)
{ 
  return manager.BeforeMemberJoining(this, member); 
}

void Conference::OnMemberJoining(ConferenceMember * member)
{ 
  manager.OnMemberJoining(this, member); 
}

void Conference::OnMemberLeaving(ConferenceMember * member)
{ 
  manager.OnMemberLeaving(this, member); 
}


///////////////////////////////////////////////////////////////////////////

ConferenceMember::ConferenceMember(Conference * _conference, ConferenceMemberId _id, PBoolean _isMCU)
  : conference(_conference), id(_id), isMCU(_isMCU)
{
  audioLevel = 0;
  terminalNumber = -1;
  memberIsJoined = FALSE;

#if OPENMCU_VIDEO
  videoMixer = NULL;
  fsConverter = PColourConverter::Create("YUV420P", "YUV420P", CIF_WIDTH, CIF_HEIGHT);
  MCUVideoMixer::FillCIFYUVFrame(memberFrameStores.GetFrameStore(CIF_WIDTH, CIF_HEIGHT).data.GetPointer(), 0, 0, 0);
  totalVideoFramesReceived = 0;
  firstFrameReceiveTime = -1;
  totalVideoFramesSent = 0;
  firstFrameSendTime = -1;
#endif
}

ConferenceMember::~ConferenceMember()
{
#if OPENMCU_VIDEO
  delete videoMixer;
#endif
}


PBoolean ConferenceMember::AddToConference(Conference * _conference)
{
  //if (conference != NULL)
  //  return FALSE;
  //conference = _conference;

#if OPENMCU_VIDEO
  if (!conference->UseSameVideoForAllMembers())
    videoMixer = conference->GetVideoMixer()->Clone();
#endif

  return TRUE;
}

void ConferenceMember::RemoveFromConference()
{
  if (conference != NULL) {
    if (conference->RemoveMember(this))
      conference->GetManager().RemoveConference(conference->GetID());
  }
}

void ConferenceMember::AddConnection(ConferenceMember * memberToAdd)
{
  ConferenceMemberId newID = memberToAdd->GetID();
  PTRACE(3, "Conference\tAdding " << newID << " to connection " << id);
  if (lock.Wait(TRUE)) {
    ConferenceConnection * conn = memberToAdd->CreateConnection();
    memberList.insert(MemberListType::value_type(newID, memberToAdd));
    connectionList.insert(ConnectionListType::value_type(newID, conn));
    lock.Signal(TRUE);
  }
}

void ConferenceMember::RemoveConnection(ConferenceMemberId idToDelete)
{
  PTRACE(3, "Conference\tRemoving member " << idToDelete << " from connection " << id);
  if (lock.Wait(TRUE)) {
    memberList.erase(idToDelete);
    connectionList.erase(idToDelete);
    lock.Signal(TRUE);
  }
}

void ConferenceMember::SendUserInputIndication(const PString & str)
{ 
  PTRACE(3, "Conference\tConnection " << id << " sending user indication " << str);
  if (lock.Wait()) {
    MemberListType::iterator r;
    for (r = memberList.begin(); r != memberList.end(); ++r)
      if (r->second != NULL)
        r->second->OnReceivedUserInputIndication(str);
    lock.Signal();
  }
}

void ConferenceMember::WriteAudio(const void * buffer, PINDEX amount)
{
  // calculate average signal level for this member
  unsigned signalLevel = 0;
  {
    int sum = 0;
    const short * pcm = (short *)buffer;
    const short * end = pcm + (amount / 2);
    while (pcm != end) {
      if (*pcm < 0)
        sum -= *pcm++;
      else
        sum += *pcm++;
    }
    signalLevel = sum/(amount/2);
  }
  audioLevel = ((signalLevel * 2) + audioLevel) / 3;

  if (lock.Wait()) {

    if (conference != NULL)
      conference->WriteMemberAudioLevel(this, audioLevel);

    MemberListType::iterator r;
    for (r = memberList.begin(); r != memberList.end(); ++r) {
      if (r->second != NULL)
        r->second->OnExternalSendAudio(id, (BYTE *)buffer, amount);
    }
    lock.Signal();
  }
}

void ConferenceMember::OnExternalSendAudio(ConferenceMemberId source, const void * buffer, PINDEX amount)
{
  if (lock.Wait()) {
    ConnectionListType::iterator r = connectionList.find(source);
    if (r != connectionList.end())
      if (r->second != NULL)
        r->second->Write((BYTE *)buffer, amount);
    lock.Signal();
  }
}

void ConferenceMember::ReadAudio(void * buffer, PINDEX amount)
{
  // First, set the buffer to empty.
  memset(buffer, 0, amount);

  if (lock.Wait()) {
    if (conference != NULL)
      conference->ReadMemberAudio(this, buffer, amount);
    lock.Signal();
  }
}

void ConferenceMember::OnExternalSetAudioLevel(ConferenceMemberId id, unsigned audioLevel)
{
#if OPENMCU_VIDEO
  if (lock.Wait()) {
    videoMixer->SetAudioLevel(id, audioLevel);
    lock.Signal();
  }
#endif // OPENMCU_VIDEO
}

#if OPENMCU_VIDEO

// called whenever the connection needs a frame of video to send
void ConferenceMember::ReadVideo(void * buffer, int width, int height, PINDEX & amount)
{
  ++totalVideoFramesSent;
  if (!firstFrameSendTime.IsValid())
    firstFrameSendTime = PTime();
  if (lock.Wait()) {
    if (conference != NULL) {
      if (conference->UseSameVideoForAllMembers())
        conference->ReadMemberVideo(this, buffer, width, height, amount);
      else if (videoMixer != NULL)
        videoMixer->ReadFrame(*this, buffer, width, height, amount);
    }
    lock.Signal();
  }
}

// called whenever the connection receives a frame of video
void ConferenceMember::WriteVideo(const void * buffer, int width, int height, PINDEX amount)
{
  ++totalVideoFramesReceived;
  if (!firstFrameReceiveTime.IsValid())
    firstFrameReceiveTime = PTime();

  if (lock.Wait()) {
    if (conference != NULL) {
      if (!conference->WriteMemberVideo(this, buffer, width, height, amount)) {
        PWaitAndSignal m(memberFrameStoreMutex);
        VideoFrameStoreList::FrameStore & fs = memberFrameStores.GetFrameStore(width, height);
        memcpy(fs.data.GetPointer(), buffer, amount);
        memberFrameStores.InvalidateExcept(width, height);
      }
    }
    lock.Signal();
  }
}

void ConferenceMember::OnExternalSendVideo(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount)
{
  if (lock.Wait()) {
    videoMixer->WriteFrame(id, buffer, width, height, amount);
    lock.Signal();
  }
}

void * ConferenceMember::OnExternalReadVideo(ConferenceMemberId id, int width, int height, PINDEX & bytesReturned)
{
  if (!lock.Wait())
    return NULL;

  memberFrameStoreMutex.Wait();

  PBoolean found;
  VideoFrameStoreList::FrameStore & nearestFs = memberFrameStores.GetNearestFrameStore(width, height, found);

  // if no valid framestores, nothing we can do
  if (!found) {
    memberFrameStoreMutex.Signal();
    lock.Signal();
    return NULL;
  }

  // if the valid framestore is a perfect match, return it
  if ((nearestFs.width == width) && (nearestFs.height == height))
    return nearestFs.data.GetPointer();

  // create a new destinationf framestore
  VideoFrameStoreList::FrameStore & destFs = memberFrameStores.GetFrameStore(width, height);

  // if src is qcif, and dest is cif, then do special case
  if ((nearestFs.width == QCIF_WIDTH) && 
      (nearestFs.height == QCIF_HEIGHT) &&
      (width == CIF_WIDTH) && 
      (height == CIF_HEIGHT)) {
    MCUVideoMixer::ConvertQCIFToCIF(nearestFs.data.GetPointer(), destFs.data.GetPointer());
    destFs.valid = TRUE;
  }

  // create an image that is the right size
  else {
    fsConverter->SetSrcFrameSize(nearestFs.width, nearestFs.height);
    fsConverter->SetDstFrameSize(width, height, TRUE);
    bytesReturned = width*height*3/2;
    fsConverter->Convert(nearestFs.data.GetPointer(), destFs.data.GetPointer(), nearestFs.width * nearestFs.height * 3 / 2, &bytesReturned);
    destFs.valid = TRUE;
  }

  return destFs.data.GetPointer();
}

void ConferenceMember::UnlockExternalVideo()
{ 
  memberFrameStoreMutex.Signal(); 
  lock.Signal();
}

PBoolean ConferenceMember::AddVideoSource(ConferenceMemberId id)
{
  PAssert(videoMixer != NULL, "attempt to add video source to NULL video mixer");
  return videoMixer->AddVideoSource(id, *this);
}

void ConferenceMember::RemoveVideoSource(ConferenceMemberId id)
{
  PAssert(videoMixer != NULL, "attempt to add video source to NULL video mixer");
  videoMixer->AddVideoSource(id, *this);
}

PBoolean ConferenceMember::OnOutgoingVideo(void * buffer, int width, int height, PINDEX & amount)
{
  return FALSE;
}

PBoolean ConferenceMember::OnIncomingVideo(const void * buffer, int width, int height, PINDEX amount)
{
  return FALSE;
}

#endif

PString ConferenceMember::GetMonitorInfo(const PString & /*hdr*/)
{ 
  return PString::Empty(); 
}

///////////////////////////////////////////////////////////////////////////

ConferenceConnection::ConferenceConnection(ConferenceMemberId _id)
  : id(_id), bufferSize(PCM_BUFFER_SIZE)
{
  buffer = new BYTE[bufferSize];
  bufferStart = bufferLen = 0;
}

ConferenceConnection::~ConferenceConnection()
{
  delete[] buffer;
}

void ConferenceConnection::Write(const BYTE * data, PINDEX amount)
{
  if (amount == 0)
    return;

  PWaitAndSignal mutex(audioBufferMutex);
  
  // if there is not enough room for the new data, make room
  PINDEX newLen = bufferLen + amount;
  if (newLen > bufferSize) {
    PINDEX toRemove = newLen - bufferSize;
    bufferStart = (bufferStart + toRemove) % bufferSize;
    bufferLen -= toRemove;
  }

  // copy data to the end of the new data, up to the end of the buffer
  PINDEX copyStart = (bufferStart + bufferLen) % bufferSize;
  if ((copyStart + amount) > bufferSize) {
    PINDEX toCopy = bufferSize - copyStart;
    memcpy(buffer + copyStart, data, toCopy);
    copyStart = 0;
    data      += toCopy;
    amount    -= toCopy;
    bufferLen += toCopy;
  }

  // copy the rest of the data
  if (amount > 0) {
    memcpy(buffer + copyStart, data, amount);
    bufferLen   += amount;
  }
}


void ConferenceConnection::ReadAudio(BYTE * data, PINDEX amount)
{
  if (amount == 0)
    return;

  PWaitAndSignal mutex(audioBufferMutex);
  
  if (bufferLen == 0) {
    memset(data, 0, amount); // nothing in the buffer. return silence
    return;
  }

  // fill output data block with silence if audiobuffer is
  // almost empty.
  if (amount > bufferLen) 
    memset(data + bufferLen, 0, amount - bufferLen);

  // only copy up to the amount of data remaining
  PINDEX copyLeft = PMIN(amount, bufferLen);

  // if buffer is wrapping, get first part
  if ((bufferStart + copyLeft) > bufferSize) {
    PINDEX toCopy = bufferSize - bufferStart;

    memcpy(data, buffer + bufferStart, toCopy);

    data        += toCopy;
    bufferLen   -= toCopy;
    copyLeft    -= toCopy;
    bufferStart = 0;
  }

  // get the remainder of the buffer
  if (copyLeft > 0) {

    memcpy(data, buffer + bufferStart, copyLeft);

    bufferLen -= copyLeft;
    bufferStart = (bufferStart + copyLeft) % bufferSize;
  }
}

void ConferenceConnection::ReadAndMixAudio(BYTE * data, PINDEX amount, PINDEX channels)
{
  if (amount == 0) {
    PTRACE(3, "Mixer\tNo data to read");
    return;
  }

  PWaitAndSignal mutex(audioBufferMutex);
  
  if (bufferLen == 0) {
    // nothing in the buffer to mix.
    return;
  }

  // only mix up to the amount of data remaining
  PINDEX copyLeft = PMIN(amount, bufferLen);

  // if buffer is wrapping, get first part
  if ((bufferStart + copyLeft) > bufferSize) {
    PINDEX toCopy = bufferSize - bufferStart;

    Mix(data, buffer + bufferStart, toCopy, channels);

    data        += toCopy;
    bufferLen   -= toCopy;
    copyLeft    -= toCopy;
    bufferStart = 0;
  }

  // get the remainder of the buffer
  if (copyLeft > 0) {

    Mix(data, buffer + bufferStart, copyLeft, channels);

    bufferLen -= copyLeft;
    bufferStart = (bufferStart + copyLeft) % bufferSize;
  }
}

void ConferenceConnection::Mix(BYTE * dst, const BYTE * src, PINDEX count, PINDEX /*channels*/)
{
#if 0
  memcpy(dst, src, count);
#else
  PINDEX i;
  for (i = 0; i < count; i += 2) {

    int srcVal = *(short *)src;
    int dstVal = *(short *)dst;

    int newVal = dstVal;

#if 0     //The loudest person gains the channel.
#define mix_abs(x) ((x) >= 0 ? (x) : -(x))
    if (mix_abs(newVal) > mix_abs(srcVal))
      dstVal = newVal;
    else
      dstVal = srcVal; 
#else   //Just add up all the channels.
    if ((newVal + srcVal) > 0x7fff)
      dstVal = 0x7fff;
    else
      dstVal += srcVal;
#endif
    *(short *)dst = (short)dstVal;

    dst += 2;
    src += 2;
  }
#endif
}

///////////////////////////////////////////////////////////////

MCULock::MCULock()
{
  closing = FALSE;
  count = 0;
}

PBoolean MCULock::Wait(PBoolean hard)
{
  mutex.Wait();
  if (hard)
    return TRUE;

  PBoolean ret = TRUE;
  if (!closing)
    count++;
  else
    ret = FALSE;

  mutex.Signal();
  return ret;
}

void MCULock::Signal(PBoolean hard)
{
  if (hard) {
    mutex.Signal();
    return;
  }

  mutex.Wait();
  if (count > 0)
    count--;
  if (closing)
    closeSync.Signal();
  mutex.Signal();
}

void MCULock::WaitForClose()
{
  mutex.Wait();
  closing = TRUE;
  PBoolean wait = count > 0;
  mutex.Signal();
  while (wait) {
    closeSync.Wait();
    mutex.Wait();
    wait = count > 0;
    mutex.Signal();
  }
}
