/*
 * h323.cxx
 *
 * H.323 routines for a simple MCU
 *
 * Copyright (C) 2000 Equivalence Pty. Ltd.
 * Copyright (C) 2004 Post Increment
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of ths code were written by by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Stonevoice, slc. http://www.stonevoice.com
 *
 * Portions of this code were written by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Citron Networks (http://www.citron.com.tw)
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 Craig Southeren (craig@postincrement.com)
 *
 * $Log: h323.cxx,v $
 * Revision 1.4  2008/05/23 11:18:15  willamowius
 * switch BOOL to PBoolean to be able to compile with Ptlib 2.2.x
 *
 * Revision 1.3  2008/02/19 06:36:11  shorne
 * Added ability to have 16CIF & 4CIF size video. Thx Marcos Jardini for the initial 4CIF patch
 *
 * Revision 1.2  2007/11/07 04:03:25  willamowius
 * fix compile on gcc
 *
 * Revision 1.1  2007/10/17 19:44:30  shorne
 * Initial Commit
 *
 * Revision 2.7.2.2  2007/08/16 20:29:01  shorne
 * Better Gatekeeper anf Gateway prefix support
 *
 * Revision 2.7.2.1  2007/02/06 11:41:26  shorne
 * Updates for OpenH323 Video Plugins
 *
 * Revision 2.7  2006/08/07 06:18:22  csoutheren
 * Fix detection of MCU members
 * Add auto-removal of conferences only containing MCU members
 *
 * Revision 2.6  2006/08/02 06:24:53  csoutheren
 * Add provision for recording input audio
 *
 * Revision 2.5  2006/07/21 08:01:40  csoutheren
 * Fixed conference member detect
 * Re-factored video mixer code slightly
 *
 * Revision 2.4  2006/07/21 05:52:51  csoutheren
 * Add flag to indicate if remote endpoint is an MCU
 *
 * Revision 2.3  2006/07/21 05:08:03  csoutheren
 * Stability fixes and more inline documentation
 * Thanks to Paolo Amadini of Stonevoice
 *
 * Revision 2.2  2006/06/21 06:11:36  csoutheren
 * Fixes for latest pwlib
 *
 * Revision 2.1  2006/06/09 04:39:59  csoutheren
 * Migrated VideoBranch to main trunk
 *
 * Revision 1.1.2.8  2006/06/06 08:24:16  csoutheren
 * Added support for echo test room
 * Fix problem with using high frame rates
 *
 * Revision 1.1.2.7  2006/05/31 08:48:02  csoutheren
 * Fixed crash on second call when entry/exit files do not exist
 * Fix crash when Cisco HOLD used. Thanks to Christian Bongiovanni of Stonevoice
 *
 * Revision 1.1.2.6  2006/04/26 13:09:08  csoutheren
 * Fix problem when connecting file not available
 * Add optional time limit for rooms
 *
 * Revision 1.1.2.5  2006/04/18 03:05:09  csoutheren
 * Fix video mix problem with > 3 members
 * Add test room
 *
 * Revision 1.1.2.4  2006/04/06 08:20:29  csoutheren
 * Retyped conference member identifier to epxlicit type
 * Added support for H.245 terminal added and terminal left
 *
 * Revision 1.1.2.3  2006/04/06 01:11:16  csoutheren
 * Latest sources include
 *   - premedia blanking and optional image display
 *   - ablity to defer conference join for authentication if required
 *   - more bulletproofing on conference join
 *   - new video copy/fill functions
 *
 * Revision 1.1.2.2  2006/04/06 00:50:30  csoutheren
 * Latest changes (more to come)
 *
 * Revision 1.1.2.1  2006/03/28 05:13:38  csoutheren
 * Normalised file headers
 * Fixed problem with QCIF video
 * Seperated H.323 and MCU process functions into seperate files
 *
 */

#include <ptlib.h>
#include <ptlib/video.h>

#include "version.h"
#include "mcu.h"
#include "h323.h"

#include <h323pdu.h>

#if OPENMCU_VIDEO

const unsigned int DefaultVideoFrameRate = 10;
const unsigned int DefaultVideoQuality   = 10;
static const char EnableVideoKey[]       = "Enable video";
static const char VideoFrameRateKey[]    = "Video frame rate";
static const char VideoQualityKey[]      = "Video quality";

#endif  // OPENMCU_VIDEO

static const char InterfaceKey[]          = "Interface";
static const char LocalUserNameKey[]      = "Local User Name";
static const char GatekeeperUserNameKey[] = "Gatekeeper Username";
static const char GatekeeperAliasKey[]    = "Gatekeeper Room Names";
static const char GatekeeperPasswordKey[] = "Gatekeeper Password";
static const char GatekeeperPrefixesKey[] = "Gatekeeper Prefixes";
static const char GatekeeperModeKey[]     = "Gatekeeper Mode";
static const char GatekeeperKey[]         = "Gatekeeper";
static const char DisableCodecsKey[]      = "Disable codecs";

static const char * GKModeLabels[] = { 
   "No gatekeeper", 
   "Find gatekeeper", 
   "Use gatekeeper", 
};

enum {
  Gatekeeper_None,
  Gatekeeper_Find,
  Gatekeeper_Explicit
};

#define GKMODE_LABEL_COUNT   (sizeof(GKModeLabels)/sizeof(char *))

#define new PNEW


///////////////////////////////////////////////////////////////

void SpliceMacro(PString & text, const PString & token, const PString & value)
{
  PRegularExpression RegEx("<?!--#status[ \t\r\n]+" + token + "[ \t\r\n]*-->?",
                           PRegularExpression::Extended|PRegularExpression::IgnoreCase);
  PINDEX pos, len;
  while (text.FindRegEx(RegEx, pos, len))
    text.Splice(value, pos, len);
}

///////////////////////////////////////////////////////////////

OpenMCUH323EndPoint::OpenMCUH323EndPoint(ConferenceManager & _conferenceManager, H323Capability::CapabilityFrameSize _size)
  : conferenceManager(_conferenceManager), MaxFrameSize(_size)
{
#if OPENMCU_VIDEO
	terminalType = e_MCUWithAVMP;
  enableVideo  = TRUE;
  videoRate    = DefaultVideoFrameRate;
#else
	terminalType = e_MCUWithAudioMP;
#endif

   gkAlias = PString();

   Initialised = FALSE;
}

void OpenMCUH323EndPoint::Initialise(PConfig & cfg, PConfigPage * rsrc)
{

#ifdef HAS_AEC
  SetAECEnabled(FALSE);
#endif

///////////////////////////////////////////
// Listeners
  PString defaultInterface = "*:1720";
  H323TransportAddressArray interfaces;
  PINDEX arraySize = cfg.GetInteger(PString(InterfaceKey) + " Array Size");
  if (arraySize == 0)
    StartListener(defaultInterface);
  else {
    for (int i = 0; i < arraySize; i++)
      interfaces.Append(new H323TransportAddress(cfg.GetString(psprintf("%s %u", InterfaceKey, i+1), "")));
    StartListeners(interfaces);
  }
  rsrc->Add(new PHTTPFieldArray(new PHTTPStringField(cfg.GetDefaultSection() + "\\" + InterfaceKey, InterfaceKey, 20, defaultInterface), FALSE));

  if (listeners.IsEmpty()) {
    PSYSTEMLOG(Fatal, "Main\tCould not open H.323 Listener");
  }

  AliasList.RemoveAll();
  localAliasNames.RemoveAll();

//////////////////////////////////////////////////////
// Gatekeeper mode
  PStringArray labels(GKMODE_LABEL_COUNT, GKModeLabels); 
  PINDEX idx = labels.GetStringsIndex(cfg.GetString(GatekeeperModeKey, labels[0]));  
  PINDEX gkMode = (idx == P_MAX_INDEX) ? 0 : idx;
  rsrc->Add(new PHTTPRadioField(GatekeeperModeKey, labels, gkMode));

  // Gatekeeper 
  PString gkName = cfg.GetString(GatekeeperKey);
  rsrc->Add(new PHTTPStringField(GatekeeperKey, 25, gkName));

  // Gatekeeper UserName
  PString gkUserName = cfg.GetString(GatekeeperUserNameKey,"MCU");
  if (gkMode == Gatekeeper_None ) {
    // Local alias name for H.323 endpoint
    SetLocalUserName(cfg.GetString(LocalUserNameKey, OpenMCU::Current().GetName() + " v" + OpenMCU::Current().GetVersion()));
  } else {
    SetLocalUserName(gkUserName);
  }
  rsrc->Add(new PHTTPStringField(GatekeeperUserNameKey, 25, gkUserName));
  AliasList.AppendString(gkUserName);

  // Gatekeeper password
  PString gkPassword = PHTTPPasswordField::Decrypt(cfg.GetString(GatekeeperPasswordKey));
  SetGatekeeperPassword(gkPassword);
  rsrc->Add(new PHTTPPasswordField(GatekeeperPasswordKey, 25, gkPassword));

  // Gatekeeper Alias
  gkAlias = cfg.GetString(GatekeeperAliasKey,"MCU*");
  rsrc->Add(new PHTTPStringField(GatekeeperAliasKey, 25, gkAlias));

  for (PINDEX k=0; k< OpenMCU::defaultRoomCount; k++) {
	  PString alias = gkAlias;
	  alias.Replace("*",k);
	  AddAliasName(alias);   // Add the alias to the endpoint aliaslist
      AliasList.AppendString(alias);  
  }

  // Gatekeeper prefixes
  PINDEX prefixSize = cfg.GetInteger(PString(GatekeeperPrefixesKey) + " Array Size");
  if (prefixSize > 0) {
	  for (int j = 0; j < prefixSize; j++) {
	     PString prefix = cfg.GetString(psprintf("%s %u", GatekeeperPrefixesKey, j+1));
         PrefixList.AppendString(prefix);

		for (PINDEX k=0; k< OpenMCU::defaultRoomCount; k++) {
			PString alias = prefix + PString(k);
			AliasList.AppendString(alias);  
		}
	  }
  }
  rsrc->Add(new PHTTPFieldArray(new PHTTPStringField(cfg.GetDefaultSection() + "\\" + GatekeeperPrefixesKey, 
	                                                                    GatekeeperPrefixesKey, 20, ""), FALSE));

#if OPENMCU_VIDEO
  enableVideo = cfg.GetBoolean(EnableVideoKey, TRUE);
  rsrc->Add(new PHTTPBooleanField(EnableVideoKey, enableVideo));

  videoRate = cfg.GetInteger(VideoFrameRateKey, DefaultVideoFrameRate);
  rsrc->Add(new PHTTPIntegerField(VideoFrameRateKey, 1, 30, videoRate));

  videoTxQuality = cfg.GetInteger(VideoQualityKey, DefaultVideoQuality);
  rsrc->Add(new PHTTPIntegerField(VideoQualityKey, 1, 30, videoTxQuality));
#endif

  /*
  if (args.GetOptionString('q').IsEmpty()) {
    endpoint.behind_masq = FALSE;
  } else {
    endpoint.masqAddressPtr = new PIPSocket::Address(args.GetOptionString('q'));
    endpoint.behind_masq = TRUE;
    cout << "Masquerading as address " << *(endpoint.masqAddressPtr) << endl;
  } */


  if (!Initialised) { 
     AddAllCapabilities(0, 0, "*");
#if OPENMCU_VIDEO
	 RemoveCapability(H323Capability::e_ExtendVideo);  // Remove extendedVideo
     SetVideoFrameSize(MaxFrameSize);				   // Set to the Maximum allowable frameSize
#endif
    Initialised = TRUE;
  }

  // disable codecs as required
  PString disableCodecs = cfg.GetString(DisableCodecsKey);
  rsrc->Add(new PHTTPStringField(DisableCodecsKey, 50, disableCodecs));
  if (!disableCodecs.IsEmpty()) {
    PStringArray toRemove = disableCodecs.Tokenise(',', FALSE);
    capabilities.Remove(toRemove);
  }

#if 0 //  old MCU options
  int videoTxQual = 10;
  if (args.HasOption("videotxquality")) 
      videoTxQual = args.GetOptionString("videotxquality").AsInteger();
  endpoint.videoTxQuality = PMAX(1, PMIN(31, videoTxQual));

  int videoF = 2;
  if (args.HasOption("videofill")) 
    videoF = args.GetOptionString("videofill").AsInteger();
  endpoint.videoFill = PMAX(1, PMIN(99, videoF));

  int videoFPS = 10;
  if (args.HasOption("videotxfps")) 
    videoFPS = args.GetOptionString("videotxfps").AsInteger();
  endpoint.videoFramesPS = PMAX(1,PMIN(30,videoFPS));

  int videoBitRate = 0; //disable setting videoBitRate.
  if (args.HasOption("videobitrate")) {
    videoBitRate = args.GetOptionString("videobitrate").AsInteger();
    videoBitRate = 1024 * PMAX(16, PMIN(2048, videoBitRate));
  }
  endpoint.videoBitRate = videoBitRate;
#endif

  AddAllUserInputCapabilities(0, 2);

  switch (gkMode) {
    default:
    case Gatekeeper_None:
      break;

    case Gatekeeper_Find:
      if (!DiscoverGatekeeper(new H323TransportUDP(*this)))
        PSYSTEMLOG(Error, "No gatekeeper found");
	  else
		PSYSTEMLOG(Info, "Found Gatekeeper: " << gatekeeper);
      break;

    case Gatekeeper_Explicit:
      if (!SetGatekeeper(gkName, new H323TransportUDP(*this)))
        PSYSTEMLOG(Error, "Error registering with gatekeeper at \"" << gkName << '"');
	  else
		PSYSTEMLOG(Info, "Registered with Gatekeeper: " << gkName);
  }

  PTRACE(2, "MCU\tCodecs (in preference order):\n" << setprecision(2) << GetCapabilities());;
}

H323Connection * OpenMCUH323EndPoint::CreateConnection(
      unsigned callReference,
      void * userData,
      H323Transport *,
      H323SignalPDU *)

{
  return new OpenMCUH323Connection(*this, callReference, userData);
}

void OpenMCUH323EndPoint::TranslateTCPAddress(PIPSocket::Address &localAddr, const PIPSocket::Address &remoteAddr)
{

//  if (this->behind_masq && 
//      (remoteAddr.Byte1() != 192)) {
//    localAddr = *(this->masqAddressPtr);
//  }
  return;
}

PString OpenMCUH323EndPoint::GetRoomStatus(const PString & block)
{
  PString substitution;
  PWaitAndSignal m(conferenceManager.GetConferenceListMutex());
  ConferenceListType & conferenceList = conferenceManager.GetConferenceList();

  ConferenceListType::iterator r;
  for (r = conferenceList.begin(); r != conferenceList.end(); ++r) {

    // make a copy of the repeating html chunk
    PString insert = block;
    PStringStream members;
    members << "<table border=1>"
               "<tr>"
                 "<th>"
                 "&nbsp;Name&nbsp;"
                 "</th><th>"
                 "&nbsp;Duration&nbsp;"
                 "</th><th>"
                 "&nbsp;Codec&nbsp;"
                 "</th><th>"
                 "&nbsp;RTP Packets/Bytes tx&nbsp;"
                 "</th><th>"
                 "&nbsp;RTP Packets/Bytes rx&nbsp;"
                 "</th>"
#if OPENMCU_VIDEO
                 "<th>"
                 "&nbsp;TX&nbsp;Video&nbsp;frame&nbsp;rate/RX&nbsp;Video&nbsp;frame&nbsp;rate&nbsp;"
                 "</th>"
#endif
                 "</tr>";

    Conference & conference = *(r->second);
    size_t memberListSize = 0;
    {
      PWaitAndSignal m(conference.GetMutex());
      Conference::MemberList & memberList = conference.GetMemberList();
      memberListSize = memberList.size();
      Conference::MemberList::const_iterator s;
      for (s = memberList.begin(); s != memberList.end(); ++s) {
        ConferenceMember * member = s->second;
        H323Connection_ConferenceMember * connMember = dynamic_cast<H323Connection_ConferenceMember *>(member);
        if (connMember != NULL) {
          OpenMCUH323Connection * conn = (OpenMCUH323Connection *)FindConnectionWithLock(connMember->GetH323Token());
          if (conn != NULL) {
            PTime now;
            PTime callStart = conn->GetConnectionStartTime();
            RTP_Session * session = conn->GetSession(RTP_Session::DefaultAudioSessionID);
            members << "<tr>"
                      "<td>"
                    << conn->GetRemotePartyName()
                    << (member->IsMCU() ? "<BR>(MCU)" : "")

                    << "</td><td>"   
                    << (now -callStart)
                    << "</td><td>"   
                    << conn->GetAudioTransmitCodecName() << '/' << conn->GetAudioReceiveCodecName()
#if OPENMCU_VIDEO
                    << "<BR>" << conn->GetVideoTransmitCodecName() << '/' << conn->GetVideoReceiveCodecName()
#endif
                    << "</td><td>"   
                    << session->GetPacketsSent() << '/' << session->GetOctetsSent() 
                    << "</td><td>"   
                    << session->GetPacketsReceived() << '/' << session->GetOctetsReceived()
                    << "</td>";
            conn->Unlock();
          }
        }
        members 
#if OPENMCU_VIDEO
                << "<td>"   
                << member->GetVideoTxFrameRate() << '/' << member->GetVideoRxFrameRate()
                << "</td>"   
#endif
                << "</tr>";
      }
    }
    members << "</table>";
    SpliceMacro(insert, "RoomName",        conference.GetNumber());
    SpliceMacro(insert, "RoomMemberCount", PString(PString::Unsigned, (long)memberListSize));
    SpliceMacro(insert, "RoomMembers",     members);
    substitution += insert;
  }

  return substitution;
}

PString OpenMCUH323EndPoint::GetMonitorText()
{
  PStringStream output;

  PWaitAndSignal m(conferenceManager.GetConferenceListMutex());
  ConferenceListType & conferenceList = conferenceManager.GetConferenceList();

  output << "Room Count: " << (int)conferenceList.size() << "\n"
         << "Max Room Count: " << conferenceManager.GetMaxConferenceCount() << "\n";

  ConferenceListType::iterator r;
  PINDEX confNum = 0;
  for (r = conferenceList.begin(); r != conferenceList.end(); ++r) {

    Conference & conference = *(r->second);
    PWaitAndSignal m(conference.GetMutex());
    Conference::MemberList & memberList = conference.GetMemberList();

    output << "\n[Conference "     << ++confNum << "]\n"
           << "Title: "            <<  conference.GetNumber() << "\n"
           << "ID: "               << conference.GetID() << "\n"
           << "Duration: "         << (PTime() - conference.GetStartTime()) << "\n"
           << "Member Count: "     << (int)memberList.size() << "\n"
           << "Max Member Count: " << conference.GetMaxMemberCount() << "\n";
 
    Conference::MemberList::const_iterator s;
    PINDEX num = 0;
    for (s = memberList.begin(); s != memberList.end(); ++s) {
      ConferenceMember * member = s->second;
      if (member != NULL) {
        PStringStream hdr; hdr << "Member " << ++num << " ";
        output << hdr << "Title: " << member->GetTitle() << "\n"
               << hdr << "Duration: " << (PTime() - member->GetStartTime()) << "\n"
               << member->GetMonitorInfo(hdr);
      }
    }
  }

  return output;
}

PBoolean OpenMCUH323EndPoint::OutgoingConferenceRequest(const PString & room)
{
  // create/find the conference
  PBoolean stat = conferenceManager.MakeAndLockConference(room) != NULL;
  conferenceManager.UnlockConference();
  return stat;
}

PString OpenMCUH323EndPoint::IncomingConferenceRequest(H323Connection & connection, 
                                                  const H323SignalPDU & setupPDU)
{
  const H225_Setup_UUIE & setup = setupPDU.m_h323_uu_pdu.m_h323_message_body;

  /*
   Here is the algorithm used for joining a conference.

   - If the conference goal is e_invite, then refuse the call

   - if the conference ID matches the ID of an existing conference, then join that conference

   - If there is no destination address, join the default conference

   - If there is a destination address, try and match the destination address
     to a conference number. If there is a match, join to that conference

   - If the destination address does not match any conference, create a new conference
  */

  // get the conference ID from the incoming call
  OpalGloballyUniqueID conferenceID = setup.m_conferenceID;

  PString roomToJoin;

  // check the conference ID
  if (conferenceManager.HasConference(conferenceID, roomToJoin)) {
    PTRACE(3, "MCU\tUsing conference ID to join existing room " << roomToJoin);
    return roomToJoin;
  }

  // look at destination addresses
  PINDEX i;
  for (i = 0; (i < setup.m_destinationAddress.GetSize()); i++) {
    roomToJoin = H323GetAliasAddressString(setup.m_destinationAddress[i]);
    if (conferenceManager.HasConference(roomToJoin)) {
      PTRACE(3, "MCU\tJoining room specified by destination address " << roomToJoin);
      return roomToJoin;
    }
  }

  // look at Q931 called party number
  if (roomToJoin.IsEmpty() && !setupPDU.GetQ931().GetCalledPartyNumber(roomToJoin) && roomToJoin.IsEmpty()) {
    if (conferenceManager.HasConference(roomToJoin)) {
      PTRACE(3, "MCU\tJoining room specified by Q.931 called party " << roomToJoin);
      return roomToJoin;
    }
  }

  // if there is a room to create, then join this call to that conference
  if (roomToJoin.IsEmpty()) 
    roomToJoin = OpenMCU::Current().GetDefaultRoomName();

  if (!roomToJoin.IsEmpty()) {
    PTRACE(3, "MCU\tJoining default room " << roomToJoin);
    return roomToJoin;
  }

  PTRACE(3, "MCU\tRefusing call because no room specified, and no default room");
  return PString::Empty();
}

///////////////////////////////////////////////////////////////

NotifyH245Thread::NotifyH245Thread(Conference & conference, PBoolean _join, ConferenceMember * _memberToIgnore)
  : PThread(10000, AutoDeleteThread), join(_join), memberToIgnore(_memberToIgnore)
{ 
  mcuNumber  = conference.GetMCUNumber();
  terminalIdToSend = memberToIgnore->GetTerminalNumber();

  // create list of connections to notify
  Conference::MemberList::const_iterator r;
  for (r = conference.GetMemberList().begin(); r != conference.GetMemberList().end(); r++) {
    ConferenceMember * mbr = r->second;
    if (mbr != memberToIgnore) {
      H323Connection_ConferenceMember * h323Mbr = dynamic_cast<H323Connection_ConferenceMember *>(mbr);
      if (h323Mbr != NULL)
        tokens += h323Mbr->GetH323Token();
    }
  }

  Resume(); 
}



void NotifyH245Thread::Main()
{
  OpenMCUH323EndPoint & ep = OpenMCU::Current().GetEndpoint();

  // send H.245 message on each connection in turn
  PINDEX i;
  for (i = 0; i < tokens.GetSize(); ++i) {
    H323Connection * conn = ep.FindConnectionWithLock(tokens[i]);
    if (conn != NULL) {
      OpenMCUH323Connection * h323Conn = dynamic_cast<OpenMCUH323Connection *>(conn);

      H323ControlPDU pdu;
      H245_ConferenceIndication & ind = pdu.Build(H245_IndicationMessage::e_conferenceIndication);
      ind.SetTag(join ? H245_ConferenceIndication::e_terminalJoinedConference : H245_ConferenceIndication::e_terminalLeftConference);
      H245_TerminalLabel & terminalId = ind;
      terminalId.m_mcuNumber      = mcuNumber;
      terminalId.m_terminalNumber = terminalIdToSend;

      h323Conn->WriteControlPDU(pdu);

      h323Conn->Unlock();
    }
  }
}

///////////////////////////////////////////////////////////////

OpenMCUH323Connection::OpenMCUH323Connection(OpenMCUH323EndPoint & _ep, unsigned callReference, void * userData)
  : H323Connection(_ep, callReference), ep(_ep), isMCU(FALSE)
{
  conference       = NULL;
  conferenceMember = NULL;
  welcomeState     = NotStartedYet;

  if (userData != NULL) {
    requestedRoom    = *(PString *)userData;
    delete (PString *)userData;
  }

  audioReceiveCodecName = audioTransmitCodecName = "none";

#if OPENMCU_VIDEO
  videoGrabber = NULL;
  videoDisplay = NULL;
  videoReceiveCodecName = videoTransmitCodecName = "none";
#endif
}

OpenMCUH323Connection::~OpenMCUH323Connection()
{
}

PBoolean OpenMCUH323Connection::OnReceivedSignalSetup(const H323SignalPDU & setupPDU)
{
  // get a good name from the other end
  const H225_Setup_UUIE & setup = setupPDU.m_h323_uu_pdu.m_h323_message_body;

  remoteName = setupPDU.GetQ931().GetDisplayName();
  if (remoteName.IsEmpty()) {
    if (setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress))
      remoteName = H323GetAliasAddressString(setup.m_sourceAddress[0]);
  }

  if (remoteName.IsEmpty()) {
    if (!setupPDU.GetQ931().GetCallingPartyNumber(remoteName))
      remoteName.MakeEmpty();
  }

  if (remoteName.IsEmpty())
    remoteName = GetRemotePartyName();

  isMCU = setup.m_sourceInfo.m_mc;

  return H323Connection::OnReceivedSignalSetup(setupPDU);
}

PBoolean OpenMCUH323Connection::OnReceivedCallProceeding(const H323SignalPDU & proceedingPDU)
{
  const H225_CallProceeding_UUIE & proceeding = proceedingPDU.m_h323_uu_pdu.m_h323_message_body;
  isMCU = proceeding.m_destinationInfo.m_mc;
  return H323Connection::OnReceivedCallProceeding(proceedingPDU);
}

void OpenMCUH323Connection::CleanUpOnCallEnd()
{
  LeaveConference();

  H323Connection::CleanUpOnCallEnd();
}

H323Connection::AnswerCallResponse OpenMCUH323Connection::OnAnswerCall(const PString & /*caller*/,
                                                                  const H323SignalPDU & setupPDU,
                                                                  H323SignalPDU & /*connectPDU*/)
{
  requestedRoom = ep.IncomingConferenceRequest(*this, setupPDU);

  if (requestedRoom.IsEmpty())
    return AnswerCallDenied;

  return AnswerCallNow;
}

void OpenMCUH323Connection::OnEstablished()
{
  H323Connection::OnEstablished();
}

class MemberDeleteThread : public PThread
{
  public:
    MemberDeleteThread(OpenMCUH323EndPoint * _ep, Conference * _conf, ConferenceMember * _cm)
      : PThread(10000, AutoDeleteThread), ep(_ep), conf(_conf), cm(_cm)
    {
      Resume();
    }

    void Main()
    {
      cm->WaitForClose();
      if (conf->RemoveMember(cm))
        ep->GetConferenceManager().RemoveConference(conf->GetID());
      delete cm;
    }

  protected:
    OpenMCUH323EndPoint * ep;
    Conference * conf;
    ConferenceMember * cm;
};

void OpenMCUH323Connection::JoinConference(const PString & roomToJoin)
{
  PWaitAndSignal m(connMutex);

  if (conference != NULL)
    return;

  PBoolean joinSuccess = FALSE;

  if (!roomToJoin.IsEmpty()) {
    // create or join the conference
    ConferenceManager & manager = ((OpenMCUH323EndPoint &)ep).GetConferenceManager();
    Conference * newConf = manager.MakeAndLockConference(roomToJoin);
    if (newConf != NULL) {
      conference = newConf;
      conferenceIdentifier = conference->GetID();

      conferenceMember = new H323Connection_ConferenceMember(conference, ep, GetCallToken(), this, isMCU);

      if (!conferenceMember->IsJoined())
        PTRACE(1, "MCU\tMember connection refused");
      else
        joinSuccess = TRUE;

      manager.UnlockConference();

      if(!joinSuccess) {
        new MemberDeleteThread(&ep, conference, conferenceMember);
        conferenceMember = NULL;
        conference = NULL;
      }
    }
  }

  if(!joinSuccess)
    ChangeWelcomeState(JoinFailed);
}

void OpenMCUH323Connection::LeaveConference()
{
  PWaitAndSignal m(connMutex);

  if (conference != NULL && conferenceMember != NULL) {
    LogCall();

    new MemberDeleteThread(&ep, conference, conferenceMember);
    conferenceMember = NULL;
    conference = NULL;

    // - called from another thread than usual
    // - may clear the call immediately
    ChangeWelcomeState(ConferenceEnded);
  }
}

PBoolean OpenMCUH323Connection::OpenAudioChannel(PBoolean isEncoding, unsigned /* bufferSize */, H323AudioCodec & codec)
{
  PWaitAndSignal m(connMutex);

  PString codecName = codec.GetMediaFormat();

  codec.SetSilenceDetectionMode( H323AudioCodec::NoSilenceDetection );

  if (!isEncoding) {
    audioReceiveCodecName = codecName;
    codec.AttachChannel(new IncomingAudio(ep, *this), TRUE);
  } else {
    audioTransmitCodecName = codecName;
    codec.AttachChannel(new OutgoingAudio(ep, *this), TRUE);
  }

  return TRUE;
}

#if OPENMCU_VIDEO
PBoolean OpenMCUH323Connection::OpenVideoChannel(PBoolean isEncoding, H323VideoCodec & codec)
{
  PWaitAndSignal m(connMutex);

  if (isEncoding) {
    videoTransmitCodecName = codec.GetMediaFormat();

    PVideoChannel * channel = new PVideoChannel;
    videoGrabber = new PVideoInputDevice_OpenMCU(*this);
    if (videoGrabber == NULL) {
      PTRACE(3, "Cannot create MCU video input driver");
      return FALSE;
    }

    if (!InitGrabber(videoGrabber, CIF_WIDTH, CIF_HEIGHT)) {
      delete videoGrabber;
      videoGrabber = NULL;
      return FALSE;
    }

    codec.SetTxQualityLevel(ep.GetVideoTxQuality());

    videoGrabber->Start();
    channel->AttachVideoReader(videoGrabber);

    if (!codec.AttachChannel(channel,TRUE))
      return FALSE;

  } else {

    videoReceiveCodecName = codec.GetMediaFormat();

    videoDisplay = new PVideoOutputDevice_OpenMCU(*this);

    if (!videoDisplay->Open("")) {
      delete videoDisplay;
      return FALSE;
    }
     
    videoDisplay->SetFrameSize(codec.GetWidth(), codec.GetHeight()); // needed to enable resize
    videoDisplay->SetColourFormatConverter("YUV420P");

    PVideoChannel * channel = new PVideoChannel; 
    channel->AttachVideoPlayer(videoDisplay); 
    if (!codec.AttachChannel(channel,TRUE))
      return FALSE;

  }
 
  return TRUE;
}

PBoolean OpenMCUH323Connection::InitGrabber(PVideoInputDevice * grabber, int newFrameWidth, int newFrameHeight)
{
  PTRACE(4, "Video grabber set to " << newFrameWidth << "x" << newFrameHeight);

  //if (!(pfdColourFormat.IsEmpty()))
  //  grabber->SetPreferredColourFormat(pfdColourFormat);

  if (!grabber->Open("", FALSE)) {
    PTRACE(3, "Failed to open the video input device");
    return FALSE;
  }

  //if (!grabber->SetChannel(ep.GetVideoPlayMode())) {
  //  PTRACE(3, "Failed to set channel to " << ep.GetVideoPlayMode());
  //  return FALSE;
  //}

  //if (!grabber->SetVideoFormat(
  //    ep.GetVideoIsPal() ? PVideoDevice::PAL : PVideoDevice::NTSC)) {
  //  PTRACE(3, "Failed to set format to " << (ep.GetVideoIsPal() ? "PAL" : "NTSC"));
  //  return FALSE;
  //}

  if (!grabber->SetColourFormatConverter("YUV420P") ) {
    PTRACE(3,"Failed to set format to yuv420p");
    return FALSE;
  }

  if (ep.GetVideoFrameRate() != 0) {
    if (!grabber->SetFrameRate(ep.GetVideoFrameRate())) {
      PTRACE(3, "Failed to set framerate to " << ep.GetVideoFrameRate());
      return FALSE;
    }
  }

  if (!grabber->SetFrameSizeConverter(newFrameWidth,newFrameHeight,FALSE)) {
    PTRACE(3, "Failed to set frame size to " << newFrameWidth << "x" << newFrameHeight);
    return FALSE;
  }

  return TRUE;
}

#endif

void OpenMCUH323Connection::OnUserInputString(const PString & str)
{
  PWaitAndSignal m(connMutex);

  if (conferenceMember != NULL)
    conferenceMember->SendUserInputIndication(str);
}


PBoolean OpenMCUH323Connection::OnIncomingAudio(const void * buffer, PINDEX amount)
{
  PWaitAndSignal m(connMutex);

  // If record file is open, write data to it
  if (recordFile.IsOpen()) {
    recordFile.Write(buffer, amount);

    recordDuration += amount / 2;
    if (recordDuration > recordLimit) {
      recordFile.Close();
      OnFinishRecording();
    }
    else {
      const WORD * samples = (const WORD *)buffer;
      PINDEX sampleCount = amount / 2;
      PBoolean silence = TRUE;
      while (sampleCount-- > 0 && silence) {
        if (*samples > 100 || *samples < -100)
          silence = FALSE;
        ++samples;
      }
      if (!silence)
        recordSilenceCount = 0;
      else {
        recordSilenceCount += amount / 2;
        if ((recordSilenceThreshold > 0) && (recordSilenceCount >= recordSilenceThreshold)) {
          recordFile.Close();
          OnFinishRecording();
        }
      }
    }
  }

  else if (conferenceMember != NULL)
    conferenceMember->WriteAudio(buffer, amount);

  return TRUE;
}

void OpenMCUH323Connection::StartRecording(const PFilePath & filename, unsigned limit, unsigned threshold)
{
  if (!recordFile.Open(filename, PFile::ReadWrite, PFile::Create | PFile::Truncate))
    return;

  recordSilenceCount = 0;
  recordDuration = 0;

  recordSilenceThreshold = threshold * 8000;
  recordLimit            = limit * 8000;
}

void OpenMCUH323Connection::OnFinishRecording()
{
}

PBoolean OpenMCUH323Connection::OnOutgoingAudio(void * buffer, PINDEX amount)
{
  // When the prodedure begins, play the welcome file
  if (welcomeState == NotStartedYet) {
    ChangeWelcomeState(PlayingWelcome);
  }

  for (;;) {
    // Do actions that are not triggered by events
    OnWelcomeProcessing();

    // If a wave is not playing, we may continue now
    if (!playFile.IsOpen())
      break;

    // Wait for wave file completion
    if (playFile.Read(buffer, amount)) {
      int len = playFile.GetLastReadCount();
      if (len < amount) {
        memset(((BYTE *)buffer)+len, 0, amount-len);
      }
      //playDelay.Delay(amount/16);

      // Exit now since the buffer is ready
      return TRUE;
    }

    PTRACE(4, "MCU\tFinished playing file");
    playFile.Close();

    // Wave completed, if no event should be fired
    //  then we may continue now
    if(!wavePlayingInSameState)
      break;

    // Fire wave completion event
    OnWelcomeWaveEnded();

    // We should repeat the loop now because the callback
    //  above might have started a new wave file
  }

  PWaitAndSignal m(connMutex);

  // If a we are connected to a conference and no wave
  //  is playing, read data from the conference
  if (conferenceMember != NULL) {
    conferenceMember->ReadAudio(buffer, amount);
    return TRUE;
  }

  // Generate silence
  return FALSE;
}

///////////////////////////////////////////////////////////////////////////////////////

void OpenMCUH323Connection::ChangeWelcomeState(int newState)
{
  PWaitAndSignal m(connMutex);

  if(welcomeState != newState)
  {
    PTRACE(4, "MCU\tEntering welcome state " << newState);
    welcomeState = newState;
    wavePlayingInSameState = FALSE;
    OnWelcomeStateChanged();
  }
}

void OpenMCUH323Connection::PlayWelcomeFile(PBoolean useTheFile, PFilePath & fileToPlay)
{
  playFile.Close();

  wavePlayingInSameState = TRUE;

  if(useTheFile) {
    if(playFile.Open(fileToPlay, PFile::ReadOnly))
    {
      PTRACE(4, "MCU\tPlaying welcome procedure file " << fileToPlay);
      return;
    }
    else
      PTRACE(3, "MCU\tFailed to play welcome procedure file " << fileToPlay);
  }

  // File not played, call the wave end callback anyway
  OnWelcomeWaveEnded();
}

void OpenMCUH323Connection::OnWelcomeStateChanged()
{
  PFilePath fn;

  OpenMCU & mcu = OpenMCU::Current();

  switch(welcomeState) {

    case PlayingWelcome:
      // Welcome file not implemented yet
      PlayWelcomeFile(FALSE, fn);
      break;

    case PlayingConnecting:
      PlayWelcomeFile(mcu.GetConnectingWAVFile(fn), fn);
      break;

    case CompleteConnection:
      JoinConference(requestedRoom);
      break;

    case JoinFailed:
    case ConferenceEnded:
      // Goodbye file not implemented yet
      PlayWelcomeFile(FALSE, fn);
      break;

    default:
      // Do nothing
      break;
  }
}

void OpenMCUH323Connection::OnWelcomeProcessing()
{
}

void OpenMCUH323Connection::OnWelcomeWaveEnded()
{
  switch(welcomeState) {

    case PlayingWelcome:
      ChangeWelcomeState(PlayingConnecting);
      break;

    case PlayingConnecting:
      ChangeWelcomeState(CompleteConnection);
      break;

    case JoinFailed:
    case ConferenceEnded:
      ClearCall();
      break;

    default:
      // Do nothing
      break;
  }
}

///////////////////////////////////////////////////////////////////////////////////////

#if OPENMCU_VIDEO

//
// this function is called whenever a connection needs a frame of video for output
//

PBoolean OpenMCUH323Connection::OnOutgoingVideo(void * buffer, int width, int height, PINDEX & amount)
{
  PWaitAndSignal m(connMutex);

  if (conferenceMember != NULL)
    conferenceMember->ReadVideo(buffer, width, height, amount);
  else if (!GetPreMediaFrame(buffer, width, height, amount)) {
    if ((width == CIF16_WIDTH) && (height == CIF16_HEIGHT))
      MCUVideoMixer::Fill16CIFYUVFrame(buffer, 0, 0, 0);
    else if ((width == CIF4_WIDTH) && (height == CIF4_HEIGHT))
      MCUVideoMixer::Fill4CIFYUVFrame(buffer, 0, 0, 0);
    else if ((width == CIF_WIDTH) && (height == CIF_HEIGHT))
      MCUVideoMixer::FillCIFYUVFrame(buffer, 0, 0, 0);
    else if ((width == QCIF_WIDTH) && (height == QCIF_HEIGHT))
      MCUVideoMixer::FillQCIFYUVFrame(buffer, 0, 0, 0);
  }

  return TRUE;
}

PBoolean OpenMCUH323Connection::GetPreMediaFrame(void * buffer, int width, int height, PINDEX & amount)
{
  return OpenMCU::Current().GetPreMediaFrame(buffer, width, height, amount);
}

//
// this function is called whenever a connection receives a frame of video
//

PBoolean OpenMCUH323Connection::OnIncomingVideo(const void * buffer, int width, int height, PINDEX amount)
{
  if (conferenceMember != NULL)
    conferenceMember->WriteVideo(buffer, width, height, amount);
  return TRUE;
}

#endif // OPENMCU_VIDEO


///////////////////////////////////////////////////////////////

H323Connection_ConferenceMember::H323Connection_ConferenceMember(Conference * _conference, OpenMCUH323EndPoint & _ep, const PString & _h323Token, ConferenceMemberId _id, PBoolean _isMCU)
  : ConferenceMember(_conference, _id, _isMCU), ep(_ep), h323Token(_h323Token)
{ 
  conference->AddMember(this);
}

H323Connection_ConferenceMember::~H323Connection_ConferenceMember()
{
  PTRACE(4, "H323Connection_ConferenceMember deleted");
}

void H323Connection_ConferenceMember::Close()
{
  OpenMCUH323Connection * conn = (OpenMCUH323Connection *)ep.FindConnectionWithLock(h323Token);
  if (conn != NULL) {
    conn->LeaveConference();
    conn->Unlock();
  }
}

PString H323Connection_ConferenceMember::GetTitle() const
{
  PString output;
  OpenMCUH323Connection * conn = (OpenMCUH323Connection *)ep.FindConnectionWithLock(h323Token);
  if (conn != NULL) {
    output = conn->GetRemoteName();
    conn->Unlock();
  }
  return output;
}

PString H323Connection_ConferenceMember::GetMonitorInfo(const PString & hdr)
{ 
  PStringStream output;
  OpenMCUH323Connection * conn = (OpenMCUH323Connection *)ep.FindConnectionWithLock(h323Token);
  if (conn != NULL) {
    output << hdr << "Remote Address: " << conn->GetRemotePartyAddress() << "\n"
           << hdr << "AudioCodecs: " << conn->GetAudioTransmitCodecName() << '/' << conn->GetAudioReceiveCodecName() << "\n"
#if OPENMCU_VIDEO
           << hdr << "VideoCodecs: " << conn->GetVideoTransmitCodecName() << '/' << conn->GetVideoReceiveCodecName() << "\n"
#endif           
           ;
    conn->Unlock();
  }
  return output;
}

///////////////////////////////////////////////////////////////

void OpenMCUH323Connection::LogCall(const PBoolean accepted)
{
  H323TransportAddress address = GetControlChannel().GetRemoteAddress();
  PIPSocket::Address ip;
  WORD port;
  PStringStream stringStream, timeStream;
  address.GetIpAndPort(ip, port);
  timeStream << GetConnectionStartTime().AsString("hh:mm:ss");
  stringStream << ' ' << "caller-ip:" << ip << ':' << port << ' '
	             << GetRemotePartyName() 
               << " room:" << ((conference != NULL) ? conference->GetNumber() : PString());

  if (accepted) {
    PStringStream connectionDuration;
    connectionDuration << setprecision(0) << setw(5) << (PTime() - GetConnectionStartTime());
    OpenMCU::Current().LogMessage(timeStream + stringStream	+ " connection duration:" + connectionDuration);
  }
  else 
    OpenMCU::Current().LogMessage(timeStream + " Call denied:" + stringStream);		
}

///////////////////////////////////////////////////////////////

OutgoingAudio::OutgoingAudio(H323EndPoint & _ep, OpenMCUH323Connection & _conn)
  : ep(_ep), conn(_conn)
{
  os_handle = 0;
}

void OutgoingAudio::CreateSilence(void * buffer, PINDEX amount)
{
  memset(buffer, 0, amount);
  lastReadCount = amount;
}

PBoolean OutgoingAudio::Read(void * buffer, PINDEX amount)
{
  PWaitAndSignal mutexR(audioChanMutex);
  
  if (!IsOpen())
    return FALSE;

  // do the read call here, by calling conn.OnOutgoingAudio():
  if (!conn.OnOutgoingAudio(buffer, amount))
    CreateSilence(buffer, amount);

  delay.Delay(amount / 16);

  lastReadCount = amount;

  return TRUE;
}

PBoolean OutgoingAudio::Close()
{
  if (!IsOpen()) 
    return FALSE;

  PWaitAndSignal mutexC(audioChanMutex);
  os_handle = -1;

  return TRUE;
}

///////////////////////////////////////////////////////////////////////////

IncomingAudio::IncomingAudio(H323EndPoint & _ep, OpenMCUH323Connection & _conn)
  : ep(_ep), conn(_conn)
{
  os_handle = 0;
}

PBoolean IncomingAudio::Write(const void * buffer, PINDEX amount)
{
  PWaitAndSignal mutexW(audioChanMutex);
  
  if (!IsOpen())
    return FALSE;

  conn.OnIncomingAudio(buffer, amount);

  delay.Delay(amount / 16);

  return TRUE;
}

PBoolean IncomingAudio::Close()
{
  if (!IsOpen())
    return FALSE;

  PWaitAndSignal mutexA(audioChanMutex);
  os_handle = -1;

  return TRUE;
}

///////////////////////////////////////////////////////////////////////////

