/*
 * Copyright (c) 2002 by Louis Zechtzer
 *
 * Permission to use, copy and distribute this software is hereby granted
 * under the terms of version 2 or any later version of the GNU General Public
 * License, as published by the Free Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/* 
 * Authors: Louis Zechtzer (lou@clarity.net)
 */

#ifndef __NET_H
#define __NET_H

#include <netinet/in.h>
#include <net/if.h>
#include <sys/types.h>

/*
 * The following values are used by callers of the networking code.  They
 * distinguish the type of message received for which message handler can
 * define behavior.
 */
#define NET_MSG_JOIN 	0x1
#define NET_MSG_LEAVE 	0x2
#define NET_MSG_ACK	0x3
#define NET_MSG_UNKNOWN 0x4

/* Return codes for all networking functions */
#define NET_SUCCESS	0x0
#define NET_FAIL	0x1
#define NET_INTERRUPTED 0x2

/* Maximum number of interfaces supported */
#define NET_MAX_IFS 6

/* All received messages are encapsulated for callers in struct net_msg */
typedef struct net_msg {
	struct in_addr src;
	struct in_addr src_ifs[NET_MAX_IFS - 1];
	unsigned short if_cnt;
	unsigned short type;
} net_msg_t;


/* 
 * Chose an orginization-local-scope multicast group address. See RFC2365.
 * Use a different address for testing to avoid problems with an existing 
 * cluster.
 */
#ifndef TESTING 
#define NET_MCAST_GROUP "239.192.0.1"
#else
#define NET_MCAST_GROUP "239.192.0.2"
#endif
#define NET_MCAST_PORT 1334

/* Constants for messaging */

#define NET_MAGIC_NUMBER 0x64c4f01e
#define NET_JOIN_MSG "j"
#define NET_EXIT_MSG "e"  /* XXX - need a way to remove entries */
#define NET_ACK_MSG "a"
#define NET_MSG_SIZE 1

/* Aggregate message size is (magic number) + (type) + (6 4-byte IP addrs) */
#define NET_AG_MSG_SIZE 4 + NET_MSG_SIZE + (4 * NET_MAX_IFS)

/* Used for net_send_msg to specify transmission from all interfaces */
#define NET_ALL_IFS -1

/*
 * net_*_params structures are used to encapsulate socket connection
 * information.
 */
typedef union ctl_un {
	struct 			cmsghdr cm_hdr;
	char 			cm_data[CMSG_SPACE(sizeof(struct cmsghdr)) +
					CMSG_SPACE(sizeof(struct in_pktinfo))];
} net_ctl_un_t;

typedef struct net_send_params {
	int 			sp_sfd;
} net_send_params_t;

typedef struct net_recv_params {
	int			rp_sfd;
	net_ctl_un_t 		rp_ctl;
} net_recv_params_t;

typedef struct net_params {
	net_send_params_t 	np_sp[NET_MAX_IFS];
	net_recv_params_t 	np_rp;
	struct sockaddr_in 	np_sin;
	struct in_addr 		np_if_addrs[NET_MAX_IFS];
	unsigned short 		np_if_cnt;
	unsigned short 		np_multicast_ttl;
	unsigned short 		np_initialized;
} net_params_t;

extern int net_initialize(net_params_t *, char *, int);
extern int net_send_join_msg(net_params_t *);
extern int net_recv_msg(net_params_t *, void(*)(net_msg_t *, void *), void *);
extern int net_send_ack_msg(net_params_t *);
extern int net_finalize(net_params_t *);
extern int net_get_if_addrs(net_params_t *, struct in_addr *, int *);

#ifdef TESTING
extern void net_test();
#endif

#endif /* __NET_H */
