/*************************************************************************
 *
 *  $RCSfile: BtreeDict.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/01/27 10:08:26 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
package com.sun.xmlsearch.db;

import java.io.*;
import com.sun.xmlsearch.util.IntegerArray;

public class BtreeDict {
  protected static final int ENTHEADERLEN = 6;
  protected static final int BLOCKSIZE = 2048;
  protected static final int DATALEN = BLOCKSIZE - Block.HEADERLEN;
  protected static final int MaxKeyLength = 255;
  //!!! Careful with that number, Eugene
  protected static final int lastPtrIndex = 508;

  protected class DictBlock extends Block {
    public DictBlock() {
      super(BLOCKSIZE);
    }
  
    public final int free() {
      return _free + firstEntry();
    }
  
    public final int numberOfEntries() {
      return integerAt(0);
    }
  
    public final int nthPointer(int n) {
      return integerAt(4*(n + 1));
    }
  
    public final int getChildIdx(int index) {
      return nthPointer(lastPtrIndex - index);
    }
  
    public final int entryKeyLength(int i) {
      return _data[i] & 0xFF;
    }
  
    public final int entryCompression(int i) {
      return _data[i + 1] & 0xFF;
    }
  
    public final int entryID(int i) {
      return integerAt(i + 2);
    }
  
    public final int entryLength(int entry) {
      return ENTHEADERLEN + entryKeyLength(entry);
    }
     
    public final int entryKey(int entry) {
      return entry + ENTHEADERLEN;
    }

    public final int firstEntry() {
      return 4;
    }

    public final int nextEntry(int entry) {
      return entry + entryLength(entry);
    }
  
    public final void restoreKeyInBuffer(int entry, byte[] buffer) {
      int howMany = entryKeyLength(entry);
      int where = entryCompression(entry);
      int from = entryKey(entry);
      while (howMany-- > 0)
	buffer[where++] = _data[from++];
    }
    
    public final String restoreKey(int entry, byte[] buffer) {
      int howMany = entryKeyLength(entry);
      int where = entryCompression(entry);
      int from = entryKey(entry);
      while (howMany-- > 0)
	buffer[where++] = _data[from++];
      return new String(buffer, 0, where);
    }
  
    public final String findID(int id) throws Exception {
      byte[] buffer = new byte[MaxKeyLength];
      int freeSpace = free();
      for (int ent = firstEntry(); ent < freeSpace; ent = nextEntry(ent))
	if (entryID(ent) == id) // found
	  return restoreKey(ent, buffer);
	else
	  restoreKeyInBuffer(ent, buffer);
      throw new Exception("ID not found in block");
    }
  
    protected void setBlockNumbers(int[] blocks) {
      for (int e = firstEntry(); e < _free; e = nextEntry(e))
	blocks[entryID(e)] = _number;
    }
  
    public void listBlock() {
      byte[] buffer = new byte[MaxKeyLength];
      final int freeSpace = free();
      int entryPtr = firstEntry();
      if (_isLeaf)
	while (entryPtr < freeSpace) {
	  System.out.println(restoreKey(entryPtr, buffer) + " " +
			     entryID(entryPtr));
	  entryPtr = nextEntry(entryPtr);
	}
      else
	System.out.println("not leaf");
    }

    protected void doMap(BtreeDict owner, EntryProcessor processor)
      throws Exception {
	byte[] buffer = new byte[MaxKeyLength];
	final int freeSpace = free();
	int entryPtr = firstEntry();
	if (_isLeaf)
	  while (entryPtr < freeSpace) {
	    processor.processEntry(restoreKey(entryPtr, buffer),
				   entryID(entryPtr));
	    entryPtr = nextEntry(entryPtr);
	  } else {
	    owner.lock(this);
	    int entryIdx  = 0;
	    while (entryPtr < freeSpace) {
	      owner.accessBlock(getChildIdx(entryIdx)).doMap(owner,processor);
	      processor.processEntry(restoreKey(entryPtr, buffer),
				     entryID(entryPtr));
	      entryPtr = nextEntry(entryPtr);
	      ++entryIdx;
	    }
	    owner.accessBlock(getChildIdx(entryIdx)).doMap(owner, processor);
	    owner.unlock(this);
	  }
    }
    
    public void withPrefix(BtreeDict owner, String prefix, int prefLen, IntegerArray result)
      throws Exception {
      byte[] buffer = new byte[MaxKeyLength];
      final int freeSpace = free();
      int entryPtr = firstEntry();
      if (_isLeaf)
	while (entryPtr < freeSpace) {
	  if (restoreKey(entryPtr, buffer).startsWith(prefix))
	    result.add(entryID(entryPtr));
	  entryPtr = nextEntry(entryPtr);
	}
      else {
	owner.lock(this);
	int entryIndex  = 0;
	while (entryPtr < freeSpace) {
	  String key = restoreKey(entryPtr, buffer);
	  if (key.length() > prefLen)
	    key = key.substring(0, prefLen);
	  int cmp = key.compareTo(prefix);

	  if (cmp < 0) {
	    entryPtr = nextEntry(entryPtr);
	    ++entryIndex;
	  }
	  else if (cmp == 0) {
	    result.add(entryID(entryPtr));
	    owner.accessBlock(getChildIdx(entryIndex)).withPrefix(owner, prefix, prefLen, result);
	    entryPtr = nextEntry(entryPtr);
	    ++entryIndex;
	  }
	  else {
	    owner.unlock(this);
	    owner.accessBlock(getChildIdx(entryIndex)).withPrefix(owner, prefix, prefLen, result);
	    return;
	  }
	}
	owner.unlock(this);
	owner.accessBlock(getChildIdx(numberOfEntries())).withPrefix(owner, prefix, prefLen, result);
      }
    }
  }
  // end of internal class

  protected BlockManager blockManager;
  protected int          root;
  protected int[]        blocks;
  
  protected BtreeDict() {/*empty*/}

  public BtreeDict(BtreeDictParameters params) throws Exception {
    init(params, false, new BlockFactory() {
      public Block makeBlock() {
	return new DictBlock();
      }
    });
    blocks = new int[params.getFreeID()];
    setBlocks(blocks);
  }

  public int fetch(String key) throws Exception {
    //    System.err.println("fetching " + key);
    int length = key.length();
    byte[] Key = new byte[length + 1];
    System.arraycopy(key.getBytes("UTF8"), 0, Key, 0, length);
    Key[length] = 0;		// sentinel
    return find(accessBlock(root), Key);
  }
  
  public String fetch(int conceptID) throws Exception {
    return findID(blocks[conceptID], conceptID);
  }
  
  public IntegerArray withPrefix(String prefix) throws Exception {
    IntegerArray result = new IntegerArray();
    accessBlock(root).withPrefix(this, prefix, prefix.length(), result);
    return result;
  }

  public void close() throws Exception {
    blockManager.close();
  }
  
  protected void init(BtreeDictParameters params, boolean update,
		      BlockFactory bfactory)
    throws Exception {
      blockManager = new BlockManager(params, update, bfactory);
      root = params.getRootPosition();
  }
  
  protected void lock(Block bl) {
    blockManager.lockBlock(bl._number);
  }
  
  protected void unlock(Block bl) {
    blockManager.unlockBlock(bl._number);
  }
  
  protected DictBlock accessBlock(int index) throws Exception {
    return (DictBlock)blockManager.accessBlock(index);
  }
  
  protected DictBlock child(DictBlock bl, int index) throws Exception {
    return accessBlock(bl.getChildIdx(index));
  }
  
  private String findID(int blNum, int id) throws Exception {
    return accessBlock(blNum).findID(id);
  }

  private int find(DictBlock bl, byte[] key, int index) throws Exception {
    return bl._isLeaf ? 0 : find(child(bl, index), key);
  }

  private int find(DictBlock bl, byte[] key) throws Exception {
    int inputKeyLen = key.length - 1;
    int entryPtr    = bl.firstEntry();
    int freeSpace   = bl.free();
    int nCharsEqual = 0;
    int compression = 0;

    for (int entryIdx = 0;;) {
      if (entryPtr == freeSpace)
	return find(bl, key, bl.numberOfEntries());
      else if (compression == nCharsEqual) {
	int keyLen = bl.entryKeyLength(entryPtr);
	int keyPtr = bl.entryKey(entryPtr), i;
	for (i = 0;
	     i < keyLen && key[nCharsEqual] == bl._data[keyPtr + i];
	     i++)
	  ++nCharsEqual;
	if (i == keyLen) {
	  if (nCharsEqual == inputKeyLen)
	    return bl.entryID(entryPtr);
	}
	else if ((key[nCharsEqual]&0xFF) < (bl._data[keyPtr + i]&0xFF))
	  return find(bl, key, entryIdx);
      }
      else if (compression < nCharsEqual) // compression dropped
	return find(bl, key, entryPtr == freeSpace
		    ? bl.numberOfEntries() : entryIdx);
      do {
	entryPtr = bl.nextEntry(entryPtr);
	++entryIdx;
      }
      while (bl.entryCompression(entryPtr) > nCharsEqual);
      compression = bl.entryCompression(entryPtr);
    }
  }

  protected void setBlocks(final int[] blocks) throws Exception {
    long start = System.currentTimeMillis();
    blockManager.mapBlocks(new BlockProcessor() {
      public void process(Block block) {
	((DictBlock)block).setBlockNumbers(blocks);
      }
    });
    debug((System.currentTimeMillis() - start) + " msec; DICTIONARY");
  }

  //    can go to Full
  public void map(EntryProcessor processor) throws Exception {
    accessBlock(root).doMap(this, processor);
  }

  /**
   * Debug code
   */
  private boolean debug=true;
  private void debug(String msg) {
    if (debug) {
      System.err.println("BtreeDict: "+msg);
    }
  }

  public void test() throws Exception {
    
    accessBlock(651).listBlock();
  }

  /*
  public static void main(String[] args) {
    try {
      Schema schema = new Schema(args[0], false);
      BtreeDictParameters params = new BtreeDictParameters(schema, "DICTIONARY");
      BtreeDict source = new BtreeDict(params);
      source.test();
    }
    catch (Exception e) {
      System.err.println(e);
      e.printStackTrace();
    }
  }
  */
}
