/*************************************************************************
 *
 *  $RCSfile: dx_parametricpolypolygon.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/26 17:22:23 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <canvas/debug.hxx>

#include "dx_parametricpolypolygon.hxx"
#include <dx_impltools.hxx>

#ifndef _DRAFTS_COM_SUN_STAR_RENDERING_TEXTURINGMODE_HPP_
#include <drafts/com/sun/star/rendering/TexturingMode.hpp>
#endif


using namespace ::com::sun::star;
using namespace ::drafts::com::sun::star;

namespace dxcanvas
{
	typedef ::boost::shared_ptr< Gdiplus::PathGradientBrush > 	PathGradientBrushSharedPtr;
	typedef ::boost::shared_ptr< Gdiplus::LinearGradientBrush > LinearGradientBrushSharedPtr;

    ParametricPolyPolygon* ParametricPolyPolygon::createLinearHorizontalGradient( const uno::Sequence< double >& 						leftColor, 
                                                                                  const uno::Sequence< double >& 						rightColor,
                                                                                  const uno::Reference< rendering::XGraphicDevice >& 	xDevice )
    {
        // TODO(P2): hold gradient brush statically, and only setup
        // the colors

        // Maybe cannot use this version, because LinearGradientBrush
        // seems to ignore WrapModeClamp

        const Gdiplus::ARGB aLeftColor( tools::sequenceToArgb( xDevice,
                                                               leftColor ) );
        const Gdiplus::ARGB aRightColor( tools::sequenceToArgb( xDevice,
                                                                rightColor ) );

        LinearGradientBrushSharedPtr pBrush(
            new Gdiplus::LinearGradientBrush( 
                Gdiplus::PointF(0.0f,
                                0.5f),
                Gdiplus::PointF(1.0f,
                                0.5f),
                tools::sequenceToArgb( xDevice,
                                       leftColor ),
                tools::sequenceToArgb( xDevice,
                                       rightColor ) ) );

        // TODO(F2): Clipping for non-functional CLAMP mode
        VectorOfBrushes aBrushes;
        aBrushes.push_back( BrushEntry( 
                                BrushSharedPtr( 
                                    new Gdiplus::SolidBrush( aLeftColor ) ),
                                        GraphicsPathSharedPtr() ) );
        aBrushes.push_back( BrushEntry( pBrush,
                                        GraphicsPathSharedPtr() ) );

        return new ParametricPolyPolygon( aBrushes );
    }

    ParametricPolyPolygon* ParametricPolyPolygon::createAxialHorizontalGradient( const uno::Sequence< double >& 					middleColor, 
                                                                                 const uno::Sequence< double >& 					endColor,
                                                                                 const uno::Reference< rendering::XGraphicDevice >& xDevice )
    {
        // TODO(P2): hold gradient brush statically, and only setup
        // the colors

        // Maybe cannot use this version, because LinearGradientBrush
        // seems to ignore WrapModeClamp

        const Gdiplus::ARGB aMiddleColor( tools::sequenceToArgb( xDevice,
                                                                 middleColor ) );
        const Gdiplus::ARGB aEndColor( tools::sequenceToArgb( xDevice,
                                                              endColor ) );

        LinearGradientBrushSharedPtr pBrush(
            new Gdiplus::LinearGradientBrush( 
                Gdiplus::PointF(-1.0f,
                                0.5f),
                Gdiplus::PointF(2.0f,
                                0.5f),
                aEndColor,
                aEndColor ) );

        Gdiplus::Color aColors[] = 
            {
                aEndColor,		// at 0.0
                aMiddleColor,	// at 0.5
                aEndColor		// at 1.0
            };
        
        Gdiplus::REAL aPositions[] = 
            {
                0.0,
                0.5,
                1.0
            };

        if( Gdiplus::Ok != pBrush->SetInterpolationColors( aColors,
                                                           aPositions,
                                                           sizeof( aPositions ) / sizeof(Gdiplus::REAL) ) )
        {
            // SetInterpolationColors failed
            throw uno::RuntimeException();
        }
        
        // TODO(F2): Clipping for non-functional CLAMP mode
        VectorOfBrushes aBrushes;
        aBrushes.push_back( BrushEntry( 
                                BrushSharedPtr( 
                                    new Gdiplus::SolidBrush( aEndColor ) ),
                                        GraphicsPathSharedPtr() ) );
        aBrushes.push_back( BrushEntry( pBrush,
                                        GraphicsPathSharedPtr() ) );

        return new ParametricPolyPolygon( aBrushes );
    }

    ParametricPolyPolygon* ParametricPolyPolygon::createCircularGradient( const uno::Sequence< double >& 						centerColor, 
                                                                          const uno::Sequence< double >& 						endColor,
                                                                          const uno::Reference< rendering::XGraphicDevice >& 	xDevice )
    {
        // TODO(P2): hold gradient brush statically, and only setup
        // the colors

        Gdiplus::GraphicsPath aPath;
        aPath.AddEllipse(0.0f, 0.0f, 1.0f, 1.0f);

        PathGradientBrushSharedPtr pBrush( 
            new Gdiplus::PathGradientBrush( &aPath ) );

        const Gdiplus::ARGB aCenterColor( tools::sequenceToArgb( xDevice,
                                                                 centerColor ) );
        const Gdiplus::ARGB aEndColor( tools::sequenceToArgb( xDevice,
                                                              endColor ) );
        Gdiplus::Color aColors[] = 
            {                
                aEndColor
            };

        INT nCount(1);

        if( Gdiplus::Ok != pBrush->SetSurroundColors( aColors,
                                                      &nCount ) ||
            nCount != 1 )
        {
            // SetSurroundColors failed
            throw uno::RuntimeException();
        }
            
        pBrush->SetCenterColor( aCenterColor );

        VectorOfBrushes aBrushes;
        aBrushes.push_back( BrushEntry( 
                                BrushSharedPtr( 
                                    new Gdiplus::SolidBrush( aEndColor ) ),
                                        GraphicsPathSharedPtr() ) );
        aBrushes.push_back( BrushEntry( pBrush,
                                        GraphicsPathSharedPtr() ) );

        return new ParametricPolyPolygon( aBrushes );
    }

    ParametricPolyPolygon* ParametricPolyPolygon::createRectangularGradient( const uno::Sequence< double >& 					centerColor, 
                                                                             const uno::Sequence< double >& 					endColor,
                                                                             const geometry::RealRectangle2D&					boundRect,
                                                                             const uno::Reference< rendering::XGraphicDevice >& xDevice)
    {
        Gdiplus::GraphicsPath aPath;
        aPath.AddRectangle( 
            Gdiplus::RectF( static_cast<Gdiplus::REAL>(boundRect.X1), 
                            static_cast<Gdiplus::REAL>(boundRect.Y1),
                            static_cast<Gdiplus::REAL>(boundRect.X2 - boundRect.X1),
                            static_cast<Gdiplus::REAL>(boundRect.Y2 - boundRect.Y1) ) );

        PathGradientBrushSharedPtr pBrush( 
            new Gdiplus::PathGradientBrush( &aPath ) );

        const Gdiplus::ARGB aCenterColor( tools::sequenceToArgb( xDevice,
                                                                 centerColor ) );
        const Gdiplus::ARGB aEndColor( tools::sequenceToArgb( xDevice,
                                                              endColor ) );
        Gdiplus::Color aColors[] = 
            {                
                aEndColor
            };

        INT nCount(1);

        if( Gdiplus::Ok != pBrush->SetSurroundColors( aColors,
                                                      &nCount ) ||
            nCount != 1 )
        {
            // SetSurroundColors failed
            throw uno::RuntimeException();
        }
            
        pBrush->SetCenterColor( aCenterColor );

        VectorOfBrushes aBrushes;
        aBrushes.push_back( BrushEntry( 
                                BrushSharedPtr( 
                                    new Gdiplus::SolidBrush( aEndColor ) ),
                                        GraphicsPathSharedPtr() ) );
        aBrushes.push_back( BrushEntry( pBrush,
                                        GraphicsPathSharedPtr() ) );

        return new ParametricPolyPolygon( aBrushes );
    }

    void SAL_CALL ParametricPolyPolygon::disposing()
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        maBrushes.clear();
    }

    uno::Reference< rendering::XPolyPolygon2D > SAL_CALL ParametricPolyPolygon::getOutline( double t ) throw (lang::IllegalArgumentException, uno::RuntimeException)
    {
        // TODO(F1): outline NYI
        return uno::Reference< rendering::XPolyPolygon2D >();
    }

    uno::Sequence< double > SAL_CALL ParametricPolyPolygon::getColor( double t ) throw (lang::IllegalArgumentException, uno::RuntimeException)
    {
        // TODO(F1): color NYI
        return uno::Sequence< double >();
    }

    uno::Sequence< double > SAL_CALL ParametricPolyPolygon::getPointColor( const geometry::RealPoint2D& point ) throw (lang::IllegalArgumentException, uno::RuntimeException)
    {
        // TODO(F1): point color NYI
        return uno::Sequence< double >();
    }

#define SERVICE_NAME "drafts.com.sun.star.rendering.ParametricPolyPolygon"

    ::rtl::OUString SAL_CALL ParametricPolyPolygon::getImplementationName(  ) throw (uno::RuntimeException)
    {
        return ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( PARAMETRICPOLYPOLYGON_IMPLEMENTATION_NAME ) );
    }

    sal_Bool SAL_CALL ParametricPolyPolygon::supportsService( const ::rtl::OUString& ServiceName ) throw (uno::RuntimeException)
    {
        return ServiceName.equalsAsciiL( RTL_CONSTASCII_STRINGPARAM ( SERVICE_NAME ) );
    }

    uno::Sequence< ::rtl::OUString > SAL_CALL ParametricPolyPolygon::getSupportedServiceNames(  ) throw (uno::RuntimeException)
    {
        uno::Sequence< ::rtl::OUString > aRet(1);
        aRet[0] = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM ( SERVICE_NAME ) );

        return aRet;
    }

    bool ParametricPolyPolygon::fill( SurfaceGraphicsSharedPtr&		rGraphics,
                                      const GraphicsPathSharedPtr&	rPath,
                                      const rendering::ViewState& 	viewState, 
                                      const rendering::RenderState&	renderState,
                                      const rendering::Texture& 	texture ) const
    {
        // TODO(T2): It is potentially dangerous to lock here, since
        // this method is called from the CanvasHelper with _locked_
        // canvas mutex!

        // currently, need to lock here because of potential
        // concurrent disposing().
        ::osl::MutexGuard aGuard( m_aMutex );

        Gdiplus::Matrix aMatrix;
        tools::gdiPlusMatrixFromAffineMatrix2D( aMatrix, 
                                                texture.AffineTransform );

        VectorOfBrushes::const_iterator aIter( maBrushes.begin() );
        VectorOfBrushes::const_iterator aEnd ( maBrushes.end()   );
        while( aIter != aEnd )
        {
            applyTextureParameters( aIter,
                                    aMatrix,
                                    texture );

            bool 					bNeedRestore( false );
            Gdiplus::GraphicsState 	aState;

            // if we also have a clipping for this gradient (only
            // needed for CLAMPed fillings), transform clip path with
            // texture transformation, and apply
            if( aIter->mpClipPath.get() &&
                texture.RepeatModeX == rendering::TexturingMode::CLAMP &&
                texture.RepeatModeY == rendering::TexturingMode::CLAMP )
            {
                GraphicsPathSharedPtr pLocalClipPath( aIter->mpClipPath->Clone() );
                pLocalClipPath->Transform( &aMatrix );
                
                aState = (*rGraphics)->Save();

                if( Gdiplus::Ok != (*rGraphics)->SetClip( pLocalClipPath.get(), 
                                                          Gdiplus::CombineModeIntersect ) )
                {
                    ENSURE_AND_RETURN( false,
                                       "ParametricPolyPolygon::fill(): could not apply clip" );
                }
            }

            (*rGraphics)->FillPath( aIter->mpBrush.get(), rPath.get() );

            if( bNeedRestore )
                (*rGraphics)->Restore( aState );

            ++aIter;
        }

#if defined(VERBOSE) && defined(DBG_UTIL)
        (*rGraphics)->MultiplyTransform( &aMatrix );
        
        Gdiplus::Pen aPen( Gdiplus::Color( 255, 255, 0, 0 ),
                           0.0001f );
        
        (*rGraphics)->DrawRectangle( &aPen,
                                     Gdiplus::RectF( 0.0f, 0.0f,
                                                     1.0f, 1.0f ) );
#endif

        return true;
    }

    ParametricPolyPolygon::~ParametricPolyPolygon()
    {
    }

    ParametricPolyPolygon::ParametricPolyPolygon( const VectorOfBrushes& rBrushes ) :
        ParametricPolyPolygon_Base( m_aMutex ),        
        maBrushes( rBrushes )
    {
    }

    template< typename BrushType > void ParametricPolyPolygon::applyTextureParams( BrushType*				 pBrush,
                                                                                   const Gdiplus::Matrix&	 rMatrix,
                                                                                   const rendering::Texture& rTexture ) const
    {
        if( pBrush )
        {
            pBrush->SetTransform( &rMatrix );

            const bool bTileX( rTexture.RepeatModeX != rendering::TexturingMode::CLAMP );
            const bool bTileY( rTexture.RepeatModeY != rendering::TexturingMode::CLAMP );

            if( bTileX && bTileY )
                pBrush->SetWrapMode( Gdiplus::WrapModeTile );
            else
            {
                OSL_ENSURE( bTileY == bTileX,
                            "CanvasHelper::applyTextureParams(): Cannot have repeat x and repeat y differ!" );

                pBrush->SetWrapMode( Gdiplus::WrapModeClamp );
            }
        }
    }

    void ParametricPolyPolygon::applyTextureParameters( const VectorOfBrushes::const_iterator&	rIter,
                                                        const Gdiplus::Matrix&	 				rMatrix,
                                                        const rendering::Texture& 				rTexture ) const
    {
        // Oh man. MS APIs simply suck big time. Why on earth is there
        // a Brush interface, but lots of shared methods _not_ in
        // Brush, but repeated over lots of Brush implementations
        // (SetTransform(), SetWrapMode() etc.)? Have to down-cast
        // according to type here, and apply the same (templated) code
        // to all of them. WTF.
        switch( rIter->mpBrush->GetType() )
        {
            case Gdiplus::BrushTypeSolidColor:
                // nothing to change here
                break;

            case Gdiplus::BrushTypeHatchFill:
                // nothing to change here
                break;

            case Gdiplus::BrushTypeTextureFill:
                applyTextureParams( static_cast<Gdiplus::TextureBrush*>( rIter->mpBrush.get() ),
                                    rMatrix,
                                    rTexture );
                break;

            case Gdiplus::BrushTypePathGradient:
                applyTextureParams( static_cast<Gdiplus::PathGradientBrush*>( rIter->mpBrush.get() ),
                                    rMatrix,
                                    rTexture );
                break;

            case Gdiplus::BrushTypeLinearGradient:
                applyTextureParams( static_cast<Gdiplus::LinearGradientBrush*>( rIter->mpBrush.get() ),
                                    rMatrix,
                                    rTexture );
                break;

            default:
                ENSURE_AND_THROW( false,
                                  "CanvasHelper::applyTextureParameters(): Unexpected brush type" );
                break;
        }
    }

}
