/*************************************************************************
 *
 *  $RCSfile: drawshape.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/26 18:51:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

// must be first
#include <canvas/debug.hxx>
#include <canvas/verbosetrace.hxx>
#include <drawshape.hxx>

#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif
#ifndef _COM_SUN_STAR_AWT_RECTANGLE_HPP_
#include <com/sun/star/awt/Rectangle.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif
#ifndef _COM_SUN_STAR_AWT_FONTWEIGHT_HPP_
#include <com/sun/star/awt/FontWeight.hpp>
#endif

#ifndef _SV_METAACT_HXX
#include <vcl/metaact.hxx>
#endif
#ifndef _SV_GDIMTF_HXX
#include <vcl/gdimtf.hxx>
#endif

#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif
#ifndef _BGFX_RANGE_RANGEEXPANDER_HXX
#include <basegfx/range/rangeexpander.hxx>
#endif

#ifndef INCLUDED_RTL_MATH_HXX
#include <rtl/math.hxx>
#endif

#ifndef BOOST_BIND_HPP_INCLUDED
#include <boost/bind.hpp>
#endif
#ifndef BOOST_MEM_FN_HPP_INCLUDED
#include <boost/mem_fn.hpp>
#endif

#include <canvas/canvastools.hxx>

#include <cmath> // for trigonometry and fabs
#include <algorithm>
#include <functional>
#include <limits>

#include <wakeupevent.hxx>
#include <intrinsicanimationactivity.hxx>
#include <slideshowexceptions.hxx>
#include <tools.hxx>
#include <lerp.hxx>
#include <gdimtftools.hxx>

using namespace ::drafts::com::sun::star;
using namespace ::com::sun::star;


namespace presentation
{
    namespace internal
    {

        //////////////////////////////////////////////////////////////////////
        //
        // Private methods
        //
        //////////////////////////////////////////////////////////////////////

        void DrawShape::endLineFormat( DocTreeNode& 	rCurrShapeFormatTree,
                                       DocTreeNode& 	rCurrLine,
                                       sal_Int32		nCurrIndex				) const
        {
            rCurrLine.setEndIndex( nCurrIndex );
            rCurrShapeFormatTree.appendChild( rCurrLine );
            
            // start fresh node
            rCurrLine = DocTreeNode( nCurrIndex, nCurrIndex, DocTreeNode::NODETYPE_FORMATTING_LINE );
        }

        void DrawShape::endShapeFormat( DocTreeNode& 	rCurrShapeFormatTree,
                                        DocTreeNode& 	rCurrLine,
                                        sal_Int32		nCurrIndex				) const
        {
            // append last line node, if not empty
            if( rCurrLine.getStartIndex() != nCurrIndex )
                endLineFormat( rCurrShapeFormatTree, rCurrLine, nCurrIndex );
                            
            rCurrShapeFormatTree.setEndIndex( nCurrIndex );
            maFormattedDocTree.push_back( rCurrShapeFormatTree );
        }

        void DrawShape::endCharCellLogic( DocTreeNode& 	rCurrWord,
                                          DocTreeNode& 	rCurrCharCell,
                                          sal_Int32		nCurrIndex		) const
        {
            rCurrCharCell.setEndIndex( nCurrIndex );
            rCurrWord.appendChild( rCurrCharCell );

            // start fresh node
            rCurrCharCell = DocTreeNode( nCurrIndex, nCurrIndex, DocTreeNode::NODETYPE_LOGICAL_CHARACTER_CELL );
        }

        void DrawShape::endWordLogic( DocTreeNode& 	rCurrSentence,
                                      DocTreeNode& 	rCurrWord,
                                      DocTreeNode& 	rCurrCharCell,
                                      sal_Int32		nCurrIndex		) const
        {
            // append last character node, if not empty
            if( rCurrCharCell.getStartIndex() != nCurrIndex )
                endCharCellLogic(rCurrWord,rCurrCharCell,nCurrIndex);
                                    
            rCurrWord.setEndIndex( nCurrIndex );
            rCurrSentence.appendChild( rCurrWord );

            // start fresh node
            rCurrWord = DocTreeNode( nCurrIndex, nCurrIndex, DocTreeNode::NODETYPE_LOGICAL_WORD );
        }

        void DrawShape::endSentenceLogic( DocTreeNode& 	rCurrPara,
                                          DocTreeNode& 	rCurrSentence,
                                          DocTreeNode& 	rCurrWord,
                                          DocTreeNode& 	rCurrCharCell,
                                          sal_Int32		nCurrIndex		) const
        {
            // append last word node, if not empty
            if( rCurrWord.getStartIndex() != nCurrIndex )
                endWordLogic(rCurrSentence,rCurrWord,rCurrCharCell,nCurrIndex);
                                
            rCurrSentence.setEndIndex( nCurrIndex );
            rCurrPara.appendChild( rCurrSentence );

            // start fresh node
            rCurrSentence = DocTreeNode( nCurrIndex, nCurrIndex, DocTreeNode::NODETYPE_LOGICAL_SENTENCE );
        }

        void DrawShape::endParaLogic( DocTreeNode& 	rCurrShapeLogicTree,
                                      DocTreeNode& 	rCurrPara,
                                      DocTreeNode& 	rCurrSentence,
                                      DocTreeNode& 	rCurrWord,
                                      DocTreeNode& 	rCurrCharCell,
                                      sal_Int32		nCurrIndex				) const
        {
            // append last sentence node, if not empty
            if( rCurrSentence.getStartIndex() != nCurrIndex )
                endSentenceLogic(rCurrPara,rCurrSentence,rCurrWord,rCurrCharCell,nCurrIndex);

            rCurrPara.setEndIndex( nCurrIndex );
            rCurrShapeLogicTree.appendChild( rCurrPara );
                                
            // start fresh node
            rCurrPara = DocTreeNode( nCurrIndex, nCurrIndex, DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH );
        }

        void DrawShape::endShapeLogic( DocTreeNode& rCurrShapeLogicTree,
                                       DocTreeNode& rCurrPara,
                                       DocTreeNode& rCurrSentence,
                                       DocTreeNode& rCurrWord,
                                       DocTreeNode& rCurrCharCell,
                                       sal_Int32	nCurrIndex				) const
        {
            // append last paragraph node, if not empty
            if( rCurrPara.getStartIndex() != nCurrIndex )
                endParaLogic( rCurrShapeLogicTree,rCurrPara,rCurrSentence,rCurrWord,rCurrCharCell,nCurrIndex );

            rCurrShapeLogicTree.setEndIndex( nCurrIndex );
            maLogicalDocTree.push_back( rCurrShapeLogicTree );
        }

        void DrawShape::initDocTrees() const
        {
            // no shape, no doc tree
            if( !mxShape.is() )
                return;

            if( !mbMtfHasComments )
            {
                // re-fetch metafile with comments
                mpCurrMtf = loadMtf( mxShape, mxPage, true );
                mbMtfHasComments = true;
            }

            // TODO(P1): Maybe do that lazy across the hierarchies. That is, if somebody 
            // requests only paragraph structure, don't generate trees down to characters.

            // already initialized?
            if( mbDocTreeInitialized )
                return; // yes, early exit

            // init output variables
            maFormattedDocTree.clear();
            maLogicalDocTree.clear();

            OSL_ENSURE( mpCurrMtf.get() != NULL,
                        "DrawShape::initTextInfo() invalid metafile" );

            if( mpCurrMtf.get() == NULL )
                return; // don't throw here, the doctree will simply
                        // remain empty on error

            // accumulators (here, we collect the various metafile comments into). 
            // Everytime a corresponding comment arrives, one of those variables will be 
            // pushed into the DocTree
            DocTreeNode aCurrShapeFormatTree(0,0,DocTreeNode::NODETYPE_FORMATTING_SHAPE);
            DocTreeNode aCurrLine(0,0,DocTreeNode::NODETYPE_FORMATTING_LINE);

            DocTreeNode aCurrShapeLogicTree(0,0,DocTreeNode::NODETYPE_LOGICAL_SHAPE);
            DocTreeNode aCurrPara(0,0,DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH);
            DocTreeNode aCurrSentence(0,0,DocTreeNode::NODETYPE_LOGICAL_SENTENCE);
            DocTreeNode aCurrWord(0,0,DocTreeNode::NODETYPE_LOGICAL_WORD);
            DocTreeNode aCurrCharCell(0,0,DocTreeNode::NODETYPE_LOGICAL_CHARACTER_CELL);

            // search metafile for text output
            MetaAction* pCurrAct;

            sal_Int32 nActionIndex;
            for( nActionIndex=0, pCurrAct = mpCurrMtf->FirstAction(); 
                 pCurrAct; 
                 ++nActionIndex, pCurrAct = mpCurrMtf->NextAction() )
            {
                // check for one of our special text doctree comments
                if( pCurrAct->GetType() == META_COMMENT_ACTION )
                {
                    MetaCommentAction* pAct = static_cast<MetaCommentAction*>(pCurrAct);

                    // early exit if no special XTEXT comment
                    if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT", 5 ) == COMPARE_EQUAL )
                    {
                        if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_PAINTSHAPE_BEGIN" ) == COMPARE_EQUAL )
                        {
                            // initialize start indices
                            aCurrShapeFormatTree.setStartIndex( nActionIndex );
                            aCurrShapeLogicTree.setStartIndex( nActionIndex );

                            aCurrLine.setStartIndex( nActionIndex );
                            aCurrPara.setStartIndex( nActionIndex );
                            aCurrSentence.setStartIndex( nActionIndex );
                            aCurrWord.setStartIndex( nActionIndex );
                            aCurrCharCell.setStartIndex( nActionIndex );
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_PAINTSHAPE_END" ) == COMPARE_EQUAL )
                        {
                            // this comment implicitely closes all embedded open entities

                            // Formatting tree
                            // ===============

                            endShapeFormat( aCurrShapeFormatTree, aCurrLine, nActionIndex );


                            // Logical tree
                            // ============
                            
                            endShapeLogic(aCurrShapeLogicTree,
                                          aCurrPara,
                                          aCurrSentence,
                                          aCurrWord,
                                          aCurrCharCell,
                                          nActionIndex);
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_EOP" ) == COMPARE_EQUAL )
                        {
                            // fill out paragraph tree node
                            // ============================

                            endParaLogic( aCurrShapeLogicTree,
                                          aCurrPara,
                                          aCurrSentence,
                                          aCurrWord,
                                          aCurrCharCell,
                                          nActionIndex );

                            // fill out line tree node
                            // =======================

                            // a paragraph also ends a line!
                            endLineFormat( aCurrShapeFormatTree,
                                           aCurrLine,
                                           nActionIndex );
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_EOL" ) == COMPARE_EQUAL )
                        {
                            endLineFormat( aCurrShapeFormatTree,
                                           aCurrLine,
                                           nActionIndex );
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_EOS" ) == COMPARE_EQUAL )
                        {
                            endSentenceLogic( aCurrPara,
                                              aCurrSentence,
                                              aCurrWord,
                                              aCurrCharCell,
                                              nActionIndex );
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_EOW" ) == COMPARE_EQUAL )
                        {
                            endWordLogic( aCurrSentence,
                                          aCurrWord,
                                          aCurrCharCell,
                                          nActionIndex	);
                        }
                        else if( pAct->GetComment().CompareIgnoreCaseToAscii( "XTEXT_EOC" ) == COMPARE_EQUAL )
                        {
                            endCharCellLogic( aCurrWord,
                                              aCurrCharCell,
                                              nActionIndex	);
                        }
                    }
                }                
            }

            mbDocTreeInitialized = true;
            mnNumDocTreeItems = nActionIndex;
        }

        GDIMetaFileSharedPtr DrawShape::loadMtf( const uno::Reference< drawing::XShape >&		xShape, 
                                                 const uno::Reference< drawing::XDrawPage >&	xContainingPage,
                                                 bool 											bWithComments ) const
        {
            GDIMetaFileSharedPtr pMtf( new GDIMetaFile() );

            // Fetch metafile
            getMetaFile( xShape, xContainingPage, *pMtf, bWithComments );
            
            return pMtf;
        }

        void DrawShape::updateStateIds() const
        {
            // update the states, we've just redrawn
            if( mpAttributeLayer.get() )
            {
                mnAttributeTransformationState = mpAttributeLayer->getTransformationState();
                mnAttributeClipState = mpAttributeLayer->getClipState();
                mnAttributeAlphaState = mpAttributeLayer->getAlphaState();
                mnAttributePositionState = mpAttributeLayer->getPositionState();
                mnAttributeContentState = mpAttributeLayer->getContentState();
            }
        }

        void DrawShape::updateSubsetBounds( const SubsetEntry& rSubsetEntry )
        {
            // TODO(F1): This removes too much from non-contiguous subsets
            mnMinSubsetActionIndex = ::std::min(
                mnMinSubsetActionIndex,
                rSubsetEntry.mnStartActionIndex );
            mnMaxSubsetActionIndex = ::std::max(
                mnMaxSubsetActionIndex,
                rSubsetEntry.mnEndActionIndex );
        }

        void DrawShape::updateSubsets()
        {
            maCurrentSubsets.clear();

            if( !maSubsetShapes.empty() )
            {
                if( maSubset.isEmpty() )
                {
                    // non-subsetted node, with some child subsets
                    // that subtract from it
                    maCurrentSubsets.push_back( DocTreeNode( 0,
                                                             mnMinSubsetActionIndex,
                                                             DocTreeNode::NODETYPE_INVALID ) );
                    maCurrentSubsets.push_back( DocTreeNode( mnMaxSubsetActionIndex,
                                                             mnNumDocTreeItems,
                                                             DocTreeNode::NODETYPE_INVALID ) );
                }
                else
                {
                    // subsetted node, from which some further child
                    // subsets subtract content
                    maCurrentSubsets.push_back( DocTreeNode( maSubset.getStartIndex(),
                                                             mnMinSubsetActionIndex,
                                                             DocTreeNode::NODETYPE_INVALID ) );
                    maCurrentSubsets.push_back( DocTreeNode( mnMaxSubsetActionIndex,
                                                             maSubset.getEndIndex(),
                                                             DocTreeNode::NODETYPE_INVALID ) );
                }
            }
            else
            {
                // no further child subsets, simply add our subset (if any)
                if( !maSubset.isEmpty() )
                {
                    // subsetted node, without any subset children
                    maCurrentSubsets.push_back( maSubset );
                }
            }
        }

        namespace
        {
            ::basegfx::B2DRectangle getShapeBounds( const uno::Reference< drawing::XShape >& xShape )
            {
                uno::Reference< beans::XPropertySet > xPropSet( xShape, 
                                                                uno::UNO_QUERY_THROW );
                // read bound rect
                awt::Rectangle aTmpRect;
                if( !(xPropSet->getPropertyValue( 
                          ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("BoundRect") ) ) >>= aTmpRect) )
                {
                    ENSURE_AND_THROW( false,
                                      "getShapeBounds(): Could not get \"BoundRect\" property from shape" );
                }

                return ::basegfx::B2DRectangle( aTmpRect.X, 
                                                aTmpRect.Y,
                                                aTmpRect.X+aTmpRect.Width, 
                                                aTmpRect.Y+aTmpRect.Height );
            }

            double getShapePrio( const uno::Reference< drawing::XShape >& xShape )
            {
                uno::Reference< beans::XPropertySet > xPropSet( xShape, 
                                                                uno::UNO_QUERY_THROW );
                // read prio
                sal_Int32 nPrio;
                if( !(xPropSet->getPropertyValue( 
                          ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("ZOrder") ) ) >>= nPrio) )
                {
                    ENSURE_AND_THROW( false,
                                      "getShapePrio(): Could not get \"ZOrder\" property from shape" );
                }

                // TODO(F2): Check and adapt the range of possible values here. 
                // Maybe we can also take the total number of shapes here
                return nPrio / 65535.0;
            }
        }

        bool DrawShape::implRender( int nUpdateFlags ) const
        {
            RTL_LOGFILE_CONTEXT( aLog, "::presentation::internal::DrawShape::implRender()" );
            RTL_LOGFILE_CONTEXT_TRACE1( aLog, "::presentation::internal::DrawShape: 0x%X", this );

            ENSURE_AND_RETURN( !maViewShapes.empty(),
                               "DrawShape::implRender(): render called on DrawShape without views" );

            // gcc again...
            const ::basegfx::B2DRectangle& rCurrBounds( DrawShape::getPosSize() );

            if( rCurrBounds.getRange().equalZero() )
            {
                // zero-sized shapes are effectively invisible,
                // thus, we save us the rendering...
                return true;
            }

            // redraw all view shapes, by calling their update() method
            if( ::std::count_if( maViewShapes.begin(),
                                 maViewShapes.end(),
                                 ::boost::bind<bool>(
                                     ::boost::mem_fn( &ViewShape::update ), // though _theoretically_,
                                     										// bind should eat this even
                                     										// with _1 being a shared_ptr,
                                     										// it does _not_ for MSVC without
                                     										// the extra mem_fn. WTF.
                                     _1,
                                     ::boost::cref( mpCurrMtf ),
                                     ::boost::cref(
                                         ViewShape::RenderArgs(
                                             maBounds,
                                             rCurrBounds,
                                             mpAttributeLayer,
                                             maCurrentSubsets ) ), 
                                     nUpdateFlags,
                                     isVisible() ) ) 
                != static_cast<ViewShapeVector::difference_type>(maViewShapes.size()) )
            {
                // at least one of the ViewShape::update() calls did return 
                // false - update failed on at least one ViewLayer
                return false;
            }

            // successfully redrawn - update state IDs to detect next changes
            updateStateIds();

            return true;
        }

        int DrawShape::getUpdateFlags() const
        {
            // default: update nothing, unless ShapeAttributeStack tells us below
            int nUpdateFlags(ViewShape::NONE);

            // determine what has to be updated
            // --------------------------------

            // do we have an attribute layer?
            if( mpAttributeLayer.get() )
            {
                // TODO(P1): This can be done without conditional branching. 
                // See HAKMEM.
                if( mpAttributeLayer->getPositionState() != mnAttributePositionState )
                {
                    nUpdateFlags |= ViewShape::POSITION;
                }
                if( mpAttributeLayer->getAlphaState() != mnAttributeAlphaState )
                {
                    nUpdateFlags |= ViewShape::ALPHA;
                }
                if( mpAttributeLayer->getClipState() != mnAttributeClipState )
                {
                    nUpdateFlags |= ViewShape::CLIP;
                }
                if( mpAttributeLayer->getTransformationState() != mnAttributeTransformationState )
                {                
                    nUpdateFlags |= ViewShape::TRANSFORMATION;
                }
                if( mpAttributeLayer->getContentState() != mnAttributeContentState )
                {                
                    nUpdateFlags |= ViewShape::CONTENT;
                }
            }

            return nUpdateFlags;
        }


        //////////////////////////////////////////////////////////////////////
        //
        // Public methods
        //
        //////////////////////////////////////////////////////////////////////

        DrawShape::DrawShape( const uno::Reference< drawing::XShape >& 		xShape,
                              const uno::Reference< drawing::XDrawPage >&	xContainingPage,
                              double										nPrio ) :
            mxShape( xShape ),
            mxPage( xContainingPage ),
            maAnimationFrames(), // empty, we don't have no intrinsic animation
            mnCurrFrame(0),
            mpIntrinsicAnimationActivity(),
            mpCurrMtf( loadMtf( xShape, xContainingPage, false ) ),
            mnPriority( nPrio ), // TODO(F1): When ZOrder someday becomes usable: make this ( getShapePrio( xShape ) ),
            maBounds( getShapeBounds( xShape ) ),
            mpAttributeLayer(),
            mnAttributeTransformationState(0),
            mnAttributeClipState(0),
            mnAttributeAlphaState(0),
            mnAttributePositionState(0),
            mnAttributeContentState(0),
            maViewShapes(),
            maSubsetShapes(),
            mnMinSubsetActionIndex( SAL_MAX_INT32 ),
            mnMaxSubsetActionIndex(0),
            mnIsAnimatedCount(0),
            maFormattedDocTree(),
            maLogicalDocTree(),
            mnNumDocTreeItems(0),
            maSubset(),
            maCurrentSubsets(),
            mbIsVisible( true ),
            mbDocTreeInitialized( false ),
            mbMtfHasComments( false ),
            mbForceUpdate( false )
        {
            ENSURE_AND_THROW( mxShape.is(), "DrawShape::DrawShape(): Invalid XShape" );
            ENSURE_AND_THROW( mxPage.is(), "DrawShape::DrawShape(): Invalid containing page" );
            ENSURE_AND_THROW( mpCurrMtf.get(), "DrawShape::DrawShape(): Invalid metafile" );
        }

        DrawShape::DrawShape( const uno::Reference< drawing::XShape >& 		xShape,
                              const uno::Reference< drawing::XDrawPage >&	xContainingPage,
                              double										nPrio,
                              const Graphic&								rGraphic ) :
            mxShape( xShape ),
            mxPage( xContainingPage ),
            maAnimationFrames(),
            mnCurrFrame(0),
            mpIntrinsicAnimationActivity(),
            mpCurrMtf(),
            mnPriority( nPrio ), // TODO(F1): When ZOrder someday becomes usable: make this ( getShapePrio( xShape ) ),
            maBounds( getShapeBounds( xShape ) ),
            mpAttributeLayer(),
            mnAttributeTransformationState(0),
            mnAttributeClipState(0),
            mnAttributeAlphaState(0),
            mnAttributePositionState(0),
            mnAttributeContentState(0),
            maViewShapes(),
            maSubsetShapes(),
            mnMinSubsetActionIndex( SAL_MAX_INT32 ),
            mnMaxSubsetActionIndex(0),
            mnIsAnimatedCount(0),
            maFormattedDocTree(),
            maLogicalDocTree(),
            mnNumDocTreeItems(0),
            maSubset(),
            maCurrentSubsets(),
            mbIsVisible( true ),
            mbDocTreeInitialized( false ),
            mbMtfHasComments( false ),
            mbForceUpdate( false )
        {
            ENSURE_AND_THROW( rGraphic.IsAnimated(),
                              "DrawShape::DrawShape(): Graphic is no animation" );

            getAnimationFromGraphic( maAnimationFrames,
                                     rGraphic );

            ENSURE_AND_THROW( !maAnimationFrames.empty() &&
                              maAnimationFrames.front().mpMtf.get(),
                              "DrawShape::DrawShape(): " );
            mpCurrMtf = maAnimationFrames.front().mpMtf;

            ENSURE_AND_THROW( mxShape.is(), "DrawShape::DrawShape(): Invalid XShape" );
            ENSURE_AND_THROW( mxPage.is(), "DrawShape::DrawShape(): Invalid containing page" );
            ENSURE_AND_THROW( mpCurrMtf.get(), "DrawShape::DrawShape(): Invalid metafile" );
        }

        DrawShape::DrawShape( const DrawShape& 		rSrc, 
                              const DocTreeNode& 	rTreeNode,
                              double				nPrio ) :
            mxShape( rSrc.mxShape ),
            mxPage( rSrc.mxPage ),
            maAnimationFrames(), // don't copy animations for subsets,
                                 // only the current frame!
            mnCurrFrame(0),
            mpIntrinsicAnimationActivity(),
            mpCurrMtf( rSrc.mpCurrMtf ),
            mnPriority( nPrio ),
            maBounds(), // leave bounds empty for subset shapes. this
                        // member will be initialized with the first
                        // addViewLayer() call
            mpAttributeLayer(),
            mnAttributeTransformationState(0),
            mnAttributeClipState(0),
            mnAttributeAlphaState(0),
            mnAttributePositionState(0),
            mnAttributeContentState(0),
            maViewShapes(),
            maSubsetShapes(),
            mnMinSubsetActionIndex( SAL_MAX_INT32 ),
            mnMaxSubsetActionIndex(0),
            mnIsAnimatedCount(0),
            maFormattedDocTree(),
            maLogicalDocTree(),
            mnNumDocTreeItems(0),
            maSubset( rTreeNode ),
            maCurrentSubsets( 1, rTreeNode ),
            mbIsVisible( rSrc.mbIsVisible ),
            mbDocTreeInitialized( false ),
            mbMtfHasComments( rSrc.mbMtfHasComments ),
            mbForceUpdate( false )
        {
            ENSURE_AND_THROW( mxShape.is(), "DrawShape::DrawShape(): Invalid XShape" );
            ENSURE_AND_THROW( mpCurrMtf.get(), "DrawShape::DrawShape(): Invalid metafile" );
        }

        DrawShape::~DrawShape()
        {
        }

        uno::Reference< drawing::XShape > DrawShape::getXShape() const
        {
            return mxShape;
        }

        void DrawShape::addViewLayer( const ViewLayerSharedPtr& rNewLayer,
                                      bool						bRedrawLayer )
        {
            ViewShapeVector::iterator aEnd( maViewShapes.end() );

            // already added?
            if( ::std::find_if( maViewShapes.begin(), 
                                aEnd, 
                                ::boost::bind<bool>( 
                                    ::std::equal_to< ViewLayerSharedPtr >(),
                                    ::boost::bind( &ViewShape::getViewLayer,
                                                   _1 ),
                                    ::boost::cref( rNewLayer ) ) ) != aEnd )
            {
                // yes, nothing to do
                return;
            }

            maViewShapes.push_back( 
                ViewShapeSharedPtr( 
                    new ViewShape( rNewLayer ) ) );

            // render the Shape on the newly added ViewLayer
            if( bRedrawLayer )
            {
                maViewShapes.back()->update( mpCurrMtf, 
                                             ViewShape::RenderArgs( 
                                                 maBounds,
                                                 DrawShape::getPosSize(), 
                                                 mpAttributeLayer, 
                                                 maCurrentSubsets ),
                                             ViewShape::FORCE, 
                                             isVisible() );
            }

            // retrieve actual shape bounds, if this is a subset shape
            if( !maSubset.isEmpty() )
            {
                // TODO(P1): Might pay off to also re-calc bounds of
                // master shape (that from which the subsets are
                // taken)
                ::basegfx::B2DRectangle aTotalBounds;

                // for each viewshape, call getActualBounds() and
                // merge result with aTotalBounds
                ::std::for_each( maViewShapes.begin(),
                                 maViewShapes.end(),
                                 ::boost::bind( ::basegfx::B2DRangeExpander( aTotalBounds ),
                                                ::boost::bind(
                                                    &ViewShape::getActualBounds,
                                                    _1,
                                                    ::boost::cref(mpCurrMtf), 
                                                    ::boost::cref(mpAttributeLayer),
                                                    ::boost::cref(maCurrentSubsets) ) ) );

                static ::basegfx::B2DRectangle aStandardBounds(0.0,0.0,1.0,1.0);

                if( aTotalBounds.isEmpty() )
                {
                    // Optimization: if the bounds are empty,
                    // i.e. this shape itself is empty, we remove all
                    // viewshapes (and render nothing, effectively)
                    maViewShapes.clear();
                    
                    // TODO(F3): Hack. Maybe DrawShape should have a dispose,
                    // too. Because here, we try to break the circular
                    // reference via IntrinsicAnimationActivity, when the last
                    // view is removed.
                    if( maViewShapes.empty() )
                        mpIntrinsicAnimationActivity.reset();

                    // set shape bounds to empty
                    maBounds = aTotalBounds;
                }
                else if( aTotalBounds == aStandardBounds )
                {
                    maBounds = getShapeBounds( mxShape );
                }
                else
                {
                    // update shape bounds: aTotalBounds contains relative
                    // positions in a one-by-one rectangle, i.e. denoting
                    // relative positions in the original shape rectangle,
                    // returned by getShapeBounds()
                    maBounds = getShapeBounds( mxShape );
                    maBounds = ::basegfx::B2DRectangle(
                        lerp( maBounds.getMinX(),
                              maBounds.getMaxX(),
                              aTotalBounds.getMinX() ),
                        lerp( maBounds.getMinY(),
                              maBounds.getMaxY(),
                              aTotalBounds.getMinY() ),
                        lerp( maBounds.getMinX(),
                              maBounds.getMaxX(),
                              aTotalBounds.getMaxX() ),
                        lerp( maBounds.getMinY(),
                              maBounds.getMaxY(),
                              aTotalBounds.getMaxY() ) );
                }
            }
        }

        bool DrawShape::removeViewLayer( const ViewLayerSharedPtr& rLayer )
        {
            const ViewShapeVector::iterator aEnd( maViewShapes.end() );

            OSL_ENSURE( ::std::count_if(maViewShapes.begin(), 
                                        aEnd, 
                                        ::boost::bind<bool>( 
                                            ::std::equal_to< ViewLayerSharedPtr >(),
                                            ::boost::bind( &ViewShape::getViewLayer,
                                                           _1 ),
                                            ::boost::cref( rLayer ) ) ) < 2,
                        "DrawShape::removeViewLayer(): Duplicate ViewLayer entries!" );

            ViewShapeVector::iterator aIter;

            if( (aIter=::std::remove_if( maViewShapes.begin(), 
                                         aEnd, 
                                         ::boost::bind<bool>( 
                                             ::std::equal_to< ViewLayerSharedPtr >(),
                                             ::boost::bind( &ViewShape::getViewLayer,
                                                            _1 ),
                                             ::boost::cref( rLayer ) ) )) == aEnd )
            {
                // view layer seemingly was not added, failed
                return false;
            }

            // actually erase from container
            maViewShapes.erase( aIter, aEnd );

            // TODO(F3): Hack. Maybe DrawShape should have a dispose,
            // too. Because here, we try to break the circular
            // reference via IntrinsicAnimationActivity, when the last
            // view is removed.
            if( maViewShapes.empty() )
                mpIntrinsicAnimationActivity.reset();

            return true;
        }

        bool DrawShape::update() const
        {
            if( mbForceUpdate )
            {
                return render();
            }
            else
            {
                return implRender( getUpdateFlags() );
            }
        }

        bool DrawShape::render() const
        {
            // will perform the update now
            mbForceUpdate = false;

            // force redraw. Have to also pass on the update flags,
            // because e.g. content update (regeneration of the
            // metafile renderer) is normally not performed. A simple
            // ViewShape::FORCE would only paint the metafile in its
            // old state.
            return implRender( ViewShape::FORCE | getUpdateFlags() );
        }

        bool DrawShape::isUpdateNecessary() const
        {
            return mbForceUpdate ? 
                true : 
                getUpdateFlags() != ViewShape::NONE;
        }


        ::basegfx::B2DRectangle DrawShape::getPosSize() const
        {
            // little optimization: for non-modified shapes, we don't
            // create an ShapeAttributeStack, and therefore also don't
            // have to check it.

            // an already empty shape bound need no further
            // treatment. In fact, any changes applied below would
            // actually remove the special empty state, thus, don't
            // change!
            if( !mpAttributeLayer.get() ||
                maBounds.isEmpty() )
            {
                return maBounds;
            }
            else
            {
                // cannot use maBounds anymore, attributes might have been 
                // changed by now.
                // Have to use absolute values here, as negative sizes
                // (aka mirrored shapes) _still_ have the same bounds,
                // only with mirrored content.
                ::basegfx::B2DSize aSize;
                aSize.setX( fabs( mpAttributeLayer->isWidthValid() ?
                                  mpAttributeLayer->getWidth() :
                                  maBounds.getWidth() ) );
                aSize.setY( fabs( mpAttributeLayer->isHeightValid() ?
                                  mpAttributeLayer->getHeight() :
                                  maBounds.getHeight() ) );

                ::basegfx::B2DPoint aPos;
                aPos.setX( mpAttributeLayer->isPosXValid() ?
                           mpAttributeLayer->getPosX() :
                           maBounds.getCenterX() );
                aPos.setY( mpAttributeLayer->isPosYValid() ?
                           mpAttributeLayer->getPosY() :
                           maBounds.getCenterY() );

                return ::basegfx::B2DRectangle( aPos - 0.5*aSize, 
                                                aPos + 0.5*aSize );
            }
        }

        namespace
        {
            /** Functor expanding AA border for each passed ViewShape

            	Could not use ::boost::bind here, since
            	B2DRange::expand is overloaded.
             */
            class Expander
            {
            public: 
                Expander( ::basegfx::B2DSize& rBounds ) :
                    mrBounds( rBounds )
                {
                }

                void operator()( const ViewShapeSharedPtr& rShape ) const
                {
                    const ::basegfx::B2DSize& rShapeBorder( rShape->getAntialiasingBorder() );

                    mrBounds.setX( 
                        ::std::max(
                            rShapeBorder.getX(),
                            mrBounds.getX() ) );
                    mrBounds.setY( 
                        ::std::max(
                            rShapeBorder.getY(),
                            mrBounds.getY() ) );
                }

            private:
                ::basegfx::B2DSize& mrBounds;
            };
        }

        ::basegfx::B2DRectangle DrawShape::getUpdateArea() const
        {
            OSL_ENSURE( !maViewShapes.empty(),
                        "DrawShape::getUpdateArea(): called on DrawShape without views" );

            ::basegfx::B2DRectangle aBounds( DrawShape::getPosSize() );

            // an already empty shape bound need no further
            // treatment. In fact, any changes applied below would
            // actually remove the special empty state, thus, don't
            // change!
            if( !aBounds.isEmpty() )
            {
                // determine border needed for antialiasing the shape
                ::basegfx::B2DSize aAABorder(0.0,0.0);

                // for every view, get AA border and 'expand' aAABorder
                // appropriately.
                ::std::for_each( maViewShapes.begin(),
                                 maViewShapes.end(),
                                 Expander( aAABorder ) );

                // add calculated border to aBounds
                aBounds = ::basegfx::B2DRectangle( aBounds.getMinX() - aAABorder.getX(),
                                                   aBounds.getMinY() - aAABorder.getY(),
                                                   aBounds.getMaxX() + aAABorder.getX(),
                                                   aBounds.getMaxY() + aAABorder.getY() );

                if( mpAttributeLayer.get() )
                {
                    aBounds = getShapeUpdateArea( aBounds,
                                                  getShapeTransformation( maBounds,
                                                                          aBounds,
                                                                          mpAttributeLayer,
                                                                          true ),
                                                  mpAttributeLayer );
                }
            }

            return aBounds;
        }

        bool DrawShape::isVisible() const
        {
            bool bIsVisible( mbIsVisible );

            if( mpAttributeLayer.get() )
            {
                // check whether visibility and alpha are default (we
                // take mbIsVisible then), or visibility is on and
                // alpha is not 0.0 (fully transparent)
                if( mpAttributeLayer->isVisibilityValid() )
                    bIsVisible = mpAttributeLayer->getVisibility();

                // only touch bIsVisible, if the shape is still
                // visible - if getVisibility already made us
                // invisible, no alpha value will make us appear again.
                if( bIsVisible && mpAttributeLayer->isAlphaValid() )
                    bIsVisible = !::basegfx::fTools::equalZero( mpAttributeLayer->getAlpha() );
            }

            return bIsVisible;
        }

        double DrawShape::getPriority() const
        {
            return mnPriority;
        }

        bool DrawShape::isBackgroundDetached() const
        {
            return mnIsAnimatedCount > 0;
        }

        bool DrawShape::hasIntrinsicAnimation() const
        {
            return !maAnimationFrames.empty();
        }

        bool DrawShape::startIntrinsicAnimation( const SlideShowContext& rContext,
                                                 const ShapeSharedPtr&	 rShape )
        {
            if( !hasIntrinsicAnimation() )
                return false;

            ::std::vector<double> aTimeout;
            ::std::transform( maAnimationFrames.begin(),
                              maAnimationFrames.end(),
                              ::std::back_insert_iterator< ::std::vector<double> >( aTimeout ),
                              ::boost::mem_fn( &MtfAnimationFrame::getDuration ) );

            WakeupEventSharedPtr pWakeupEvent( new WakeupEvent(rContext.mrActivitiesQueue) );

            mpIntrinsicAnimationActivity.reset( 
                new IntrinsicAnimationActivity(rContext,
                                               ::boost::dynamic_pointer_cast< DrawShape >(rShape),
                                               pWakeupEvent,
                                               aTimeout) );

            pWakeupEvent->setActivity( mpIntrinsicAnimationActivity );

            return rContext.mrActivitiesQueue.addActivity( mpIntrinsicAnimationActivity );
        }

        bool DrawShape::endIntrinsicAnimation()
        {
            if( !mpIntrinsicAnimationActivity.get() )
                return false;

            mpIntrinsicAnimationActivity->end();
            mpIntrinsicAnimationActivity.reset();

            return true;
        }

        bool DrawShape::setIntrinsicAnimationFrame( ::std::size_t nCurrFrame )
        {
            ENSURE_AND_RETURN( nCurrFrame >= 0 &&
                               nCurrFrame < maAnimationFrames.size(),
                               "DrawShape::setIntrinsicAnimationFrame(): frame index out of bounds" );

            mnCurrFrame = nCurrFrame;
            mpCurrMtf   = maAnimationFrames[ mnCurrFrame ].mpMtf;

            return true;
        }


        // AnimatableShape methods
        // ======================================================

        void DrawShape::enterAnimationMode()
        {
            OSL_ENSURE( !maViewShapes.empty(),
                        "DrawShape::enterAnimationMode(): called on DrawShape without views" );

            if( mnIsAnimatedCount == 0 )
            {
                // notify all ViewShapes, by calling their enterAnimationMode method.
                // We're now entering animation mode
                ::std::for_each( maViewShapes.begin(),
                                 maViewShapes.end(),
                                 ::boost::mem_fn( &ViewShape::enterAnimationMode ) );
            }

            ++mnIsAnimatedCount;
        }

        void DrawShape::leaveAnimationMode()
        {
            OSL_ENSURE( !maViewShapes.empty(),
                        "DrawShape::leaveAnimationMode(): called on DrawShape without views" );

            --mnIsAnimatedCount;

            if( mnIsAnimatedCount == 0 )
            {
                // notify all ViewShapes, by calling their leaveAnimationMode method.
                // we're now leaving animation mode
                ::std::for_each( maViewShapes.begin(),
                                 maViewShapes.end(),
                                 ::boost::mem_fn( &ViewShape::leaveAnimationMode ) );
            }
        }


        // AttributableShape methods
        // ======================================================

        ShapeAttributeLayerSharedPtr DrawShape::createAttributeLayer()
        {
            // create new layer, with last as its new child
            mpAttributeLayer.reset( new ShapeAttributeLayer( mpAttributeLayer ) );

            return mpAttributeLayer;
        }

        bool DrawShape::revokeAttributeLayer( const ShapeAttributeLayerSharedPtr& rLayer )
        {
            if( !mpAttributeLayer.get() )
                return false; // no layers

            if( mpAttributeLayer == rLayer )
            {
                // it's the toplevel layer
                mpAttributeLayer = mpAttributeLayer->getChildLayer();
                
                // force redraw, all state variables have possibly 
                // changed
                mbForceUpdate = true;

                return true;
            }
            else
            {
                // pass on to the layer, to try the children
                return mpAttributeLayer->revokeChildLayer( rLayer );
            }
        }

        void DrawShape::setVisibility( bool bVisible )
        {
            if( mbIsVisible != bVisible )
            {
                mbIsVisible = bVisible;
                mbForceUpdate = true;
            }
        }

        DocTreeNode DrawShape::getEffectiveSubset() const
        {
            return maSubset;
        }

        AttributableShapeSharedPtr DrawShape::querySubset( const DocTreeNode& rTreeNode ) const
        {
            RTL_LOGFILE_CONTEXT( aLog, "::presentation::internal::DrawShape::querySubset()" );

            // subset shape already created for this DocTreeNode?
            SubsetEntry aEntry;
            
            aEntry.mnStartActionIndex 	= rTreeNode.getStartIndex();
            aEntry.mnEndActionIndex 	= rTreeNode.getEndIndex();

            ShapeSet::iterator aIter;
            if( (aIter=maSubsetShapes.find( aEntry )) != maSubsetShapes.end() )
            {
                // already created, return found entry
                return aIter->mpShape;
            }

            return AttributableShapeSharedPtr();
        }

        bool DrawShape::createSubset( AttributableShapeSharedPtr& 	o_rSubset,
                                      const DocTreeNode& 			rTreeNode )
        {
            RTL_LOGFILE_CONTEXT( aLog, "::presentation::internal::DrawShape::createSubset()" );

            // subset shape already created for this DocTreeNode?
            SubsetEntry aEntry;
            
            aEntry.mnStartActionIndex 	= rTreeNode.getStartIndex();
            aEntry.mnEndActionIndex 	= rTreeNode.getEndIndex();

            ShapeSet::iterator aIter;
            if( (aIter=maSubsetShapes.find( aEntry )) != maSubsetShapes.end() )
            {
                // already created, increment use count and return

                // safe cast, since set order does not depend on
                // mnSubsetQueriedCount
                const_cast<SubsetEntry&>(*aIter).mnSubsetQueriedCount++;

                o_rSubset = aIter->mpShape;

                return false; // reusing existing subset
            }
            else
            {
                // not yet created, init entry
                aEntry.mnSubsetQueriedCount = 1;
                aEntry.mpShape.reset( new DrawShape( *this, 
                                                     rTreeNode, 
                                                     aEntry.getHashValue() ) );
                
                maSubsetShapes.insert( aEntry );
                
                // update cached subset borders
                updateSubsetBounds( aEntry );
                updateSubsets();

                o_rSubset = aEntry.mpShape;

                return true; // subset newly created
            }
        }

        bool DrawShape::revokeSubset( const AttributableShapeSharedPtr& rShape )
        {
            RTL_LOGFILE_CONTEXT( aLog, "::presentation::internal::DrawShape::revokeSubset()" );

            // lookup subset shape
            SubsetEntry aEntry;
            const DocTreeNode& rEffectiveSubset( rShape->getEffectiveSubset() );
            
            aEntry.mnStartActionIndex 	= rEffectiveSubset.getStartIndex();
            aEntry.mnEndActionIndex 	= rEffectiveSubset.getEndIndex();

            ShapeSet::iterator aIter;
            if( (aIter=maSubsetShapes.find( aEntry )) == maSubsetShapes.end() )
                return false; // not found, subset was never queried

            // last client of the subset revoking?
            if( aIter->mnSubsetQueriedCount > 1 )
            {
                // no, still clients out there. Just decrement use count
                // safe cast, since order does not depend on mnSubsetQueriedCount
                const_cast<SubsetEntry&>(*aIter).mnSubsetQueriedCount--;

                VERBOSE_TRACE( "Subset summary: shape 0x%X, %d open subsets, revoked subset has refcount %d",
                               this,
                               maSubsetShapes.size(),
                               aIter->mnSubsetQueriedCount );

                return false; // not the last client
            }

            VERBOSE_TRACE( "Subset summary: shape 0x%X, %d open subsets, cleared subset has range [%d,%d]",
                           this,
                           maSubsetShapes.size(),
                           aEntry.mnStartActionIndex,
                           aEntry.mnEndActionIndex );

            // yes, remove from set
            maSubsetShapes.erase( aIter );

            // init bounds
            mnMinSubsetActionIndex = SAL_MAX_INT32;
            mnMaxSubsetActionIndex = 0;

            // if still other subset shapes registered, update min/max indices
            if( !maSubsetShapes.empty() )
            {
                // TODO(P2): This is quite expensive, when 
                // after every subset effect end, we have to scan
                // the whole shape set

                // determine new subset range
                ::std::for_each( maSubsetShapes.begin(),
                                 maSubsetShapes.end(),
                                 ::boost::bind(&DrawShape::updateSubsetBounds,
                                               this,
                                               _1 ) );

                updateSubsets();
            }

            // force redraw, content has possibly changed
            mbForceUpdate = true;

            return true;
        }

        sal_Int32 DrawShape::getNumberOfTreeNodes() const
        {
            initDocTrees();

            ENSURE_AND_THROW( maFormattedDocTree.size() == maLogicalDocTree.size(),
                              "DrawShape::getNumberOfTreeNodes(): Inconsistent tree sizes" );

            return maLogicalDocTree.size();
        }

        DocTreeNode DrawShape::getFormattingDocTreeNode( sal_Int32 nIndex ) const
        {
            initDocTrees();

            ENSURE_AND_THROW( nIndex >= 0 && static_cast<sal_Int32>(maFormattedDocTree.size()) > nIndex,
                              "DrawShape::getFormattingDocTreeNode(): Invalid index" );

            return maFormattedDocTree[nIndex];
        }

        DocTreeNode DrawShape::getLogicalDocTreeNode( sal_Int32 nIndex ) const
        {
            initDocTrees();

            ENSURE_AND_THROW( nIndex >= 0 && static_cast<sal_Int32>(maLogicalDocTree.size()) > nIndex,
                              "DrawShape::getLogicalDocTreeNode(): Invalid index" );

            return maLogicalDocTree[nIndex];
        }

    }
}
