'''
Copyright (c) 2010 Enzo Michelangeli and IT Vision Ltd

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
'''
from numpy import *
from qlcp import qlcp
try:
    import openopt
except:
    pass # if OpenOpt is not installed, the qpsolver kwarg can't be specified
    
def _simple_grad(f, x, delta = 1e-8):
    nvars = x.shape[0]
    Id = eye(nvars)*delta
    grad = array([(f(x+Id[i,:]) - f(x-Id[i,:]))/(2*delta) for i in xrange(nvars)])
    return grad

def _simple_hessian(f, x, delta = 1e-4):    # generally too slow for use
    g = lambda x: _simple_grad(f, x, delta = delta) # g(x) is the gradient of f
    return _simple_grad(g, x, delta=delta)

def _simple_hessdiag(f, x, delta = 1e-4):
    nvars = x.shape[0]
    Id = eye(nvars)*delta
    hd = array([(f(x+Id[i,:]) + f(x-Id[i,:]) - 2*f(x))/delta**2 for i in xrange(nvars)]).flatten()
    return diag(hd)

def sqlcp(f, x0, df=None, A=None, b=None, Aeq=None, beq=None, lb=None, ub=None, minstep=1e-15, minfchg=1e-15, qpsolver=None, callback = None):
    '''
    SQP solver. Approximates f in x0 with paraboloid with same gradient and hessian,
    then finds its minimum with a quadratic solver (qlcp by default) and uses it as new point, 
    iterating till changes in x and/or f drop below given limits. 
    Requires the Hessian to be definite positive.
    The Hessian is initially approximated by its principal diagonal, and then
    updated at every step with the BFGS method.
    f:        objective function of x to be minimized
    x0:       initial value for f
    df:       gradient of f: df(f) should return a function of such as f(x) would
              return the gradient of f in x. If missing or None, an approximation 
              will be calculated with an internal finite-differences procedure.
    A:        array of inequality constraints (A x >= b)
    b:        right-hand side of A x >= b
    Aeq:      array of equality constraints (Aeq x = beq)
    beq:      right-hand side of Aeq x >= beq
    lb:       lower bounds for x (assumed -Inf if missing)
    ub:       upper bounds for x (assumed +Inf if missing)
    minstep:  iterations terminate when updates to x become < minstep (default: 1e-15)
    minfchg:  iterations terminate when RELATIVE changes in f become < minfchg (default: 1e-15)
    qpsolver: if None, qlcp; else a solver accepted by openopt.QP (if OpenOpt and 
              that particular solver are installed)
    '''
       
    nvars = x0.shape[0]
    x = x0.copy()
    niter = 0
    deltah = 1e-4
    deltag = deltah**2
    
    if df == None:  # df(x) is the gradient of f in x
        df = lambda x: _simple_grad(f, x, deltag)
    
    twoI = 2.*eye(nvars)
    oldfx = f(x)
    gradfx = df(x)  # return the gradient of f() at x
    hessfx = _simple_hessdiag(f,x,delta=deltah) # good enough, and much faster, but only works if REAL Hessian is DP!
    invhessfx = linalg.inv(hessfx)
    while True:
        niter += 1
        #print "f(",x,"):",f(x)
        
        # compute the b, beq, lb and ub for the QP sub-problem (as bx, beqx, lbx, ubx)
        bx = b if b == None else b-dot(A,x)
        beqx = beq if beq == None else beq-dot(Aeq,x)
        lbx = lb if lb == None else lb - x
        ubx = ub if ub == None else ub - x

        if qpsolver == None:
            deltax = qlcp(hessfx, gradfx, A=A, b=bx, Aeq=Aeq, beq=beqx, lb=lbx, ub=ubx, QI=invhessfx)
        else:
            p = openopt.QP(hessfx, gradfx, A=A, b=bx, Aeq=Aeq, beq=beqx, lb=lbx, ub=ubx)
            p.ftol = 1.e-10
            r = p.solve(qpsolver, iprint = -1)
            deltax = p.xf
        #print "iter:",niter,", deltax:\n",deltax
        
        if deltax == None:
            print "Cannot converge, sorry."
            x = None
            break
        
        x += deltax
        if linalg.norm(deltax) < minstep:
            break
        fx = f(x)
        if abs(fx-oldfx) < minfchg*abs(fx):
            break
        if callback is not None and callback(x):
            break
            
        oldfx = fx
        oldgradfx = gradfx.copy()
        gradfx = df(x)  # return the gradient of f() at the new x
        # we might also put a termination test on the norm of grad...
        
        '''
        # recalc hessian afresh would be sloooow...
        #hessfx = _simple_hessian(f,x,delta=deltah)  # return the hessian of f() at x
        hessfx = _simple_hessdiag(f,x,delta=deltah)  # return the hessian (diag only) of f() at x
        invhessfx = linalg.inv(hessfx)
        '''
        # update Hessian and its inverse with BFGS based on current Hessian, deltax and deltagrad    
        # See http://en.wikipedia.org/wiki/BFGS
        deltagrad = gradfx - oldgradfx
        hdx = dot(hessfx, deltax)
        dgdx = dot(deltagrad,deltax)
        #if dgdx < 0.:
        #    print "deltagrad * deltax < 0!" # a bad sign
        hessfx += ( outer(deltagrad,deltagrad) / dgdx - 
                    outer(hdx, hdx) / dot(deltax, hdx) )
        # now update inverse of Hessian  
        '''
        invhessfx = linalg.inv(hessfx)
        '''
        hidg = dot(invhessfx,deltagrad)
        oIdgdeltax = outer(hidg,deltax)
        invhessfx += ( (dgdx+dot(deltagrad,hidg))*outer(deltax,deltax)/(dgdx**2) -
                (oIdgdeltax+oIdgdeltax.T)/dgdx ) # just because invhessfx is symmetric, or else:
                #(oIdgdeltax+outer(deltax,dot(invhessfx.T,deltagrad)))/dgdx )
    return x, niter
    
if __name__ == "__main__":

    import time
    
    set_printoptions(suppress=True) # no annoying auto-exp print format

    """
    # find minimum of upside-down multi-gaussian with circular contour, 
    # centered in [1, 1, 1...]
    f = lambda x: -exp(-dot(x-1.,x-1.)) # min in (1., 1., ....) == -1.     

    #nvars = 10
    #x0 = array([0.8]*nvars)
    nvars = 2
    x0 = array([0.8, 1.2])
    
    x, niter = sqlcp(f, x0, lb=array([-2.]*nvars), ub=array([2.]*nvars))
    #x, niter = sqlcp(f, x0, lb=array([-2.]*nvars), ub=array([2.]*nvars), qpsolver='cvxopt_qp' )
    
    print "after",niter,"iterations:" 
    if x != None:
        print "x:", x, ", f(x):", f(x)
    else:
        print "sqlcp() did not converge (encountered Hessian <= 0.)"
    """
    # use sqlcp to find the weights for three types of optimal portfolios: 
    # 1. Minimum Variance
    # 2. Maximum Diversification (http://www.iijournals.com/doi/abs/10.3905/JPM.2008.35.1.40 )
    # 3. Equal Risk Contribution (http://papers.ssrn.com/sol3/papers.cfm?abstract_id=1271972 )
    # Sigma is the dispersion matrix, which under Gaussian assumptions represents the covariance matrix.
    # We assume no shorting (0 <= xi <=1) and of course sum(x) == 1. 
    # No assumptions are made on expected returns.
    
    # use an empirical correlation matrix retrieved for the 30 components of the Dow-Jones Index
    sigma = array([[0.0017435594132612848, 0.0010405075139230527, 0.00056466729819163461, 0.0014618239124703065, 0.00087099318127095769, 0.00050433410634966502, 0.00045294023343118989, 0.00077900963445113821, 0.00064551028137508606, 0.00080006503831363772, 0.00050051512503320047, 0.00045525902737951512, 0.00031947052049844483, 0.00049688387664230458, 0.00016427079349056259, 0.00089807137476054493, 0.00022444556558971345, 0.00021411998767545076, 0.00020438646017042702, 0.00050631075739055516, 0.00023322237399884792, 0.00039037816864125963, 0.00029913036320444953, 0.00023540571988939568, 0.00028133876738337934, 0.00044814737070270565, 0.00047100827898206409, 0.0002293732134027734, 0.0001151661020701042, 0.00034956386212793268],
        [0.0010405075139230527, 0.0016661247698765902, 0.00053577277519254612, 0.0015591064818056158, 0.00085066378295772706, 0.00054281534068404715, 0.0004516250204401675, 0.00073848390801105898, 0.00063951670252916572, 0.00080136477944982906, 0.00052949856267577261, 0.00040144007079662082, 0.00029311242979176169, 0.0004985911648442055, 0.0001500703861243619, 0.0012323133194618962, 0.0002389758827616485, 0.00017600281305940374, 0.00028187551327867917, 0.00047661695501160074, 0.00027902053638324527, 0.00042661543812216548, 0.00029866595961370397, 0.00025333089433295883, 0.00023686014973323312, 0.00054269863246943272, 0.00045408120413772864, 0.00025431124101053809, 0.00015806483236219663, 0.00036608380956765018],
        [0.00056466729819163461, 0.00053577277519254612, 0.00058296970630232784, 0.00059232165951076028, 0.00047215679919594098, 0.00028289223604706739, 0.00024757481364009949, 0.00036742626931361667, 0.00035037273277543695, 0.00044608987589074951, 0.00027888413121055015, 0.00025441274768871463, 0.00015786494799895374, 0.00022899756344165492, 9.8915472995446816e-05, 0.00040776058638684648, 0.00013966293456808379, 0.0001265780905955414, 0.00017078387005001509, 0.00028242094760212364, 0.00014717800088689083, 0.00020150429153939855, 0.0001804161831177259, 0.00016356032739216081, 0.00015455073650058815, 0.0002285763054239806, 0.00033401531454719244, 0.00012979292187265189, 8.4618562187853557e-05, 0.00022737521240542987],
        [0.0014618239124703065, 0.0015591064818056158, 0.00059232165951076028, 0.0035989339658248727, 0.00096705837017097831, 0.00053358474987085446, 0.00054802082274502584, 0.0008190560104748137, 0.0007084787581578346, 0.0012313407623434052, 0.00052692217627915185, 0.00052198238190612038, 0.00028061522630743016, 0.00060142801269925906, 0.0001884325832935911, 0.0018917170920358306, 0.00026333920432991146, 0.00020690633553601388, 0.00026542382233141345, 0.00057308399323228185, 0.00035013535137353282, 0.00038183823922010749, 0.0003824252729169804, 0.00032482678395824772, 0.00034334967915255349, 0.00061715858073235709, 0.00056972673173667245, 0.00027693893402751019, 0.00013526791937798442, 0.00037371858959263013],
        [0.0008709931812709578, 0.00085066378295772706, 0.00047215679919594109, 0.00096705837017097853, 0.00091813672926430905, 0.00043512730805575847, 0.00036309823590887952, 0.0005860109199890245, 0.00049591981971075711, 0.00061734781237656833, 0.00038986376324520559, 0.00033270739864234424, 0.00025707956774102824, 0.00037417829199901218, 0.00012915525212199206, 0.00069964521903762781, 0.00017731992552811688, 0.00015783980537053491, 0.00017247045715603734, 0.00038086966734724906, 0.00020835708812144135, 0.00030773948766229706, 0.00025389331963889303, 0.00018472606100431395, 0.00022756761514561047, 0.00034521363934918539, 0.00037085736920376336, 0.00019493592215352507, 0.0001310169357371338, 0.00029265196795308167],
        [0.00050433410634966513, 0.00054281534068404726, 0.00028289223604706739, 0.00053358474987085457, 0.00043512730805575847, 0.00041889500758687364, 0.00024565393954736995, 0.00039710488484683779, 0.00033951400215707141, 0.00032238434804585938, 0.00026377078797005046, 0.00025921832981638262, 0.00020889965699904376, 0.000318847209652629, 9.2165548853581964e-05, 0.00046415309338276119, 0.00012877417863273185, 0.00011174191951053907, 0.00012596879244018946, 0.00026477463717112409, 0.00014543357841827968, 0.00026489239387270385, 0.00015771990640726436, 0.00013451190624674838, 0.000190254053980895, 0.00024053591427795524, 0.00024336776292291286, 0.00017041499648757824, 9.7701956569977517e-05, 0.00020894145084646119],
        [0.00045294023343119, 0.00045162502044016756, 0.00024757481364009955, 0.00054802082274502605, 0.00036309823590887947, 0.00024565393954736989, 0.00029447401236937806, 0.00033057362534812454, 0.00028096805658627731, 0.00032116753843818355, 0.00022539172553117295, 0.00019276696616502632, 0.00015409585552172549, 0.00021184820486699692, 8.1854669250682588e-05, 0.00042067775659101006, 0.00011771416630732901, 0.00010857726334914074, 0.00012405248570211051, 0.00020948704581906245, 0.00015910525010018578, 0.00019584222838357269, 0.00017411300803114348, 0.00013310220180731734, 0.00015647907405611866, 0.00021880654773886728, 0.00021433168777137458, 0.00014018925130944334, 8.648283567106417e-05, 0.0002311388286125348],
        [0.00077900963445113821, 0.00073848390801105898, 0.00036742626931361667, 0.0008190560104748137, 0.0005860109199890245, 0.00039710488484683774, 0.00033057362534812454, 0.00071968167604944636, 0.00045365574012408178, 0.00050295524712624246, 0.00039184945196733692, 0.0003018939803224561, 0.0002562780207084904, 0.00035256223765973461, 0.00012147129495995746, 0.00065301601567662403, 0.00020013119911731464, 0.0001662937659832261, 0.00018608896271779036, 0.00035421334600078155, 0.00021969994878973516, 0.0002790465613446321, 0.00024894465167794004, 0.00021320657741853033, 0.00022216839490000448, 0.00035658100862726678, 0.00032770575352231564, 0.00020030186133016182, 0.00013015145804703637, 0.0002675472348099964],
        [0.00064551028137508606, 0.00063951670252916572, 0.00035037273277543695, 0.0007084787581578346, 0.00049591981971075711, 0.00033951400215707136, 0.00028096805658627726, 0.00045365574012408178, 0.00054568955045936215, 0.00044873742326412519, 0.00033655997273961304, 0.00027710610723971651, 0.00019990387470636547, 0.0002974519946057676, 0.00012465572856625978, 0.00055019541365383185, 0.00017387866912952843, 0.000158575722821764, 0.0001815466966705969, 0.00031689035991943617, 0.00020709202457190782, 0.00025409598820977299, 0.00021100527847087895, 0.00018934150735024427, 0.00019498170996055995, 0.00027474992573554993, 0.00032788436403440621, 0.00017398641895252875, 0.00011659928317611998, 0.00023243264945995069],
        [0.00080006503831363783, 0.00080136477944982927, 0.00044608987589074962, 0.0012313407623434055, 0.00061734781237656833, 0.00032238434804585938, 0.0003211675384381836, 0.00050295524712624256, 0.00044873742326412524, 0.00098001653983379334, 0.00038276415415058152, 0.00029306205150374964, 0.00020689824252678417, 0.00037041715169705954, 0.00012987718655091884, 0.00079705762857893922, 0.00019738651955453137, 0.00014992347652441533, 0.00018003339494237645, 0.00037134112724537772, 0.00019334359622117072, 0.00023074361384305348, 0.0002092683378845653, 0.00019452434760099672, 0.0001904003820318507, 0.00035227076445222679, 0.0003676572300701601, 0.00017661238008652639, 0.00010673297848303192, 0.00025523190869119198],
        [0.00050051512503320047, 0.00052949856267577261, 0.00027888413121055015, 0.00052692217627915185, 0.00038986376324520559, 0.00026377078797005046, 0.00022539172553117295, 0.00039184945196733692, 0.00033655997273961304, 0.00038276415415058152, 0.00044312013673476172, 0.00020747479420909663, 0.00018245506660017244, 0.00027172145381190016, 9.133372618595779e-05, 0.00046645049575170562, 0.00011851597633105662, 0.00011785282625077351, 0.00015153112961909383, 0.00024439016177593757, 0.00014920568538962385, 0.00020137301595761625, 0.00014730525009581, 0.00015435251887144035, 0.00016923438468894342, 0.00023978498705455752, 0.0002390949694361017, 0.00014588973585025775, 0.00011682128440779384, 0.00018518049410855636],
        [0.00045525902737951512, 0.00040144007079662082, 0.00025441274768871463, 0.00052198238190612038, 0.00033270739864234424, 0.00025921832981638262, 0.00019276696616502632, 0.0003018939803224561, 0.00027710610723971651, 0.00029306205150374958, 0.00020747479420909663, 0.00036787548362997736, 0.00016837851343330884, 0.00023658139671792363, 8.083346693377389e-05, 0.00047019335298772449, 0.00012198356871541827, 8.9405709411980957e-05, 0.00010045783308054581, 0.00022808033936284146, 0.00013362030046355132, 0.00021291321708959, 0.0001573261782760199, 0.00014387239381937652, 0.00015111026203677477, 0.00020181030310989669, 0.00023166967979039663, 0.00012413222923612392, 7.4620675379250451e-05, 0.00015617307983369265],
        [0.00031947052049844483, 0.00029311242979176175, 0.00015786494799895377, 0.00028061522630743016, 0.00025707956774102824, 0.00020889965699904376, 0.00015409585552172552, 0.0002562780207084904, 0.00019990387470636547, 0.00020689824252678417, 0.00018245506660017244, 0.00016837851343330884, 0.00023107026361982608, 0.00018567902537361874, 6.9326774148376134e-05, 0.00031595737405367059, 8.9519293168416079e-05, 8.5998711040916303e-05, 8.6828606088841254e-05, 0.00015142261656553435, 9.2773732447991257e-05, 0.0001510804979791662, 9.650639252824877e-05, 0.00010457913681021546, 0.00013268698047005432, 0.00017277563661199515, 0.00015297225460623895, 0.00011129774830947888, 7.8604901436313474e-05, 0.00012407425415965047],
        [0.00049688387664230458, 0.0004985911648442055, 0.00022899756344165492, 0.00060142801269925906, 0.00037417829199901218, 0.00031884720965262895, 0.00021184820486699692, 0.00035256223765973467, 0.0002974519946057676, 0.00037041715169705954, 0.00027172145381190016, 0.00023658139671792363, 0.00018567902537361874, 0.00046517357580024644, 8.0465748891663888e-05, 0.00044888198848268995, 0.00011121561202485625, 9.5205546689932169e-05, 0.00013252866912141138, 0.00023007836343556077, 0.00015474923395013938, 0.00024369241817449126, 0.00012834830316827268, 0.000131691278319244, 0.00018506575905211054, 0.00023358623818372479, 0.00019858724436221429, 0.00015948432842276337, 8.5868246924908714e-05, 0.00018264376590228153],
        [0.00016427079349056261, 0.00015007038612436193, 9.891547299544683e-05, 0.00018843258329359112, 0.00012915525212199206, 9.2165548853581964e-05, 8.1854669250682602e-05, 0.00012147129495995748, 0.0001246557285662598, 0.00012987718655091884, 9.133372618595779e-05, 8.083346693377389e-05, 6.9326774148376134e-05, 8.0465748891663902e-05, 0.00010428471183505144, 0.00014787698406028965, 6.6240213159287777e-05, 6.9376966330453408e-05, 5.6113111975882908e-05, 9.4398781940796753e-05, 0.00012949631171749973, 8.0388481847987634e-05, 0.00011464136559210671, 8.8344093316637979e-05, 6.3972641829948662e-05, 0.00010223158195070267, 0.00010110609641920815, 6.6912239718361398e-05, 5.6095352114077815e-05, 8.395219369043004e-05],
        [0.00089807137476054515, 0.0012323133194618962, 0.00040776058638684654, 0.0018917170920358306, 0.00069964521903762781, 0.00046415309338276119, 0.00042067775659101006, 0.00065301601567662403, 0.00055019541365383196, 0.00079705762857893922, 0.00046645049575170562, 0.0004701933529877246, 0.0003159573740536707, 0.00044888198848268995, 0.00014787698406028963, 0.0016742536850046081, 0.00022720219223050579, 0.00016619229843107029, 0.00022273492194469996, 0.0004168729853846042, 0.00028045575991525278, 0.00035024297373478157, 0.00029575178828031035, 0.00025944087674286862, 0.00028485307561327731, 0.00054533859490938058, 0.0004097325853965413, 0.00023491593446346948, 0.00016136393944419993, 0.00031858474690962444],
        [0.00022444556558971345, 0.0002389758827616485, 0.00013966293456808379, 0.00026333920432991146, 0.00017731992552811685, 0.00012877417863273185, 0.00011771416630732901, 0.00020013119911731464, 0.00017387866912952843, 0.00019738651955453134, 0.00011851597633105662, 0.00012198356871541825, 8.9519293168416079e-05, 0.00011121561202485625, 6.6240213159287763e-05, 0.00022720219223050576, 0.00021187982906743487, 9.3593681412892525e-05, 8.2779420985979436e-05, 0.00011512263275389122, 0.00010092405608077621, 9.1247934331627938e-05, 0.00012521984617953322, 9.920680831837084e-05, 9.1342905496383841e-05, 0.00016977839573291658, 0.00014478406286612977, 7.8948695046664012e-05, 6.0047791249488371e-05, 0.00011347028632503243],
        [0.00021411998767545079, 0.00017600281305940374, 0.0001265780905955414, 0.00020690633553601393, 0.00015783980537053491, 0.00011174191951053905, 0.00010857726334914074, 0.0001662937659832261, 0.000158575722821764, 0.0001499234765244153, 0.00011785282625077351, 8.9405709411980957e-05, 8.5998711040916303e-05, 9.5205546689932142e-05, 6.9376966330453394e-05, 0.00016619229843107029, 9.3593681412892539e-05, 0.00016142650539165007, 7.3842022970240919e-05, 0.0001080024113698925, 0.00012621093380989603, 9.1120516067696208e-05, 0.00010983781134866504, 9.0724090730583712e-05, 8.3041387343100377e-05, 0.00013469960974003925, 0.00012129873937046912, 7.6109991091428936e-05, 5.8192696828496036e-05, 0.00010309673992417592],
        [0.00020438646017042702, 0.00028187551327867917, 0.00017078387005001503, 0.00026542382233141345, 0.00017247045715603732, 0.00012596879244018946, 0.00012405248570211051, 0.00018608896271779036, 0.0001815466966705969, 0.00018003339494237643, 0.00015153112961909383, 0.00010045783308054581, 8.6828606088841254e-05, 0.00013252866912141136, 5.6113111975882894e-05, 0.0002227349219446999, 8.2779420985979436e-05, 7.3842022970240919e-05, 0.00019220437618269273, 0.00012007823354348225, 9.71706402546939e-05, 0.00011150545680662856, 0.00011406586740214443, 0.00011080036698884069, 7.8981828197308959e-05, 0.00015206956558398962, 0.00013858898026874062, 7.5479525574200534e-05, 7.1979276864682991e-05, 0.0001204500588058864],
        [0.00050631075739055516, 0.00047661695501160074, 0.00028242094760212358, 0.00057308399323228185, 0.00038086966734724906, 0.00026477463717112409, 0.00020948704581906242, 0.00035421334600078155, 0.00031689035991943617, 0.00037134112724537767, 0.00024439016177593757, 0.0002280803393628414, 0.00015142261656553435, 0.00023007836343556077, 9.439878194079674e-05, 0.0004168729853846042, 0.00011512263275389122, 0.00010800241136989248, 0.00012007823354348225, 0.00032407081472971737, 0.0001610919781533154, 0.00021577274454158643, 0.00015876978388210906, 0.00014278198011607001, 0.00014131070885554036, 0.00024587022218439716, 0.00025756528292658994, 0.00013071723203530281, 9.0242310852156821e-05, 0.00018021949045034721],
        [0.00023322237399884792, 0.00027902053638324532, 0.00014717800088689083, 0.00035013535137353282, 0.00020835708812144129, 0.00014543357841827965, 0.00015910525010018578, 0.00021969994878973516, 0.00020709202457190782, 0.00019334359622117067, 0.00014920568538962385, 0.00013362030046355129, 9.2773732447991257e-05, 0.00015474923395013938, 0.00012949631171749973, 0.00028045575991525278, 0.00010092405608077621, 0.00012621093380989603, 9.71706402546939e-05, 0.0001610919781533154, 0.00039218870340375686, 0.00013623578678980007, 0.00024190438865547343, 0.00013472927379201832, 0.00011071422247985879, 0.00020180502048113913, 0.00014303552674120712, 0.00011242325527703542, 9.0214763886804777e-05, 0.00014559031959435407],
        [0.00039037816864125963, 0.00042661543812216548, 0.00020150429153939855, 0.00038183823922010755, 0.00030773948766229706, 0.00026489239387270385, 0.00019584222838357266, 0.0002790465613446321, 0.00025409598820977294, 0.00023074361384305345, 0.00020137301595761622, 0.00021291321708959, 0.00015108049797916617, 0.00024369241817449126, 8.038848184798762e-05, 0.00035024297373478151, 9.1247934331627938e-05, 9.1120516067696208e-05, 0.00011150545680662856, 0.00021577274454158643, 0.00013623578678980007, 0.00038844234957137761, 0.00013657914249205625, 0.00010435272371983484, 0.00013293708550145365, 0.0001995051141439769, 0.00020706260539009419, 0.00013933374168582631, 7.0332040744336917e-05, 0.00016708194976409839],
        [0.00029913036320444958, 0.00029866595961370402, 0.0001804161831177259, 0.00038242527291698045, 0.00025389331963889303, 0.00015771990640726436, 0.00017411300803114348, 0.00024894465167794004, 0.00021100527847087898, 0.0002092683378845653, 0.00014730525009581, 0.00015732617827601993, 9.650639252824877e-05, 0.00012834830316827268, 0.00011464136559210671, 0.00029575178828031035, 0.00012521984617953324, 0.00010983781134866504, 0.00011406586740214444, 0.00015876978388210908, 0.00024190438865547348, 0.00013657914249205625, 0.00034976160020538011, 0.00012647886923042892, 0.00011425512068500032, 0.00022903187186005893, 0.00015232873631107062, 0.00010873712570958563, 9.9382816653392742e-05, 0.00016575569719115351],
        [0.00023540571988939568, 0.00025333089433295889, 0.00016356032739216084, 0.00032482678395824772, 0.00018472606100431395, 0.00013451190624674838, 0.00013310220180731734, 0.00021320657741853033, 0.00018934150735024427, 0.00019452434760099669, 0.00015435251887144038, 0.00014387239381937652, 0.00010457913681021546, 0.000131691278319244, 8.8344093316637979e-05, 0.00025944087674286862, 9.920680831837084e-05, 9.0724090730583712e-05, 0.00011080036698884069, 0.00014278198011607004, 0.00013472927379201832, 0.00010435272371983484, 0.00012647886923042892, 0.00019591020238936908, 0.00011333951697164193, 0.00014726509976280362, 0.00015776523331120456, 0.00011578802877758144, 8.5984686337363525e-05, 0.00013192758394789666],
        [0.0002813387673833794, 0.00023686014973323317, 0.00015455073650058818, 0.00034334967915255349, 0.0002275676151456105, 0.000190254053980895, 0.00015647907405611872, 0.00022216839490000448, 0.00019498170996056001, 0.0001904003820318507, 0.00016923438468894347, 0.00015111026203677477, 0.00013268698047005432, 0.00018506575905211057, 6.3972641829948676e-05, 0.00028485307561327725, 9.1342905496383855e-05, 8.3041387343100391e-05, 7.8981828197308973e-05, 0.00014131070885554039, 0.00011071422247985879, 0.00013293708550145367, 0.00011425512068500032, 0.00011333951697164196, 0.00025195158921694638, 0.00014019855667905478, 0.00014764648896293105, 0.0001825025688154696, 6.8984140921488849e-05, 0.00013505034829071718],
        [0.00044814737070270565, 0.00054269863246943272, 0.00022857630542398057, 0.00061715858073235709, 0.00034521363934918539, 0.00024053591427795524, 0.00021880654773886728, 0.00035658100862726678, 0.00027474992573554993, 0.00035227076445222674, 0.00023978498705455747, 0.00020181030310989669, 0.00017277563661199512, 0.00023358623818372476, 0.00010223158195070265, 0.00054533859490938058, 0.00016977839573291658, 0.00013469960974003925, 0.00015206956558398962, 0.00024587022218439716, 0.00020180502048113913, 0.0001995051141439769, 0.00022903187186005893, 0.00014726509976280359, 0.00014019855667905476, 0.00050155902994208901, 0.00021437210542130572, 0.00014504860812757412, 0.0001284776795578305, 0.00020703298543178194],
        [0.0004710082789820642, 0.00045408120413772875, 0.00033401531454719244, 0.00056972673173667245, 0.00037085736920376336, 0.00024336776292291286, 0.00021433168777137458, 0.00032770575352231564, 0.00032788436403440626, 0.00036765723007016005, 0.0002390949694361017, 0.00023166967979039663, 0.00015297225460623895, 0.00019858724436221429, 0.00010110609641920815, 0.0004097325853965413, 0.00014478406286612982, 0.00012129873937046912, 0.00013858898026874065, 0.00025756528292658994, 0.00014303552674120712, 0.00020706260539009419, 0.00015232873631107062, 0.00015776523331120456, 0.00014764648896293105, 0.00021437210542130575, 0.00034533713430524279, 0.00013760485278009569, 0.00010339921433339071, 0.00018275337535313721],
        [0.00022937321340277343, 0.00025431124101053809, 0.00012979292187265191, 0.00027693893402751024, 0.00019493592215352507, 0.00017041499648757824, 0.00014018925130944337, 0.00020030186133016182, 0.00017398641895252875, 0.00017661238008652639, 0.00014588973585025775, 0.00012413222923612392, 0.00011129774830947888, 0.00015948432842276337, 6.6912239718361398e-05, 0.00023491593446346948, 7.8948695046664026e-05, 7.6109991091428936e-05, 7.5479525574200534e-05, 0.00013071723203530284, 0.00011242325527703544, 0.00013933374168582631, 0.00010873712570958563, 0.00011578802877758144, 0.0001825025688154696, 0.00014504860812757412, 0.00013760485278009569, 0.00022089214334952443, 7.2707605878054968e-05, 0.00013325074242873181],
        [0.00011516610207010419, 0.00015806483236219663, 8.4618562187853557e-05, 0.00013526791937798442, 0.00013101693573713377, 9.770195656997749e-05, 8.6482835671064157e-05, 0.00013015145804703637, 0.00011659928317611996, 0.00010673297848303191, 0.00011682128440779384, 7.4620675379250451e-05, 7.8604901436313461e-05, 8.58682469249087e-05, 5.6095352114077815e-05, 0.0001613639394441999, 6.0047791249488371e-05, 5.8192696828496023e-05, 7.1979276864682991e-05, 9.0242310852156821e-05, 9.0214763886804764e-05, 7.0332040744336917e-05, 9.9382816653392742e-05, 8.5984686337363525e-05, 6.8984140921488849e-05, 0.00012847767955783047, 0.00010339921433339069, 7.2707605878054955e-05, 0.00016292447472886307, 8.2618687066191485e-05],
        [0.00034956386212793279, 0.00036608380956765018, 0.00022737521240542987, 0.00037371858959263019, 0.00029265196795308167, 0.00020894145084646116, 0.0002311388286125348, 0.0002675472348099964, 0.00023243264945995069, 0.00025523190869119198, 0.00018518049410855636, 0.00015617307983369265, 0.00012407425415965047, 0.00018264376590228153, 8.395219369043004e-05, 0.00031858474690962444, 0.00011347028632503243, 0.00010309673992417592, 0.00012045005880588638, 0.00018021949045034721, 0.00014559031959435407, 0.00016708194976409839, 0.00016575569719115348, 0.00013192758394789666, 0.00013505034829071718, 0.00020703298543178194, 0.00018275337535313721, 0.00013325074242873181, 8.2618687066191485e-05, 0.0002427376673833326]
    ])
    
    def MVP(x):
        return dot(x, dot(sigma,x)) # this is the square of portfolio's risk (variance, in a Gaussian world)
    
    def MDP(x):
        s = sqrt(diag(sigma)) # individual volatilities
        return -dot(x, s)/sqrt(dot(x, dot(sigma,x))) # "-" because we want to maximize divers., not minimize

    def ERC(x):
        mr = array([multiply(x, dot(sigma, x))]) # elementwise multiplication
        md = (mr-mr.T).flatten()
        return dot(md,md)

    n = sigma.shape[0]
    x0 = ones(n)/n
    lb = zeros_like(x0)
    ub = ones_like(x0)
    print "--------- computing 1/n portfolio ---------"
    print "risk:",sqrt(dot(x0,dot(sigma,x0)))

    # Alternatives for gradients automatic calculation 
    
    grad = lambda f: None # default: uses the internal function _simple_gradient()
    
    '''
    # numdifftools is slower than DerApproximator. Maybe more accurate, though.
    # http://pypi.python.org/pypi/Numdifftools/
    import numdifftools as nd, numpy as np
    grad = lambda f: nd.Gradient(f) 
    '''
    
    '''
    # algopy is limited (can't handle certain expressions in f) and very slow: do not use.
    # http://pypi.python.org/pypi/algopy/0.2.4
    import algopy
    def _algopy_grad(f, x):
        nvars = x.shape[0]
        xu = algopy.UTPM(zeros((2,nvars,nvars)))
        xu.data[0,:] = x
        xu.data[1,:,:] = eye(nvars)
        return f(xu).data[1]
    grad = lambda f: lambda x: _algopy_grad(f,x)
    '''

    '''
    # FuncDesigner is much faster than algopy, but unable to handle expressions 
    # (such as e.g. NumPy's .flatten() method or linalg.norm() function) so it
    # doesn't work with the test case for the function ERC.
    # http://pypi.python.org/pypi/FuncDesigner , http://openopt.org/FuncDesigner 
    from FuncDesigner import *
    def _funcdes_grad(f, x):
        oox = oovar()
        oof = f(oox)
        r = oof.D({oox:x})
        return r[oox]
    grad = lambda f: lambda x: _funcdes_grad(f,x)
    '''

    '''
    # DerApproximator is more or less equivalent to the internal _simple_gradient()
    # http://pypi.python.org/pypi/DerApproximator , http://openopt.org/DerApproximator 
    import DerApproximator as DA
    grad = lambda f: lambda x: DA.get_d1(f, x, stencil=2) # stencil=3 is more accurate but slower
    '''    
    optimizers = []

    optimizers.append(['sqlcp/qlcp      ', lambda f: sqlcp(f, x0, Aeq=array([ones_like(ub)]), beq=ones(1), lb=lb, ub=ub, minstep=1e-6, df=grad(f))[0]])

    try:
        from openopt import QP
        optimizers.append(['sqlcp/cvxopt_qp ', lambda f: sqlcp(f, x0, Aeq=array([ones_like(ub)]), beq=ones(1), lb=lb, ub=ub, minstep=1e-6, df=grad(f), qpsolver='cvxopt_qp')[0]])
    except:
        pass

    try:
        from scipy.optimize import fmin_slsqp
        optimizers.append(['scipy.fmin_slsqp', lambda f: fmin_slsqp(f, x0, eqcons=[lambda x: sum(x)-1.], bounds =[(lb[i], ub[i]) for i in xrange(x0.shape[0])], iter=100000, acc=1e-14, iprint=0, full_output=False)])
    except:
        pass
    
    try:
        from openopt import NLP
        optimizers.append(['openopt.NLP-ralg', lambda f: asarray(NLP(f, x0, lb=lb, ub=ub, args=(), Aeq=ones_like(x0), beq=[1.], maxFunEvals = 1000000, ftol = 1e-12, iprint=-1).solve('ralg').xf) ])
    except:
        pass

    for f in (MVP, MDP, ERC):
        print "--------- computing",f.__name__,"portfolio ---------"
        for optimizer in optimizers:
            t = time.clock()
            x = array(optimizer[1](f))
            print optimizer[0],": risk:",sqrt(dot(x,dot(sigma,x))), ", time:",time.clock()-t

 
