/*
 * $Id: db_val.c,v 1.7.2.1 2007/01/17 12:35:40 klaus_darilion Exp $
 *
 * POSTGRES module, portions of this code were templated using
 * the mysql module, thus it's similarity.
 *
 * Copyright (C) 2003 August.Net Services, LLC
 *
 * This file is part of openser, a free SIP server.
 *
 * openser is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * openser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ---
 *
 * History
 * -------
 * 2003-04-06 initial code written (Greg Fausak/Andy Fullford)
 * 2003-04-14 gmtime changed to localtime because mktime later
 *            expects localtime, changed daylight saving bug
 *            previously found in mysql module (janakj)
 *
 */

#include <limits.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "../../db/db_val.h"
#include "../../dprint.h"
#include "defs.h"
#include "db_utils.h"
#include "con_postgres.h"
#include "aug_std.h"
#include "dbase.h"


char *strptime(const char *s, const char *format, struct tm *tm);

/*
 * Convert a string to integer
 */
static inline int str2int(const char* _s, int* _v)
{
	long tmp;

#ifdef PARANOID
	if ((!_s) || (!_v)) {
		LOG(L_ERR, "str2int(): Invalid parameter value\n");
		return -1;
	}
#endif

	tmp = strtoul(_s, 0, 10);
	if ((tmp == ULONG_MAX && errno == ERANGE) || 
	    (tmp < INT_MIN) || (tmp > UINT_MAX)) {
		printf("str2int: Value out of range\n");
		return -1;
	}

	*_v = (int)tmp;

	return 0;
}


/*
 * Convert a string to double
 */
static inline int str2double(const char* _s, double* _v)
{
#ifdef PARANOID
	if ((!_s) || (!_v)) {
		LOG(L_ERR, "str2double(): Invalid parameter value\n");
		return -1;
	}
#endif
	*_v = atof(_s);
	return 0;
}


/* 
 * Convert a string to time_t
 */
static inline int str2time(const char* _s, time_t* _v)
{
	struct tm t;
#ifdef PARANOID
	if ((!_s) || (!_v)) {
		LOG(L_ERR, "str2time(): Invalid parameter value\n");
		return -1;
	}
#endif

	memset(&t, '\0', sizeof(struct tm));
	strptime(_s,"%Y-%m-%d %H:%M:%S %z",&t);

	     /* Daylight saving information got lost in the database
	      * so let mktime to guess it. This eliminates the bug when
	      * contacts reloaded from the database have different time
	      * of expiration by one hour when daylight saving is used
	      */ 
	t.tm_isdst = -1;   
	*_v = mktime(&t);

	return 0;
}


/*
 * Convert an integer to string
 */
static inline int int2str(int _v, char* _s, int* _l)
{
#ifdef PARANOID
	if ((!_s) || (!_l) || (!*_l)) {
		LOG(L_ERR, "int2str(): Invalid parameter value\n");
		return -1;
	}
#endif
	*_l = snprintf(_s, *_l, "%-d", _v);
	return 0;
}


/*
 * Convert a double to string
 */
static inline int double2str(double _v, char* _s, int* _l)
{
#ifdef PARANOID
	if ((!_s) || (!_l) || (!*_l)) {
		LOG(L_ERR, "double2str(): Invalid parameter value\n");
		return -1;
	}
#endif
	*_l = snprintf(_s, *_l, "%-10.2f", _v);
	return 0;
}


/*
 * Convert time_t to string
 */
static inline int time2str(time_t _v, char* _s, int* _l)
{
	struct tm *t;
	int bl;
#ifdef PARANOID
	if ((!_s) || (!_l) || (*_l < 2))  {
		LOG(L_ERR, "Invalid parameter value\n");
		return -1;
	}
#endif

	t = localtime(&_v);

	if((bl=strftime(_s,(size_t)(*_l)-1,"'%Y-%m-%d %H:%M:%S %z'",t))>0)
		*_l = bl;
	
	return 0;
}

/*
 * Does not copy strings
 */
int str2valp(db_type_t _t, db_val_t* _v, char* _s, int _l, void *_p)
{
#ifdef PARANOID
	if (!_v) {
		LOG(L_ERR, "str2valp(): Invalid parameter value\n");
		return -1;
	}
#endif

	if (!_s) {
		DLOG("str2valp", "got a null value");
		VAL_TYPE(_v) = _t;
		VAL_NULL(_v) = 1;
		return 0;
	}

	switch(_t) {
	case DB_INT:
	case DB_BITMAP:
		DBG("DEBUG:postgres:str2valp: got int %s \n", _s);
		if (str2int(_s, &VAL_INT(_v)) < 0) {
			LOG(L_ERR, "str2valp(): Error while converting integer value "
				"from string\n");
			return -2;
		} else {
			VAL_TYPE(_v) = DB_INT;
			return 0;
		}
	
	case DB_DOUBLE:
		DBG("DEBUG:postgres:str2valp: got double %s \n", _s);
		if (str2double(_s, &VAL_DOUBLE(_v)) < 0) {
			LOG(L_ERR, "str2valp(): Error while converting double value "
				"from string\n");
			return -3;
		} else {
			VAL_TYPE(_v) = DB_DOUBLE;
			return 0;
		}

	case DB_STRING:
		DBG("DEBUG:postgres:str2valp: got string %s \n", _s);
		VAL_STRING(_v) = aug_strdup(_s, _p);
		VAL_TYPE(_v) = DB_STRING;
		return 0;

	case DB_STR:
		VAL_STR(_v).s = aug_alloc(_l + 1, _p);
		memcpy( _s, VAL_STR(_v).s, _l);
		VAL_STR(_v).s[_l] = (char) 0;
		VAL_STR(_v).len = _l;
		VAL_TYPE(_v) = DB_STR;
		DBG("DEBUG:postgres:str2valp: got len string %d %s\n", _l, _s);
		return 0;

	case DB_DATETIME:
		DBG("DEBUG:postgres:str2valp: got time %s \n", _s);
		if (str2time(_s, &VAL_TIME(_v)) < 0) {
			LOG(L_ERR,"str2valp(): Error converting datetime\n");
			return -4;
		} else {
			VAL_TYPE(_v) = DB_DATETIME;
			return 0;
		}

	case DB_BLOB:
		VAL_BLOB(_v).s = (char*)PQunescapeBytea((unsigned char*)_s,
				(size_t*)&(VAL_BLOB(_v).len) );
		VAL_TYPE(_v) = DB_BLOB;
		DBG("DEBUG:postgres:str2valp: got blob len %d\n", _l);
		return 0;
	}
	return -5;
}


/*
 * Used when converting result from a query
 */
int val2str(db_con_t* _con, db_val_t* _v, char* _s, int* _len)
{
	int l, ret;
	char* old_s;
	char *tmp_s;
	size_t tmp_len;

#ifdef PARANOID
	if ((!_v) || (!_s) || (!_len) || (!*_len)) {
		LOG(L_ERR, "val2str(): Invalid parameter value\n");
		return -1;
	}
#endif
	if (VAL_NULL(_v)) {
		*_len = snprintf(_s, *_len, "NULL");
		return 0;
	}
	
	switch(VAL_TYPE(_v)) {
	case DB_INT:
		if (int2str(VAL_INT(_v), _s, _len) < 0) {
			LOG(L_ERR, "val2str(): Error while converting string to int\n");
			return -2;
		} else {
			return 0;
		}
		break;

	case DB_BITMAP:
		if (int2str(VAL_BITMAP(_v), _s, _len) < 0) {
			LOG(L_ERR, "val2str: Error while converting string to int\n");
			return -3;
		} else {
			return 0;
		}
		break;

	case DB_DOUBLE:
		if (double2str(VAL_DOUBLE(_v), _s, _len) < 0) {
			LOG(L_ERR, "val2str(): Error while converting string to double\n");
			return -3;
		} else {
			return 0;
		}
		break;

	case DB_STRING:
		l = strlen(VAL_STRING(_v));
		LOG(L_ERR, "val2str(): converting %s, %d\n", VAL_STRING(_v), l);
		if (*_len < (l * 2 + 3)) {
			LOG(L_ERR, "val2str(): Destination buffer too short for string\n");
			return -4;
		} else {
			old_s = _s;
			*_s++ = '\'';
			ret = PQescapeStringConn(CON_CONNECTION(_con), _s, VAL_STRING(_v), l, 0);
			LOG(L_DBG, "PG[val2str:DB_STRING]: PQescapeStringConn: in: %d chars, out: %d chars\n", l, ret);
			_s += ret;
			*_s++ = '\'';
			*_s = '\0'; /* FIXME */
			*_len = _s - old_s;
			return 0;
		}
		break;

	case DB_STR:
		l = VAL_STR(_v).len;
		if (*_len < (l * 2 + 3)) {
			LOG(L_ERR, "val2str(): Destination buffer too short for str\n");
			return -5;
		} else {
			old_s = _s;
			*_s++ = '\'';
			ret = PQescapeStringConn(CON_CONNECTION(_con), _s, VAL_STRING(_v), l, 0);
			LOG(L_DBG, "PG[val2str:DB_STR]: PQescapeStringConn: in: %d chars, out: %d chars\n", l, ret);
			_s += ret;
			*_s++ = '\'';
			*_s = '\0'; /* FIXME */
			*_len = _s - old_s;
			return 0;
		}
		break;

	case DB_DATETIME:
		if (time2str(VAL_TIME(_v), _s, _len) < 0) {
			LOG(L_ERR, "val2str(): Error while converting string to time_t\n");
			return -6;
		} else {
			return 0;
		}
		break;

	case DB_BLOB:
		l = VAL_BLOB(_v).len;
		if (*_len < (l * 2 + 3)) {
			LOG(L_ERR, "val2str(): Destination buffer too short for blob\n");
			return -7;
		} else {
			*_s++ = '\'';
			tmp_s = (char*)PQescapeBytea((unsigned char*)VAL_STRING(_v),
					(size_t)l, (size_t*)&tmp_len);
			memcpy(_s, tmp_s, tmp_len);
			PQfreemem(tmp_s);
			tmp_len = strlen(_s);
			*(_s + tmp_len) = '\'';
			*(_s + tmp_len + 1) = '\0';
			*_len = tmp_len + 2;
			return 0;
		}
		break;

	default:
		DBG("val2str(): Unknown data type\n");
		return -7;
	}
	return -8;
}
