/* 
 * Cryptographic helper process.
 * Copyright (C) 2004 Michael C. Richardson <mcr@xelerance.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: pluto_crypt.h,v 1.4 2004/10/25 01:47:41 mcr Exp $
 */

/*
 * this is an internal interface from a master pluto process
 * and a cryptographic helper child.
 *
 * the child performs the heavy lifting of cryptographic functions
 * for pluto. It does this to avoid head-of-queue problems with aggressive
 * mode, to deal with the asynchronous nature of hardware offload,
 * and to compartamentalize lookups to LDAP/HTTP/FTP for CRL fetching
 * and checking.
 *
 */

typedef unsigned int pcr_req_id;

typedef struct wire_chunk {
  unsigned int start;
  size_t       len;
} wire_chunk_t;

#define KENONCE_SIZE 512
struct pcr_kenonce {
  /* inputs */
  u_int16_t oakley_group;
  
  /* outputs */
  wire_chunk_t thespace;
  wire_chunk_t secret;
  wire_chunk_t gi;
  wire_chunk_t n;
  unsigned char space[KENONCE_SIZE];
};

#define wire_chunk_ptr(k, sp) &k->space[(sp)->start]

struct pluto_crypto_req {
  size_t                     pcr_len;
  enum pluto_crypto_requests pcr_type;
  pcr_req_id                 pcr_id;
  enum crypto_importance     pcr_pcim;
  int                        pcr_slot;
  union {
    struct pcr_kenonce kn;
  } pcr_d;
};

struct pluto_crypto_req_cont;  /* forward reference */

typedef void (*crypto_req_func)(struct pluto_crypto_req_cont *
				, struct pluto_crypto_req *
				, err_t ugh);

struct pluto_crypto_req_cont {
  struct pluto_crypto_req_cont *pcrc_next;
  struct pluto_crypto_req      *pcrc_pcr;
  pcr_req_id                    pcrc_id;
  crypto_req_func               pcrc_func;
};

  

#define PCR_REQ_SIZE sizeof(struct pluto_crypto_req)+10

extern void init_crypto_helpers(int nhelpers);
extern err_t send_crypto_helper_request(struct pluto_crypto_req *r
					, struct pluto_crypto_req_cont *cn);
extern void pluto_crypto_helper_sockets(fd_set *readfds);
extern int  pluto_crypto_helper_ready(fd_set *readfds);

extern void pluto_do_crypto_op(struct pluto_crypto_req *r);
extern void pluto_crypto_helper(int fd, int helpernum);
extern void pluto_crypto_allocchunk(wire_chunk_t *space
				    , wire_chunk_t *new
				    , size_t howbig);
/* actual helper functions */
extern stf_status build_ke(struct pluto_crypto_req_cont *cn
			   , struct state *st
			   , const struct oakley_group_desc *group
			   , enum crypto_importance importance);
extern void calc_ke(struct pluto_crypto_req *r);

extern stf_status build_nonce(struct pluto_crypto_req_cont *cn
			      , struct state *st
			      , enum crypto_importance importance);
extern void calc_nonce(struct pluto_crypto_req *r);
