//                                               -*- C++ -*-
/**
 * @file    NearestPointAlgorithmProperty.cxx
 * @brief
 *
 * @author  Romuald Conty
 * @date    2007-01-14 11:34:14
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "NearestPointAlgorithmProperty.hxx"

// OT::UI::GUI
#include "NumericalScalarProperty.hxx"
#include "UnsignedLongProperty.hxx"

namespace OpenTURNS
{

	namespace UI
	{

		namespace GUI
		{

			NearestPointAlgorithmProperty::NearestPointAlgorithmProperty ( NearestPointAlgorithm* nearestPointAlgorithm, QObject* parent, const char* name ) : QOTProperty ( QOTProperty::MultipleChoice, parent, name )
			{
				nearestPointAlgorithm_ = nearestPointAlgorithm;

				maxAbsoluteError_ = 1.0e-5;
				maxConstraintError_ = 1.0e-5;
				maxRelativeError_ = 1.0e-5;
				maxResidualError_ = 1.0e-5;
				maxIterationsNumber_ = 100;
			}


			NearestPointAlgorithmProperty::~NearestPointAlgorithmProperty()
			{}


			QString NearestPointAlgorithmProperty::getValue()
			{
				return method_;
			}

			QStringList NearestPointAlgorithmProperty::getChoices()
			{
				QStringList choices;
				choices << "cobyla" << "abdo-rackwitz";
				return choices;
			}

			bool NearestPointAlgorithmProperty::setValue ( QString value )
			{
				if ( value != method_ )
				{
					method_ = value;

					QOTProperties subproperties = getSubproperties ( false );
					subproperties.setAutoDelete ( true );
					subproperties.clear();

					if ( method_ == "cobyla" )
					{
						autoconnect ( new NumericalScalarProperty ( &maxAbsoluteError_, true, this, "max absolute error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxConstraintError_, true, this, "max constraint error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxRelativeError_, true, this, "max relative error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxResidualError_, true, this, "max residual error" ) );
						autoconnect ( new UnsignedLongProperty ( &maxIterationsNumber_, true, this, "max iterations" ) );
					}
					else if ( method_ == "abdo-rackwitz" )
					{
						autoconnect ( new NumericalScalarProperty ( &maxAbsoluteError_, true, this, "max absolute error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxConstraintError_, true, this, "max constraint error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxRelativeError_, true, this, "max relative error" ) );
						autoconnect ( new NumericalScalarProperty ( &maxResidualError_, true, this, "max residual error" ) );
						autoconnect ( new UnsignedLongProperty ( &maxIterationsNumber_, true, this, "max iterations" ) );
					}

					generateNearestPointAlgorithm();
					emit updated();
					return true;
				}
				return false;
			}

			void NearestPointAlgorithmProperty::autoconnect ( QOTProperty* property )
			{
				connect ( property, SIGNAL ( updated() ), this, SLOT ( generateNearestPointAlgorithm() ) );
				subpropertyUpdated ( property );
			}

			void NearestPointAlgorithmProperty::generateNearestPointAlgorithm()
			{
				if ( method_ == "cobyla" )
				{
					Cobyla cobyla;
					cobyla.setSpecificParameters ( Cobyla::SpecificParameters() );
					( *nearestPointAlgorithm_ ) = NearestPointAlgorithm ( cobyla );
				}
				else if ( method_ == "abdo-rackwitz" )
				{
					AbdoRackwitz abdorackwitz;
					abdorackwitz.setSpecificParameters ( AbdoRackwitz::SpecificParameters() );
					( *nearestPointAlgorithm_ ) = NearestPointAlgorithm ( abdorackwitz );
				}
				nearestPointAlgorithm_->setMaximumRelativeError ( maxRelativeError_ );
				nearestPointAlgorithm_->setMaximumResidualError ( maxResidualError_ );
				nearestPointAlgorithm_->setMaximumAbsoluteError ( maxAbsoluteError_ );
				nearestPointAlgorithm_->setMaximumConstraintError ( maxConstraintError_ );
				nearestPointAlgorithm_->setMaximumIterationsNumber ( maxIterationsNumber_ );
			}

		}

	}

}
