//                                               -*- C++ -*-
/**
 * @file		qpropertiestable.cxx
 * @brief		Specialization of a QTable to display and edit properties
 *
 * @author	Romuald Conty
 * @date		2006-07-06 11:02:12
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

// OT::UI::GUI
#include "Parameter.hxx"

// Qt
#include <qobject.h>
#include <qmetaobject.h>
#include <qvariant.h>
#include <qregexp.h>
#include <qcombobox.h>

// Visual Components
#include "QPropertiesTable.hxx"


namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			QPropertiesTable::QPropertiesTable ( QWidget * parent, const char * name ) : QTable ( 0, 2, parent, name )
			{
				setSelectionMode ( QTable::SingleRow );
				//				horizontalHeader() ->hide();
				verticalHeader() ->hide();

				setLeftMargin ( 0 );
				show();

				setColumnStretchable ( 1, true );
				adjustColumn ( 1 );
				//hideColumn ( 0 );

				object_ = NULL;

				connect ( this, SIGNAL ( valueChanged ( int, int ) ) , this, SLOT ( propertyValueChanged ( int, int ) ) );
			}

			QPropertiesTable::~QPropertiesTable()
			{}


			QTableItem* QPropertiesTable::setPropertyItem ( int row, const QString& property, const QString& value, bool writable )
			{
				verticalHeader() ->setLabel ( row, tr ( property ) );
				QTableItem* item = writable ? new QTableItem ( this, QTableItem::OnTyping, value ) : new QTableItem ( this, QTableItem::Never, value );
				setItem ( row, 0, new QTableItem ( this, QTableItem::Never, property ) );
				setItem ( row, 1, item );
				return item;
			}

			QTableItem* QPropertiesTable::setPropertyItem ( int row, const QString& property, const QValueList<QVariant>& values, bool writable )
			{
				QStringList strings;

				for ( uint j = 0; j < values.count(); ++j )
					strings.append ( values[ j ].toString() );

				return setPropertyItem ( row, property, strings, writable );
			}

			QTableItem* QPropertiesTable::setPropertyItem ( int row, const QString& property, QStringList strings, bool writable )
			{
				verticalHeader() ->setLabel ( row, tr ( property ) );
				QString selected = "";
				if ( strings.count() )
				{
					selected = strings[ 0 ];
					strings.erase ( strings.begin() );
				}
				QComboTableItem* item = new QComboTableItem ( this, strings, false );

				item->setCurrentItem ( selected ); // default choice

				setItem ( row, 0, new QTableItem ( this, QTableItem::Never, property ) );
				setItem ( row, 1, item );
				return item;
			}

			void QPropertiesTable::displayProperties ( QOTObject * object )
			{
				currentRow = 0;
				setNumRows ( 0 );
				const QMetaObject* metaObject = object->metaObject();

				uint propertiesCount = metaObject->numProperties ( true );

				for ( uint i = 0; i < propertiesCount; i++ )
				{
					displayProperty ( object, i );
				}

				if ( object_ != object )
				{
					if ( object_ != NULL )
					{
						object_->disconnect ( this );
					}
					object_ = object;
					connect ( object_, SIGNAL ( updated ( QOTObject* ) ), this , SLOT ( displayProperties ( QOTObject* ) ) );
				}
			}

			void QPropertiesTable::displayProperty ( QOTObject * object, const uint propertyIndex )
			{
				const QString propertyName = object->metaObject() ->propertyNames ( true ).at ( propertyIndex );
				const QVariant propertyValue = object->property ( propertyName.ascii() );

				if ( propertyValue.isValid() )
				{
					if ( propertyValue.type() == QVariant::List )
					{
						QVariant variant = propertyValue.toList() [ 0 ];
						if ( variant.type() == QVariant::String )
						{
							// propertyValue is a list of choice
							setNumRows ( numRows() + 1 );
							setPropertyItem ( currentRow, propertyName, propertyValue.toList(), true );
							currentRow++;
						}
						else
						{
							//if ( variant.type() == QVariant::Map ) {
							// propertyValue is a list of parameters
							QValueList<QVariant> parameters = propertyValue.toList();
							displayParameters ( &parameters, propertyName );
						}
					}
					else
					{
						const QMetaProperty* metaProperty = object->metaObject() ->property ( propertyIndex, true );

						if ( metaProperty != NULL )
						{
							if ( !metaProperty->isEnumType() )
							{
								const QString propertyValueStr = propertyValue.toString() ;
								setNumRows ( numRows() + 1 );
								setPropertyItem ( currentRow, propertyName, propertyValueStr, metaProperty->writable() );
								currentRow++;
							}
							else
							{
								// is EnumType
								QStringList strings;
								QStrList c_strings = metaProperty->enumKeys();
								strings.append ( c_strings.at ( propertyValue.toInt() ) );
								for ( uint j = 0; j < c_strings.count(); ++j )
									strings.append ( QString ( c_strings.at ( j ) ) );

								setNumRows ( numRows() + 1 );
								setPropertyItem ( currentRow, propertyName, strings, metaProperty->writable() );
								currentRow++;
							}
						}
					}
				}
			}

			void QPropertiesTable::displayParameters ( QValueList<QVariant> * parameters, const QString& propertyName )
			{
				//setPropertyItem( currentRow, propertyName, QString::number( parameters->count() ), false );
				//currentRow++;

				QValueList<QVariant>::iterator it;
				uint pass = 0;
				for ( it = parameters->begin(); it != parameters->end(); ++it )
				{
					const QString id = propertyName + "." + QString::number ( pass ) + ".";
					displayParameter ( ( *it ).asMap() , id );
					pass++;
				}
			}

			void QPropertiesTable::displayParameter ( Parameter& parameter, const QString & id )
			{
				//			const QString parameterType = parameter.typeName();
				//			if ( parameter.type() == QVariant::Map ) {
				//				QMap<QString, QVariant> param = parameter.toMap();
				//				QValueList<QString> keys = param.keys();
				if ( !parameter.isEmpty() )
				{
					uint keysCount = 0;
					Parameter::Iterator it;
					for ( it = parameter.begin(); it != parameter.end(); ++it )
					{
						keysCount++;
					}

					setNumRows ( numRows() + keysCount );
					currentRow += keysCount;
					uint reverseRowOrder = 1;

					for ( it = parameter.begin(); it != parameter.end(); ++it )
					{
						const QString key = it.key();
						QVariant value = it.data();
						const QString valueType = value.typeName();
						if ( value.type() == QVariant::String )
						{
							setPropertyItem ( currentRow - reverseRowOrder, id + key, value.asString(), true );
							reverseRowOrder++;
						}
						else if ( ( value.type() == QVariant::StringList ) || ( value.type() == QVariant::List ) )
						{
							setPropertyItem ( currentRow - reverseRowOrder, id + key, value.asList(), true );
							reverseRowOrder++;
						}
						else
						{
							qDebug ( QString ( "Unknown type : " + valueType + " for " + key ).ascii() );
						}
					}
				}
			}

			void QPropertiesTable::propertyValueChanged ( int row, int col )
			{
				const QString propertyName = text ( row, 0 );
				const QString propertyValue = text ( row, col );

				QRegExp rx ( "^(\\w+)\\.(\\d+)\\.(\\w+)$" );
				int pos = rx.search ( propertyName );
				if ( pos > -1 )
				{
					// Value changed on Subproperty
					QString paramPropertyName = rx.cap ( 1 );
					QString paramIndex = rx.cap ( 2 );
					QString paramName = rx.cap ( 3 );

					QVariant propertyVariant = object_->property ( paramPropertyName.ascii() );
					if ( propertyVariant.isValid() )
					{
						if ( propertyVariant.type() == QVariant::List )
						{
							// Subproperty is a valid Parameter.
							QValueList<QVariant> parameters = propertyVariant.toList();
							QVariant variant = parameters[ paramIndex.toUInt() ].asMap() [ paramName ];
							if ( variant.type() == QVariant::String )
							{
								// Parameter contains QString type.
								variant = propertyValue;
							}
							else if ( variant.type() == QVariant::List )
							{
								// Parameter contains a list of choices.
								variant.asList().first() = propertyValue;
							}
							parameters[ paramIndex.toUInt() ].asMap() [ paramName ] = variant;
							object_->setProperty ( paramPropertyName, parameters );
						}
					}
				}
				else
				{
					// Value changed on property (not a subproperty)
					QVariant propertyVariant = object_->property ( propertyName.ascii() );
					if ( propertyVariant.isValid() )
					{
						if ( propertyVariant.type() == QVariant::List )
						{
							QVariant variant = propertyVariant.toList() [ 0 ];
							if ( variant.isValid() && variant.type() == QVariant::String )
							{
								// propertyValue is a list of choice
								QValueList<QVariant> choice;
								choice.append ( propertyValue );
								object_->setProperty ( propertyName, choice );
							}
							else
							{
								/*
								//if ( variant.type() == QVariant::Map ) {
								// propertyValue is a list of parameters
								QValueList<QVariant> parameters = propertyVariant.toList();
								bool ok = false;
								const uint parametersWantedCount = propertyValue.toUInt( &ok );
								const uint parametersCount = parameters.count();
								if ( ok ) {
								if ( parametersWantedCount >= parametersCount ) {
								for ( uint i = parametersCount; i < parametersWantedCount; i++ ) {
								Parameter p;
								p.insert ( QString( "name" ), QString( "x" + QString::number( i ) ) );
								p.insert ( QString( "designation" ), QString( "" ) );
								parameters.insert ( parameters.end(), QVariant( p ) );
								}
								} else {
								for ( uint i = parametersCount; i > parametersWantedCount; i-- ) {
								parameters.remove( parameters.last() );
								}
								}
								object_->setProperty( propertyName, parameters );
								}
								*/
							}
							displayProperties ( object_ );
						}
						else
						{
							// propertyVariant is not a list
							const QMetaObject* metaObject = object_->metaObject();
							const uint propertyIndex = metaObject->findProperty ( propertyName, true );
							const QMetaProperty* metaProperty = metaObject->property ( propertyIndex, true );

							if ( metaProperty != NULL )
							{
								if ( !metaProperty->isEnumType() )
								{
									// propertyVariant is not a enumerated type
									object_->setProperty ( propertyName, propertyValue );
									const QString propertyValueCheck = object_->property ( propertyName.ascii() ).toString();
									if ( propertyValueCheck != propertyValue )
									{
										// displayed and recorded values are differents
										setText ( propertyIndex, col, propertyValueCheck );
									}
								}
								else
								{
									// is EnumType
									int propertyValueInt = 0;
									QStrList c_strings = metaProperty->enumKeys();
									for ( uint j = 0; j < c_strings.count(); ++j )
									{
										if ( QString ( c_strings.at ( j ) ) == propertyValue )
										{
											propertyValueInt = j;
										}
									}
									object_->setProperty ( propertyName, QVariant ( propertyValueInt ) );
								}
							}
						}
					}
				}
			}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

