//                                               -*- C++ -*-
/**
 * @file    Result.cxx
 * @brief   Unified result display object
 *
 * @author  Romuald Conty
 * @date    2006-11-09 20:35:18
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "Result.hxx"

// OT::UI::GUI
#include "NumericalScalarProperty.hxx"
#include "NumericalPointProperty.hxx"

// std
#include <math.h>

namespace OpenTURNS
{

	namespace UI
	{

		namespace GUI
		{

			Result::Result ( QOTObject *parent, const char *name, QDateTime start, QDateTime end, SimulationResult result ) : Content ( parent, name )
			{
				delete Content::actionCopy_;
				calculationStart_ = start;
				calculationEnd_ = end;

				setDescription ( tr ( "Calculation started at : " ) + calculationStart_.toString ( Qt::LocalDate ) + tr ( ", computed in : " ) + QString::number ( calculationStart_.secsTo ( calculationEnd_ ) ) + tr ( " seconds." ) );

				addElement ( "probability", result.getProbabilityEstimate() );
				addElement ( "variance", result.getVarianceEstimate() );
				addElement ( "outer_sampling", result.getOuterSampling() );
				addElement ( "block_size", result.getBlockSize() );
			}

			Result::Result ( QOTObject *parent, const char *name, QDateTime start, QDateTime end, FormResult result ) : Content ( parent, name )
			{
				delete Content::actionCopy_;
				calculationStart_ = start;
				calculationEnd_ = end;

				setDescription ( tr ( "Calculation started at : " ) + calculationStart_.toString ( Qt::LocalDate ) + tr ( ", computed in : " ) + QString::number ( calculationStart_.secsTo ( calculationEnd_ ) ) + tr ( " seconds." ) );

				addElement ( "event probability", result.getEventProbability() );
				addElement ( "generalized reliability index", result.getGeneralisedReliabilityIndex() );
				addElement ( "standard space design point", result.getStandardSpaceDesignPoint() );
				addElement ( "physical space design point", result.getPhysicalSpaceDesignPoint() );
				addElement ( "is standard point origin in failure space?", QString ( result.getIsStandardPointOriginInFailureSpace() ? "true" : "false" ) );
				addElement ( "importance factors", result.getImportanceFactors() );
				addElement ( "Hasofer reliability index", result.getHasoferReliabilityIndex() );
			}

			Result::Result ( QOTObject *parent, const char *name, QDateTime start, QDateTime end, SormResult result ) : Content ( parent, name )
			{
				delete Content::actionCopy_;
				calculationStart_ = start;
				calculationEnd_ = end;

				setDescription ( tr ( "Calculation started at : " ) + calculationStart_.toString ( Qt::LocalDate ) + tr ( ", computed in : " ) + QString::number ( calculationStart_.secsTo ( calculationEnd_ ) ) + tr ( " seconds." ) );
				addElement ( "Breitung event probability", result.getEventProbabilityBreitung() );
				addElement ( "Breitung generalized reliability index", result.getGeneralisedReliabilityIndexBreitung() );
				addElement ( "HohenBichler event probability", result.getEventProbabilityHohenBichler() );
				addElement ( "HohenBichler generalized reliability index", result.getGeneralisedReliabilityIndexHohenBichler() );
				addElement ( "Tvedt event probability", result.getEventProbabilityTvedt() );
				addElement ( "Tvedt generalized reliability index", result.getGeneralisedReliabilityIndexTvedt() );
				addElement ( "sorted curvatures", result.getSortedCurvatures() );
				addElement ( "standard space design point", result.getStandardSpaceDesignPoint() );
				addElement ( "physical space design point", result.getPhysicalSpaceDesignPoint() );
				addElement ( "is standard point origin in failure space?", QString ( result.getIsStandardPointOriginInFailureSpace() ? "true" : "false" ) );
				addElement ( "importance factors" , result.getImportanceFactors() );
				addElement ( "Hasofer reliability index", result.getHasoferReliabilityIndex() );
			}

			Result::Result ( QOTObject *parent, const char *name, QDateTime start, QDateTime end, StrongMaximumTestResult result ) : Content ( parent, name )
			{
				delete Content::actionCopy_;
				calculationStart_ = start;
				calculationEnd_ = end;

				setDescription ( tr ( "Calculation started at : " ) + calculationStart_.toString ( Qt::LocalDate ) + tr ( ", computed in : " ) + QString::number ( calculationStart_.secsTo ( calculationEnd_ ) ) + tr ( " seconds." ) );

				addElement ( "point number", result.getPointNumber() );
				addElement ( "design point vicinity angle (rad)", acos ( result.getDesignPointVicinity() ) );
				addElement ( "design point vicinity angle (deg)", acos ( result.getDesignPointVicinity() ) * 180 / M_PI );
				addElement ( "near design point verifying event points number", result.getNearDesignPointVerifyingEventPoints().getSize() );
				addElement ( "near design point violating event points number", result.getNearDesignPointViolatingEventPoints().getSize() );
				addElement ( "far design point verifying event points number", result.getFarDesignPointVerifyingEventPoints().getSize() );
				addElement ( "far design point violating event points number", result.getFarDesignPointViolatingEventPoints().getSize() );
				addElement ( "importance level", result.getImportanceLevel() );
				addElement ( "accuracy level", result.getAccuracyLevel() );
				addElement ( "confidence level", result.getConfidenceLevel() );
			}

			Result::Result ( QOTObject *parent, const char *name, QDateTime start, QDateTime end, CentralDispersionResult result ) : Content ( parent, name )
			{
				delete Content::actionCopy_;
				calculationStart_ = start;
				calculationEnd_ = end;

				setDescription ( tr ( "Calculation started at : " ) + calculationStart_.toString ( Qt::LocalDate ) + tr ( ", computed in : " ) + QString::number ( calculationStart_.secsTo ( calculationEnd_ ) ) + tr ( " seconds." ) );

				addElement ( "mean", result.mean[ 0 ] );
				addElement ( "covariance", result.covariance ( 0, 0 ) );
			}

			Result::~Result()
			{}

			void Result::addElement ( QString element, NumericalScalar value )
			{
				new NumericalScalarProperty ( new NumericalScalar ( value ), false, this, element.ascii() );
			}

			void Result::addElement ( QString element, NumericalPoint value )
			{
				new NumericalPointProperty ( new NumericalPoint ( value ), false, this, element.ascii() );
			}

			void Result::addElement ( QString element, QString value )
			{
				Parameter p;
				p.insert ( element, value );
				elements_.append ( p );
			}

			QValueList<QVariant> Result::getElements() const
			{
				return elements_;
			}

		}
	}
}
