//                                               -*- C++ -*-
/**
 *  @file  PersistentObject.hxx
 *  @brief Class PersistentObject saves and reloads the object's internal state
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: PersistentObject.hxx 990 2008-10-29 17:54:45Z dutka $
 */
#ifndef OPENTURNS_PERSISTENTOBJECT_HXX
#define OPENTURNS_PERSISTENTOBJECT_HXX

#include "OSS.hxx"
#include "Object.hxx"
#include "StorageManager.hxx"
#include "IdFactory.hxx"
#include "Pointer.hxx"

namespace OpenTURNS
{

  /**
   * The default name of any object
   */
  extern const String DefaultName;

  namespace Base
  {

    namespace Common
    {

      class StorageManager;
      class StorageManagerAdvocate;
      template <class PERSISTENT> class Factory;

      /**
       * @class PersistentObject
       *
       * This is the base class of all objects that can be stored into
       * and reloaded from the study.
       */

      class PersistentObject
	: public Object
      {
	CLASSNAME;
      public:

	/** Default constructor */
	PersistentObject()
	  : p_name_(),
	    id_(IdFactory::getInstance().buildId()),
	    shadowedId_(id_),
	    studyVisible_(true)
	{}

	/** Standard constructor */
	explicit PersistentObject(const String & name)
	  : p_name_(new String(name)),
	    id_(IdFactory::getInstance().buildId()),
	    shadowedId_(id_),
	    studyVisible_(true)
	{}

	/** Copy constructor */
	PersistentObject(const PersistentObject & other)
	  : p_name_(other.p_name_),
	    id_(IdFactory::getInstance().buildId()),
	    shadowedId_(other.shadowedId_),
	    studyVisible_(other.studyVisible_)
	{}

	/** Virtual constructor. This method MUST be overloaded in derived classes */
	virtual PersistentObject * clone() const = 0;

	/** Destructor */
	virtual ~PersistentObject() {}

	/** Assignment */
	inline
	PersistentObject & operator =(const PersistentObject & other)
	{
	  if (this != &other) { // Other is NOT me, so I can assign it to me
	    p_name_ = other.p_name_;
	    studyVisible_ = other.studyVisible_;
	  }
	  return *this;
	}


	/** Comparison operator */
	inline
	Bool operator ==(const PersistentObject & other) const { return true; }

	/** Identity comparator */
	inline
	Bool is(const PersistentObject & other) const { return (getId() == other.getId()); }

	/** String converter */
	virtual String str() const { return OSS() << "class=" << getClassName() << " name=" << getName(); }


	/** Id accessor */
	inline
	Id getId() const { return id_; }

	/** Shadowed id accessor */
	inline
	void setShadowedId(Id id) { shadowedId_ = id; }
	inline
	Id getShadowedId() const { return shadowedId_; }

	/** Visibility accessor */
	inline
	void setVisibility(Bool visible) { studyVisible_ = visible; }
	inline
	Bool getVisibility() const { return studyVisible_; }

	/** Object name accessor */
	inline
	String getName() const
	{
	  if (p_name_.isNull()) return OT::DefaultName;
	  return *p_name_;
	}

	/** Object name accessor */
	inline
	void setName(const String & name) { p_name_.reset(new String(name)); }


	/** Method save() stores the object through the StorageManager */
	inline
	void save(const StorageManager & mgr, bool fromStudy = false) const
	{
	  if (! mgr.isSavedObject(id_)) {
	    StorageManager::Advocate adv ( mgr.registerObject(*this, fromStudy) );
	    save(adv);
	    mgr.markObjectAsSaved(id_);
	  }
	}
	virtual void save(const StorageManager::Advocate & adv) const
	{
	  adv.writeAttribute(StorageManager::IdAttribute, id_);
	  adv.writeAttribute(StorageManager::NameAttribute, getName());
	}

	/** Method load() reloads the object from the StorageManager */
	virtual void load(const StorageManager::Advocate & adv)
	{
	  String name;
	  adv.readAttribute(StorageManager::IdAttribute, shadowedId_);
	  adv.readAttribute(StorageManager::NameAttribute, name);
	  if (name == OT::DefaultName) p_name_.reset();
	  else setName(name);
	}


      protected:

      private:

	/** The name of the object */
	mutable Pointer<String> p_name_;

	/**
	 * The unique identifier of the object
	 * This identifier is needed when saving and reloading the object
	 * because it allows the chaining of objects even if they are
	 * relocated
	 */
	const Id id_;

	/**
	 * The shadowed id is used when object is reloaded. The object gets
	 * a new id that is almost always different from that stored in the study.
	 * So when we need to rebuild the objects dependency tree (ie, when
	 * object A embed object B), we have to make the id translation: the
	 * object holds the both ids, the new one (aka the Id as returned by getId)
	 * and the former one stored in the study (aka the shadowed id). This latter
	 * is never seen except by the object factory.
	 */
	Id shadowedId_;

	/**
	 * This flag is used by the Study to know if the object should be displayed
	 * even if it had been added to the study (in particular, when the object
	 * was rebuild from file)
	 */
	Bool studyVisible_;

      }; /* class PersistentObject */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_PERSISTENTOBJECT_HXX */
