//                                               -*- C++ -*-
/**
 * @file  WrapperCommon.c
 * @brief This file provides objects and functions commonly used by wrappers
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy: dutka $
 * @date   $LastChangedDate: 2008-10-09 16:58:58 +0200 (jeu 09 oct 2008) $
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdlib.h>
#include <malloc.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <regex.h>
#include <math.h>

#include <map>
#include <algorithm>
#include "OTconfig.hxx"
#include "OTtypes.hxx"
#include "OSS.hxx"
#include "WrapperCommonFunctions.hxx"
#include "WrapperCommon.h"

#ifndef RM_EXECUTABLE_NAME
#error "RM_EXECUTABLE_NAME is NOT defined. Check configuration."
#endif


extern "C" {

  /* The following declarations MUST be coherent with enumeration declarations in WrapperInterface.h */
  wrapper_CHARTAB WrapperListElementTypeAsString    = { "in", "out" };
  wrapper_CHARTAB WrapperProvidedAsString           = { "no", "yes" };
  wrapper_CHARTAB WrapperConfigurationStateAsString = { "shared", "specific" };
  wrapper_CHARTAB WrapperConfigurationModeAsString  = { "static-link", "dynamic-link", "fork" };
  wrapper_CHARTAB WrapperDataTransferModeAsString   = { "files", "pipe", "arguments", "socket", "corba" };

  wrapper_CHARTAB WrapperErrorAsString = {
    "Success",
    "Wrapper memory error. It may be an allocation failure or something related to memory storage",
    "Wrapper initialization error. The wrapper failed to initialize",
    "Wrapper execution error. The wrapper failed or the external code sent an error code to the wrapper during computation",
    "Wrapper finilization error. The wrapper failed to finalize",
    "Wrapper creation state error. The state can NOT be created or allocated",
    "Wrapper deletion state error. The state can NOT be freed",
    "Wrapper information error. The wrapper is unable to provide information to the calling function",
    "Wrapper internal error. The wrapper failed in an internal function. Please send a bug report",
    "Wrapper wrong argument. The wrapper has received a argument it can not handle",
    "Wrapper usage error. The wrapper has a configuration file not concording with its internals",
    "Wrapper method not implemented. This may be due to a bug but the general case is to use unimplemented methods to provide optional features"
  };


  /* Return a string describing the error encountered */
  extern const char * getErrorAsString(enum WrapperErrorCode errorCode)
  {
    if ( (errorCode < 0) || (errorCode > UNUSED_ERROR) ) return 0;
    return WrapperErrorAsString[errorCode];
  }

  /* Print out the user message passed as argument */
  extern void printUserMessage(const char * functionName, const char * message)
  {
    printToLogUser( OT::OSS() << "(" << functionName << ") " << message );
  }

  /* Print out the message passed as argument */
  extern void printMessage(const char * functionName, const char * message)
  {
    printToLog( OT::OSS() << "(" << functionName << ") " << message );
  }

  /* Print out a message telling that we enter a function */
  extern void printEntrance(const char * functionName)
  {
    printToLog( OT::OSS() << "(" << functionName << ") Entering" );
  }

  /* Print out a message telling that we leave a function */
  extern void printExit(const char * functionName)
  {
    printToLog( OT::OSS() << "(" << functionName << ") Exiting" );
  }

  /* Print out a message telling which state we are working on */
  extern void printState(const char * functionName, void * p_state)
  {
    printToLog( OT::OSS() << "(" << functionName << ") p_state=" << p_state );
  }

  /* Print out the content of the WrapperExchangedData structure */
  extern void printWrapperExchangedData(const char * functionName, const struct WrapperExchangedData * p_exchangedData)
  {
    OT::OSS oss;
    oss << "(" << functionName << ") p_exchangedData=" << p_exchangedData << "\n";
    oss << "(" << functionName << ") Exchanged data :\n";

    /* Print the file list */
    const struct WrapperFileList * currentFileElement = p_exchangedData->fileList_;
    while (currentFileElement) {
      const struct WrapperFileList * nextElement = currentFileElement->next_;
      oss << "(" << functionName
	  << "File (id='" << currentFileElement->file_->id_
	  << "', name='" << currentFileElement->file_->name_
	  << "', path='" << currentFileElement->file_->path_
	  << "', type='" << WrapperListElementTypeAsString[currentFileElement->file_->type_]
	  << "', subst='" << currentFileElement->file_->subst_
	  << "')\n";
      currentFileElement = nextElement;
    }

    /* Print the variable list */
    const struct WrapperVariableList * currentVariableElement = p_exchangedData->variableList_;
    while (currentVariableElement) {
      const struct WrapperVariableList * nextElement = currentVariableElement->next_;
      oss << "(" << functionName
	  << ") Variable (comment='" << currentVariableElement->variable_->comment_
	  << "', unit='" << currentVariableElement->variable_->unit_
	  << "', regexp='" << currentVariableElement->variable_->regexp_
	  << "', format='" << currentVariableElement->variable_->format_
	  << "', type='" << WrapperListElementTypeAsString[currentVariableElement->variable_->type_]
	  << "', gradient='" << WrapperProvidedAsString[currentVariableElement->variable_->gradient_]
	  << "')\n";
      currentVariableElement = nextElement;
    }

    /* Print the wrapper parameters */
    struct WrapperConfiguration * parameters = p_exchangedData->parameters_;
    oss << "(" << functionName
	<< ") Parameters (mode='" << WrapperConfigurationModeAsString[parameters->mode_]
	<< "', in='" << WrapperDataTransferModeAsString[parameters->in_]
	<< "', out='" << WrapperDataTransferModeAsString[parameters->out_]
	<< "', command='" << parameters->command_
	<< "')\n";

    /* Print the platform configuration */
    struct PlatformConfiguration * platform = p_exchangedData->platform_;
    oss << "(" << functionName
	<< ") Platform configuration (temporary directory='" << platform->generalTemporaryDirectory_
	<< "', real regexp shortcut='" << platform->realRegexpShortcut_
	<< "', integer regexp shortcut='" << platform->integerRegexpShortcut_
	<< "', separator regexp shortcut='" << platform->separatorRegexpShortcut_
	<< "', number of CPU=" << platform->nbCpus_
	<< ")";

    printToLog( oss );
  }

  /* Print out the content of the WrapperExchangedData structure */
  extern void printWrapperInformation(const char * functionName, const struct WrapperInformation * p_info)
  {
    OT::OSS oss;
    oss << "(" << functionName << ") p_info=" << p_info << "\n";
    oss << "(" << functionName << ") Information (inSize=" << p_info->inSize_
	<< ", outSize_=" << p_info->outSize_ << ")\n";
    printToLog( oss );
  }

  /* Get the number of variables of some type (in or out) in the data read from the description file */
  unsigned long getNumberOfVariables(const struct WrapperExchangedData * p_exchangedData, unsigned long type)
  {
    unsigned long number = 0;
  
    const struct WrapperVariableList * currentVariableElement = p_exchangedData->variableList_;
    while (currentVariableElement) {
      if (currentVariableElement->variable_->type_ == type) ++number;
      currentVariableElement = currentVariableElement->next_;
    }

    return number;
  }

  /* Get the number of files of some type (in or out) in the data read from the description file */
  unsigned long getNumberOfFiles(const struct WrapperExchangedData * p_exchangedData, unsigned long type)
  {
    unsigned long number = 0;
  
    const struct WrapperFileList * currentFileElement = p_exchangedData->fileList_;
    while (currentFileElement) {
      if (currentFileElement->file_->type_ == type) ++number;
      currentFileElement = currentFileElement->next_;
    }

    return number;
  }

  /* Print out the content of a point structure */
  extern void printPoint(const char * functionName, const struct point * p_point)
  {
    OT::OSS oss;
    oss << "(" << functionName
	<< ") struct point {size=" << p_point->size_
	<< ", data=[";

    oss.setPrecision(20);
    std::copy( &(p_point->data_[0]), &(p_point->data_[0]) + p_point->size_, OT::OSS_iterator<double>(oss, ",") );

    oss << "]}";

    printToLog( oss );
  }




  struct printSamplePoint : std::unary_function<struct point *, void>
  {
    OT::OSS & oss_;
    const char * sep_;
    printSamplePoint(OT::OSS & oss) : oss_(oss), sep_("") {}
    void operator() (const struct point & point) {
      oss_ << sep_ << "[";
      std::copy( &(point.data_[0]), &(point.data_[0]) + point.size_, OT::OSS_iterator<double>(oss_, ",") );
      oss_ << "]";
      sep_ = ",";
    }
  };
  

  /* Print out the content of a sample structure */
  extern void printSample(const char * functionName, const struct sample * p_sample)
  {
    OT::OSS oss;
    oss << "(" << functionName
	<< ") struct sample {size=" << p_sample->size_
	<< ", data=[";

    oss.setPrecision(20);
    std::for_each( &(p_sample->data_[0]), &(p_sample->data_[0]) + p_sample->size_, printSamplePoint(oss) );

    oss << "]}";

    printToLog(oss );
  }

  /* Print out the content of a matrix structure */
  extern void printMatrix(const char * functionName, const struct matrix * p_matrix)
  {
    OT::OSS oss;
    oss << "(" << functionName
	<< ") struct matrix {rows=" << p_matrix->nbrows_
	<< ", cols=" << p_matrix->nbcols_
	<< ", data=[";

    std::copy( &(p_matrix->data_[0]), &(p_matrix->data_[0]) + (p_matrix->nbrows_ * p_matrix->nbcols_), OT::OSS_iterator<double>(oss, ",") );

    oss << "]}\n";

    printToLog( oss );
  }

  /* Print out the content of a tensor structure */
  extern void printTensor(const char * functionName, const struct tensor * p_tensor)
  {
    OT::OSS oss;
    oss << "(" << functionName
	<< ") struct tensor {rows=" << p_tensor->nbrows_
	<< ", cols=" << p_tensor->nbcols_
	<< ", sheets=" << p_tensor->nbsheets_
	<< ", data=[";

    std::copy( &(p_tensor->data_[0]), &(p_tensor->data_[0]) + (p_tensor->nbrows_ * p_tensor->nbcols_ * p_tensor->nbsheets_), OT::OSS_iterator<double>(oss, ",") );

    oss << "]}\n";

    printToLog( oss );
  }

  /* Deep copy of a WrapperExchangedData structure (returns non-zero if error occured) */
  /* TODO : check memory leaks when an error occurs in the middle of this function */
  extern long copyWrapperExchangedData(struct WrapperExchangedData ** p_p_new_exchangedData, const struct WrapperExchangedData * p_exchangedData)
  {
    if (!p_p_new_exchangedData) { return 1; } /* no memory location to store data */
    *p_p_new_exchangedData = NULL;

    struct WrapperExchangedData * p_new_exchangedData = (struct WrapperExchangedData *) malloc(sizeof(struct WrapperExchangedData));
    if (!p_new_exchangedData) { return 1; } /* exit if no memory available */




    /* Copy the file list */
    p_new_exchangedData->fileList_ = NULL;
    const struct WrapperFileList *  currentFileElement      =   p_exchangedData->fileList_;
    struct       WrapperFileList ** p_newCurrentFileElement = & p_new_exchangedData->fileList_;
    while (currentFileElement) {
      struct       WrapperFileList * newCurrentFileElement = (struct WrapperFileList *) malloc(sizeof(struct WrapperFileList));
      if (!newCurrentFileElement) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }

      newCurrentFileElement->next_ = NULL;
      *p_newCurrentFileElement     =   newCurrentFileElement;
      p_newCurrentFileElement      = & newCurrentFileElement->next_;

      newCurrentFileElement->file_ = (struct WrapperFileListElement *) malloc(sizeof(struct WrapperFileListElement));
      if (!newCurrentFileElement->file_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }

      newCurrentFileElement->file_->id_ = (char *) malloc(sizeof(char) * (strlen(currentFileElement->file_->id_)+1));
      if (!newCurrentFileElement->file_->id_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentFileElement->file_->id_, currentFileElement->file_->id_, strlen(currentFileElement->file_->id_)+1 );

      newCurrentFileElement->file_->name_ = (char *) malloc(sizeof(char) * (strlen(currentFileElement->file_->name_)+1));
      if (!newCurrentFileElement->file_->name_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentFileElement->file_->name_, currentFileElement->file_->name_, strlen(currentFileElement->file_->name_)+1 );

      newCurrentFileElement->file_->path_ = (char *) malloc(sizeof(char) * (strlen(currentFileElement->file_->path_)+1));
      if (!newCurrentFileElement->file_->path_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentFileElement->file_->path_, currentFileElement->file_->path_, strlen(currentFileElement->file_->path_)+1 );

      newCurrentFileElement->file_->subst_ = (char *) malloc(sizeof(char) * (strlen(currentFileElement->file_->subst_)+1));
      if (!newCurrentFileElement->file_->subst_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentFileElement->file_->subst_, currentFileElement->file_->subst_, strlen(currentFileElement->file_->subst_)+1 );

      newCurrentFileElement->file_->type_ = currentFileElement->file_->type_;

      currentFileElement = currentFileElement->next_;
    } /* end while (currentFileElement) */



    /* Copy the variable list */
    p_new_exchangedData->variableList_ = NULL;
    const struct WrapperVariableList *  currentVariableElement      =   p_exchangedData->variableList_;
    struct       WrapperVariableList ** p_newCurrentVariableElement = & p_new_exchangedData->variableList_;
    while (currentVariableElement) {
      struct       WrapperVariableList * newCurrentVariableElement = (struct WrapperVariableList *) malloc(sizeof(struct WrapperVariableList));
      if (!newCurrentVariableElement) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }

      newCurrentVariableElement->next_ = NULL;
      *p_newCurrentVariableElement     =   newCurrentVariableElement;
      p_newCurrentVariableElement      = & newCurrentVariableElement->next_;

      newCurrentVariableElement->variable_ = (struct WrapperVariableListElement *) malloc(sizeof(struct WrapperVariableListElement));
      if (!newCurrentVariableElement->variable_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }

      newCurrentVariableElement->variable_->comment_ = (char *) malloc(sizeof(char) * (strlen(currentVariableElement->variable_->comment_)+1));
      if (!newCurrentVariableElement->variable_->comment_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentVariableElement->variable_->comment_, currentVariableElement->variable_->comment_, strlen(currentVariableElement->variable_->comment_)+1 );

      newCurrentVariableElement->variable_->unit_ = (char *) malloc(sizeof(char) * (strlen(currentVariableElement->variable_->unit_)+1));
      if (!newCurrentVariableElement->variable_->unit_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentVariableElement->variable_->unit_, currentVariableElement->variable_->unit_, strlen(currentVariableElement->variable_->unit_)+1 );

      newCurrentVariableElement->variable_->regexp_ = (char *) malloc(sizeof(char) * (strlen(currentVariableElement->variable_->regexp_)+1));
      if (!newCurrentVariableElement->variable_->regexp_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentVariableElement->variable_->regexp_, currentVariableElement->variable_->regexp_, strlen(currentVariableElement->variable_->regexp_)+1 );

      newCurrentVariableElement->variable_->format_ = (char *) malloc(sizeof(char) * (strlen(currentVariableElement->variable_->format_)+1));
      if (!newCurrentVariableElement->variable_->format_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
      strncpy(newCurrentVariableElement->variable_->format_, currentVariableElement->variable_->format_, strlen(currentVariableElement->variable_->format_)+1 );

      newCurrentVariableElement->variable_->type_ = currentVariableElement->variable_->type_;

      newCurrentVariableElement->variable_->gradient_ = currentVariableElement->variable_->gradient_;

      currentVariableElement = currentVariableElement->next_;
    } /* end while (currentVariableElement) */




    /* Copy the wrapper parameters */
    p_new_exchangedData->parameters_ = NULL;
    struct WrapperConfiguration * parameters     = p_exchangedData->parameters_;
    struct WrapperConfiguration * new_parameters = (struct WrapperConfiguration *) malloc(sizeof(struct WrapperConfiguration));
    if (!new_parameters) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    p_new_exchangedData->parameters_ = new_parameters;

    new_parameters->mode_ = parameters->mode_;
    new_parameters->in_   = parameters->in_;
    new_parameters->out_  = parameters->out_;

    new_parameters->command_ = (char *) malloc(sizeof(char) * (strlen(parameters->command_)+1));
    if (!new_parameters->command_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    strncpy(new_parameters->command_, parameters->command_, strlen(parameters->command_)+1 );





    /* Copy the platform configuration */
    p_new_exchangedData->platform_ = NULL;
    struct PlatformConfiguration * platform = p_exchangedData->platform_;
    struct PlatformConfiguration * new_platform = (struct PlatformConfiguration *) malloc(sizeof(struct PlatformConfiguration));
    if (!new_platform) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    p_new_exchangedData->platform_ = new_platform;

  
    new_platform->generalTemporaryDirectory_ = (char *) malloc(sizeof(char) * (strlen(platform->generalTemporaryDirectory_)+1));
    if (!new_platform->generalTemporaryDirectory_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    strncpy(new_platform->generalTemporaryDirectory_, platform->generalTemporaryDirectory_, strlen(platform->generalTemporaryDirectory_)+1 );

    new_platform->realRegexpShortcut_ = (char *) malloc(sizeof(char) * (strlen(platform->realRegexpShortcut_)+1));
    if (!new_platform->realRegexpShortcut_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    strncpy(new_platform->realRegexpShortcut_, platform->realRegexpShortcut_, strlen(platform->realRegexpShortcut_)+1 );

    new_platform->integerRegexpShortcut_ = (char *) malloc(sizeof(char) * (strlen(platform->integerRegexpShortcut_)+1));
    if (!new_platform->integerRegexpShortcut_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    strncpy(new_platform->integerRegexpShortcut_, platform->integerRegexpShortcut_, strlen(platform->integerRegexpShortcut_)+1 );

    new_platform->separatorRegexpShortcut_ = (char *) malloc(sizeof(char) * (strlen(platform->separatorRegexpShortcut_)+1));
    if (!new_platform->separatorRegexpShortcut_) { /* exit if no memory available */
	freeWrapperExchangedData( p_new_exchangedData );
	return 1;
      }
    strncpy(new_platform->separatorRegexpShortcut_, platform->separatorRegexpShortcut_, strlen(platform->separatorRegexpShortcut_)+1 );

    new_platform->nbCpus_ = platform->nbCpus_;




    /* At last we copy this structure into its destination location */
    *p_p_new_exchangedData = p_new_exchangedData;


    return 0;
  }

  /* Free a WrapperExchangedData structure */
  extern void freeWrapperExchangedData(struct WrapperExchangedData * p_exchangedData)
  {
    if (p_exchangedData) {
      /* Free the file list */
      struct WrapperFileList * currentFileElement = p_exchangedData->fileList_;
      while (currentFileElement) {
	struct WrapperFileList * nextElement = currentFileElement->next_;
	free(currentFileElement->file_->id_);
	free(currentFileElement->file_->name_);
	free(currentFileElement->file_->path_);
	free(currentFileElement->file_->subst_);
	free(currentFileElement->file_);
	free(currentFileElement);
	currentFileElement = nextElement;
      }

      /* Free the variable list */
      struct WrapperVariableList * currentVariableElement = p_exchangedData->variableList_;
      while (currentVariableElement) {
	struct WrapperVariableList * nextElement = currentVariableElement->next_;
	free(currentVariableElement->variable_->comment_);
	free(currentVariableElement->variable_->unit_);
	free(currentVariableElement->variable_->regexp_);
	free(currentVariableElement->variable_->format_);
	free(currentVariableElement->variable_);
	free(currentVariableElement);
	currentVariableElement = nextElement;
      }

      /* Free the parameters */
      free(p_exchangedData->parameters_->command_);
      free(p_exchangedData->parameters_);

      /* Free the platform configuration */
      free(p_exchangedData->platform_->generalTemporaryDirectory_);
      free(p_exchangedData->platform_->realRegexpShortcut_);
      free(p_exchangedData->platform_->integerRegexpShortcut_);
      free(p_exchangedData->platform_);

      /* Free the embedding structure */
      free(p_exchangedData);
    }
  }


  /* Create a temporary directory. This function internaly allocates the storage of the return value.
   * The symetric function deleteTemporaryDirectory should be used to avoid memory leak.
   */
  extern char * createTemporaryDirectory(const char * tempDirPrefix, const struct WrapperExchangedData * p_exchangedData)
  {
    if ( (tempDirPrefix == 0) || (strlen(tempDirPrefix) == 0) ) return 0;
    if (p_exchangedData == 0) return 0;

    OT::String tempDir = p_exchangedData->platform_->generalTemporaryDirectory_;
    tempDir += "/";
    tempDir += tempDirPrefix;
    tempDir += "_XXXXXX";

    char * tempDirName = (char *) calloc(tempDir.size()+1, sizeof(char));
    strncpy(tempDirName, tempDir.c_str(), tempDir.size()+1);
    mkdtemp(tempDirName);

    return tempDirName;
  }


  /* Delete a temporary directory if executionStatus is zero (ie, no error).
   * This function internaly frees the storage allocated by
   * createTemporaryDirectory, so be careful not to do it yourself. The content of the
   * directory is erased.
   */
  extern void deleteTemporaryDirectory(char * tempDir, long executionStatus)
  {
    if (!executionStatus) {
      long rc = 0;
    
      char * command;
      long commandLength = strlen(RM_EXECUTABLE_NAME) + 1 /* for space */
	+ 3 /* for -rf */ + 1 /* for space */
	+ strlen(tempDir);
    
      command = (char *) calloc(commandLength+1, sizeof(char));
      strncpy(command, RM_EXECUTABLE_NAME, strlen(RM_EXECUTABLE_NAME)+1);
      strcat(command, " -rf ");
      strncat(command, tempDir, strlen(tempDir)+1);
    
      if ((rc = system( command ))) {
	OT::OSS oss;
	oss << "Can NOT remove temporary directory '" << tempDir << "'";
	const OT::String msg = oss;
	printMessage( "deleteTemporaryDirectory", msg.c_str() );
      }
    
      free(command);

    } else {
      OT::OSS oss;
      oss << "Execution error. Temporary directory '" << tempDir << "' NOT removed. Check files and values";
      const OT::String msg = oss;
      printMessage( "deleteTemporaryDirectory", msg.c_str() );
    }
    free(tempDir);
  }

  /* Change working directory to path */
  extern long changeDirectory(const char * path)
  {
    long rc = chdir(path);
    if (rc) {
      OT::OSS oss;
      oss << "Unable to change to directory '" << path << "'. Reason: " << strerror(errno);
      const OT::String msg = oss;
      printMessage( "changeDirectory", msg.c_str() );
      return rc;
    }

#ifdef DEBUG
    OT::OSS oss;
    oss << "Changing to directory '" << path << "'";
    const OT::String msg = oss;
    printMessage( "changeDirectory", msg.c_str() );
#endif
    return rc;
  }

  /* Return the name of the current directory in a string. Note that this storage is
   * internaly allocated by the function, so users MUST free it (using free()).
   */
  extern char * getCurrentWorkingDirectory()
  {
    long size = 32;
    char * path = NULL;
    char * buf = NULL;

    path = (char*) calloc(size, sizeof(char));
    while ( (buf = getcwd(path, size)) == NULL) {
      if (errno == ERANGE) {
	// The buffer is not big enough
	size *= 2;
	path = (char*) realloc(path, size * sizeof(char));
      } else {
	printMessage( "getCurrentWorkingDirectory", "Can NOT get current working directory" );
	break;
      }
    }

    return path;
  }



  /* Create the files listed in the exchanged data structure in the directory.
   * This action may result in variable substitution in the files according to the
   * regular expressions listed in the exchanged data structure. Returns non-zero if
   * some failure occured.
   */
  extern long createInputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point)
  {
    if (checkDirectory(directory)) return 1;

    /* For all the input files listed in the exchanged data structure, we substitute the actual values */
    const struct WrapperFileList * currentFileElement = p_exchangedData->fileList_;
    while (currentFileElement) {
      if (currentFileElement->file_->type_ == WRAPPER_IN) {
	/* This is an input file. We allocate a buffer for the file */
	char * destFile;
	char * last_slash;
	long len;
	long sizeDiff = 0;
	struct stat file_stat;
	char * buf = readFile(currentFileElement->file_->path_, &file_stat);
	if (buf == NULL) return 1;

	/* Perform all wanted substitutions in the file */
	buf = substituteVariables(buf, p_exchangedData, p_point, sizeDiff);
	if ( strcmp(currentFileElement->file_->subst_, "") ) /* File has variables to be substituted */
	  buf = substituteVariables(buf, p_exchangedData, p_point, sizeDiff);

	/* We copy the file to the temporary directory */
	/* We set last_slash to point right after the last slash in path
	 * for only retrieving the name of the file.
	 */
	if ( (last_slash = strrchr(currentFileElement->file_->path_, '/')) == NULL) last_slash = currentFileElement->file_->path_;
	else ++last_slash;

	/* We append the name of the file to the temporary directory */
	len = strlen(directory) + 1 /* for separating slash */ + strlen(last_slash) + 1 /* for trailing NULL */;
	destFile = (char * ) calloc(len, sizeof(char));
	strncpy(destFile, directory,  strlen(directory));
	strcat(destFile, "/");
	strncat(destFile, last_slash, strlen(last_slash));
	file_stat.st_size += sizeDiff;
	writeFile(destFile, buf, file_stat);
	free(destFile);

	/* Free storage after all */
	free(buf);
      }
      currentFileElement = currentFileElement->next_;
    }

    /* OK */
    return 0;
  }

  /* Read the content of the output listed in the exchanged data structure located in
   * the directory. Returns non-zero if some failure occured. For any output variable
   * not found in output file, the corresponding coordinate in p_point is set to NaN.
   */
  extern long readOutputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, struct point * p_point)
  {
    if (checkDirectory(directory)) return 1;

    /* For all the input files listed in the exchanged data structure, we substitute the actual values */
    const struct WrapperFileList * currentFileElement = p_exchangedData->fileList_;
    while (currentFileElement) {
      if (currentFileElement->file_->type_ == WRAPPER_OUT) {
	/* This is an output file. We allocate a buffer for the file */
	char * destFile;
	char * last_slash;
	long len;
	struct stat file_stat;
	if ( (last_slash = strrchr(currentFileElement->file_->path_, '/')) == NULL) last_slash = currentFileElement->file_->path_;
	else ++last_slash;

	/* We append the name of the file to the temporary directory */
	len = strlen(directory) + 1 /* for separating slash */ + strlen(last_slash) + 1 /* for trailing NULL */;
	destFile = (char * ) calloc(len, sizeof(char));
	strncpy(destFile, directory,  strlen(directory));
	strcat(destFile, "/");
	strncat(destFile, last_slash, strlen(last_slash));

	char * buf = readFile(destFile, &file_stat);
	if (buf == NULL) return 1;

	/* Pick up data from the file */
	if ( strcmp(currentFileElement->file_->subst_, "") ) /* File has variables to be retrieved */
	  /* long rc = */ retrieveVariables(buf, p_exchangedData, p_point);

	/* We copy the file to the base directory */
	writeFile(currentFileElement->file_->path_, buf, file_stat);

	/* Free storage after all */
	free(buf);
	free(destFile);
      }
      currentFileElement = currentFileElement->next_;
    }

    /* OK */
    return 0;
  }


  /* Make a command line from the template command and the variable list. The returned value is the new
   * command with variables substituted. This buffer is internaly allocated and MUST be deallocated
   * using free.
   */
  extern char * makeCommandFromTemplate(const char * command, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point)
  {
    char * cmd = (char *) calloc(strlen(command) + 1, sizeof(char));
    if (cmd == NULL) return NULL;
    strcpy(cmd, command);

    long sizeDiff = 0;
    cmd = substituteVariables(cmd, p_exchangedData, p_point, sizeDiff);
    if (cmd == NULL) return NULL;
   
    /* OK */ 
    return cmd;
  }

  /* Insulate a shell command, so that it can be run in a random temporary directory */
  extern char * insulateCommand(char * command, const char * temporaryDir)
  {
    char * currentWorkingDirectory = getCurrentWorkingDirectory();
    OT::OSS oss;
    oss << "curdir=" << currentWorkingDirectory << " ; cd " << temporaryDir << " ; " << command << " ; rc=$? ; cd $curdir ; exit $rc";
    free ( currentWorkingDirectory );

    OT::String theCommand = oss;
    char * cmd = (char *) calloc(theCommand.size() + 1, sizeof(char));
    strcpy( cmd, theCommand.c_str() );
    free( command );

    /* OK */
    return cmd;
  }


  /* Get the command line from the exchanged data structure */
  extern const char * getCommand(const struct WrapperExchangedData * p_exchangedData)
  {
    return p_exchangedData->parameters_->command_;
  }


  /* Get the number of virtual CPUs from the exchanged data structure */
  extern unsigned long getNumberOfCPUs(const struct WrapperExchangedData * p_exchangedData)
  {
    return p_exchangedData->platform_->nbCpus_;
  }


  /* Run an insulated command based on the template read from the wrapper description file */
  extern long runInsulatedCommand(const char * temporaryDir, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point)
  {
    char * cmd = makeCommandFromTemplate(getCommand(p_exchangedData), p_exchangedData, p_point);
    cmd = insulateCommand(cmd, temporaryDir);
    long rc = system(cmd);
    free(cmd);
    return rc;
  }



} /* extern "C" */
