//                                               -*- C++ -*-
/**
 *  @file  XMLToolbox.cxx
 *  @brief This file provides basic XML functionalities
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-17 14:26:02 +0200 (Fri, 17 Oct 2008) $
 *  Id:      $Id: WrapperFile.cxx 974 2008-10-17 12:26:02Z dutka $
 */
#include "OT.hxx"
#include "OSS.hxx"
#include "Log.hxx"
#include "Exception.hxx"
#include "XMLToolbox.hxx"
#include <cassert>

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      using Common::WrapperFileParsingException;
      using Common::Log;


#if defined HAS_LIBXML2

      XMLDoc::XMLDoc() : doc_(xmlNewDoc( REINTERPRET_CAST( const xmlChar *, "1.0" ) ))
      {
	// Nothing to do
      }

      XMLDoc::XMLDoc(const XMLDoc & other) : doc_(xmlCopyDoc( other.doc_, 1 ))
      {
	// Nothing to do
      }

      XMLDoc::XMLDoc(const FileName & pathToFile) : doc_(0)
      {
	doc_ = xmlParseFile( pathToFile.c_str() );
	if (doc_ == NULL) throw WrapperFileParsingException(HERE) << "Error in parsing wrapper file " << pathToFile;
      }

      XMLDoc::~XMLDoc() throw()
      {
	xmlFreeDoc( doc_ );

      }

      XMLDoc & XMLDoc::operator =(const XMLDoc & other)
      {
	if (this != &other) {
	  xmlFreeDoc( doc_ );
	  doc_ = xmlCopyDoc( other.doc_, 1 );
	}

	return *this;
      }

      XMLDoc::operator xmlDocPtr()
      {
	return doc_;
      }

      XMLDoc::operator const xmlDocPtr() const
      {
	return doc_;
      }

      void XMLDoc::save( const FileName & fileName ) const
      {
	xmlSaveFormatFileEnc(fileName.c_str(), doc_, "UTF-8", 1);
      }


      String XML::ToString(const xmlString & st)
      {
	return String(st.begin(), st.end());
      }

      Bool XML::IsText(const Node & elt)
      {
	assert( elt != NULL );
	return (elt->type == XML_TEXT_NODE);
      }



      Bool XML::IsElement(const Node & elt)
      {
	assert( elt != NULL );
	return (elt->type == XML_ELEMENT_NODE);
      }



      Bool XML::IsElement(const Node & elt, const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	Bool isElt      = IsElement( elt );
	Bool hasName    = ( xmlStrcmp( elt->name, aName.c_str() ) == 0 );
	return isElt && hasName;
      }



      Bool XML::ElementHasAttribute(const Node & elt, const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	return xmlHasProp( elt, aName.c_str() );
      }



      String XML::GetAttributeByName(const Node & parent, const String & name)
      {
	String attrVal;
	if (parent) {
	  xmlString aName = xmlString( name.begin(), name.end() );
	  if ( xmlHasProp( parent, aName.c_str() ) ) {
	    xmlString val = xmlGetProp(parent, aName.c_str() );
	    attrVal = String(val.begin(), val.end());
	  }
	}
	return attrVal;
      }


      void XML::SetAttribute(const Node & parent, const String & attribute, const String & value)
      {
	if (parent) {
	  xmlString aAttr  = xmlString( attribute.begin(), attribute.end() );
	  xmlString aValue = xmlString( value.begin(), value.end() );
	  xmlNewProp( parent, aAttr.c_str(), aValue.c_str() );
	}
      }


      XML::Node XML::FindElementByName(const Node & parent, const String & name)
      {
	Node cur = NULL;
	if (parent) {
	  for(cur = parent->children; cur; cur = cur->next) {
	    if (IsElement(cur, name)) {
	      break;
	    }
	  }
	}
	return cur;
      }



      String XML::GetNodeValue(const Node & parent)
      {
	String value;
	Node cur = NULL;
	if (parent) {
	  for(cur = parent->children; cur; cur = cur->next) {
	    if (IsText(cur)) {
	      xmlString val = cur->content;
	      value = String(val.begin(), val.end());
	      break;
	    }
	  }
	}
	return value;
      }


      String XML::GetNodeName(const Node & parent)
      {
	String name;
	if (parent) {
	  xmlString aName = parent->name;
	  name = String( aName.begin(), aName.end() );
	}
	return name;
      }



      XML::Node XML::NewNode(const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	Node node = xmlNewNode( NULL, aName.c_str() );
	return node;
      }



      XML::Node XML::NewTextNode(const String & value)
      {
	xmlString aValue = xmlString( value.begin(), value.end() );
	Node node = xmlNewText( aValue.c_str() );
	return node;
      }



      XML::Node XML::NewNode(const String & name, const String & value)
      {
	Node node = NewNode( name );
	Node child = NewTextNode( value );
	AddChild( node, child );
	return node;
      }
	

      void XML::AddChild(const Node & parent, const Node & child)
      {
	xmlAddChild( parent, child );
      }

	
      XML::Node XML::GetRootNode( const XMLDoc & doc )
      {
	return xmlDocGetRootElement( doc );
      }


      void XML::SetRootNode( const XMLDoc & doc, const Node & root )
      {
	xmlDocSetRootElement( doc, root );
      }

      XML::Node XML::GetFirstChild( const Node & node )
      {
	return node->children;
      }

      XML::Node XML::GetNextNode( const Node & node )
      {
	return node->next;
      }



#endif /* HAS_LIXML2 */

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
