//                                               -*- C++ -*-
/**
 *  @file  ComposedNumericalMathGradientImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: ComposedNumericalMathGradientImplementation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "ComposedNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComposedNumericalMathGradientImplementation);

      static Common::Factory<ComposedNumericalMathGradientImplementation> RegisteredFactory("ComposedNumericalMathGradientImplementation");

      /* Default constructor */
      ComposedNumericalMathGradientImplementation::ComposedNumericalMathGradientImplementation(const GradientImplementation & p_leftGradient,
											       const EvaluationImplementation & p_rightFunction,
											       const GradientImplementation & p_rightGradient) throw(InvalidArgumentException)
	: NumericalMathGradientImplementation(),
	  p_leftGradient_(p_leftGradient),
	  p_rightFunction_(p_rightFunction),
	  p_rightGradient_(p_rightGradient)
      {
	UnsignedLong p_rightInputDimension(p_rightFunction->getInputNumericalPointDimension());
	UnsignedLong p_rightOutputDimension(p_rightFunction->getOutputNumericalPointDimension());
	UnsignedLong p_leftInputDimension(p_leftGradient->getInputNumericalPointDimension());
	// We check that the dimensions of the input parameters are compatible
	if ((p_leftInputDimension != p_rightOutputDimension) || (p_rightGradient->getInputNumericalPointDimension() != p_rightInputDimension) || (p_rightGradient->getOutputNumericalPointDimension() != p_rightOutputDimension)) throw InvalidArgumentException(HERE) << "The left gradient dimensions are not compatible with the right function dimensions, or the right gradient dimensions are not compatible with the right function dimensions";
      }

      /* Virtual constructor */
      ComposedNumericalMathGradientImplementation * ComposedNumericalMathGradientImplementation::clone() const
      {
	return new ComposedNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool ComposedNumericalMathGradientImplementation::operator ==(const ComposedNumericalMathGradientImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ComposedNumericalMathGradientImplementation::str() const {
	OSS oss;
	oss << "class=" << ComposedNumericalMathGradientImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftGradient=" << p_leftGradient_->str()
	    << " rightFunction=" << p_rightFunction_->str()
	    << " rightGradient=" << p_rightGradient_->str();
	return oss;
      }
  
      /* Method gradient() returns the Jacobian transposed matrix of the function at point
       * f goes from Rn to Rp and g from Rq to Rp, so h = f o g goes from Rq to Rn. Its    
       * jacobian goes from Rq to L(Rq, Rn), thus its gradient goes from Rq to L(Rn, Rq):  
       * it is an n by q matrix                                                            */
      ComposedNumericalMathGradientImplementation::Matrix ComposedNumericalMathGradientImplementation::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	callsNumber_++;
	return  p_rightGradient_->gradient(in) * p_leftGradient_->gradient(p_rightFunction_->operator()(in));
      }

      /* Accessor for input point dimension */
      UnsignedLong ComposedNumericalMathGradientImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_rightGradient_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ComposedNumericalMathGradientImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_leftGradient_->getOutputNumericalPointDimension();

      }
     
      /* Method save() stores the object through the StorageManager */
      void ComposedNumericalMathGradientImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.writeValue(*p_leftGradient_, StorageManager::MemberNameAttribute, "leftGradient_");
	adv.writeValue(*p_rightFunction_, StorageManager::MemberNameAttribute, "rightFunction_");
	adv.writeValue(*p_rightGradient_, StorageManager::MemberNameAttribute, "rightGradient_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ComposedNumericalMathGradientImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	String name;
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
	Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
	  
	StorageManager::List objList = adv.getList(StorageManager::ObjectEntity);
	if (objList.readValue(evaluationValue,  StorageManager::MemberNameAttribute, "rightFunction_"))
	  p_rightFunction_ = evaluationValue.getImplementation();
	if (objList.readValue(gradientValue, StorageManager::MemberNameAttribute, "leftGradient_"))
	  p_leftGradient_ = gradientValue.getImplementation();
	if (objList.readValue(gradientValue, StorageManager::MemberNameAttribute, "rightGradient_"))
	  p_rightGradient_ = gradientValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
